/*
 * Copyright 1999-2019 Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.csp.sentinel.adapter.reactor;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.function.Supplier;

import com.alibaba.csp.sentinel.AsyncEntry;
import com.alibaba.csp.sentinel.SphU;
import com.alibaba.csp.sentinel.Tracer;
import com.alibaba.csp.sentinel.slots.block.BlockException;

/**
 * @author Eric Zhao
 */
public class AsyncFunctional {

    public static <R> CompletableFuture<R> runAsyncWith(String resource, Supplier<R> f,
                                                        ExecutorService pool) {
        try {
            AsyncEntry entry = SphU.asyncEntry(resource);
            return CompletableFuture.supplyAsync(f, pool)
                .whenComplete((r, ex) -> {
                    try {
                        if (ex != null) {
                            Tracer.traceContext(ex, 1, entry.getAsyncContext());
                        }
                    } finally {
                        if (entry != null) {
                            entry.exit();
                        }
                    }
                });
        } catch (BlockException ex) {
            CompletableFuture<R> future = new CompletableFuture<>();
            future.completeExceptionally(ex);
            return future;
        }
    }
}
