/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.phoenix.mapreduce;

import org.apache.hadoop.io.Writable;
import org.apache.hadoop.io.WritableUtils;
import org.apache.hadoop.mapreduce.InputSplit;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;

public class ODPSInputSplit extends InputSplit implements Writable {
    private int idx;
    private long start;
    private long count;
    private long splitSize;

    public ODPSInputSplit() {
    }

    public ODPSInputSplit(int idx, long start, long count, long splitSize) {
        this.idx = idx;
        this.start = start;
        this.count = count;
        this.splitSize = splitSize;
    }

    @Override
    public void write(DataOutput dataOutput) throws IOException {
        WritableUtils.writeVInt(dataOutput, idx);
        WritableUtils.writeVLong(dataOutput, start);
        WritableUtils.writeVLong(dataOutput, count);
    }

    @Override
    public void readFields(DataInput dataInput) throws IOException {
        idx = WritableUtils.readVInt(dataInput);
        start = WritableUtils.readVLong(dataInput);
        count = WritableUtils.readVLong(dataInput);
    }

    @Override
    public long getLength() throws IOException, InterruptedException {
        return splitSize;
    }

    @Override
    public String[] getLocations() throws IOException, InterruptedException {
        return new String[]{""};
    }

    public int getIdx() {
        return idx;
    }

    public long getStart() {
        return start;
    }

    public long getCount() {
        return count;
    }

    @Override
    public String toString() {
        return "ODPSInputSplit{" + "idx=" + idx + ", start=" + start + ", count=" + count
                + ", splitSize=" + splitSize + '}';
    }
}
