/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.phoenix.mapreduce;

import com.aliyun.odps.data.Record;
import com.aliyun.odps.tunnel.TableTunnel;
import com.aliyun.odps.tunnel.TunnelException;
import com.aliyun.odps.tunnel.io.TunnelRecordReader;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.io.*;
import org.apache.hadoop.mapreduce.InputSplit;
import org.apache.hadoop.mapreduce.RecordReader;
import org.apache.hadoop.mapreduce.TaskAttemptContext;
import org.apache.hadoop.util.ReflectionUtils;
import org.apache.phoenix.mapreduce.util.PhoenixMapReduceUtil;
import org.apache.phoenix.util.ColumnInfo;

import java.io.IOException;
import java.util.List;

import static org.apache.phoenix.mapreduce.util.PhoenixMapReduceUtil.getDownloadSession;

public class ODPSRecordReader<T extends MapWritable> extends RecordReader<NullWritable, T> {
    private static final Log LOG = LogFactory.getLog(ODPSRecordReader.class);

    private final Configuration conf;
    private final Class<T> inputClass;
    private TunnelRecordReader reader;
    private NullWritable key =  NullWritable.get();
    private T value = null;
    private Record record;
    private TableTunnel.DownloadSession downloadSession;
    private List<ColumnInfo> targetTableColumns;
    private List<String> sourceTableColumns;

    public ODPSRecordReader(Class<T> inputClass, Configuration conf) {
        this.conf = conf;
        this.inputClass = inputClass;
    }

    @Override
    public void initialize(InputSplit split, TaskAttemptContext context)
            throws IOException, InterruptedException {
        ODPSInputSplit odpsSplit = (ODPSInputSplit) split;
        try {
            downloadSession = getDownloadSession(conf);
            reader = downloadSession.openRecordReader(odpsSplit.getStart(), odpsSplit.getCount());
            targetTableColumns = PhoenixMapReduceUtil.buildTargetTableColumns(conf);
            sourceTableColumns = Lists.newArrayList(Splitter.on("|").split(conf.get(ODPSMapper.ODPS_COLUMN_INFO_CONFKEY)));
            Preconditions.checkArgument(sourceTableColumns.size() == targetTableColumns.size());
            LOG.info("Initialize " + odpsSplit);
        } catch (TunnelException e) {
            throw new IOException(e.getMessage());
        }
    }

    @Override
    public boolean nextKeyValue() throws IOException, InterruptedException {
        if (key == null) {
            key = NullWritable.get();
        }

        record = reader.read();
        if (record == null) {
            return false;
        }

        value =  ReflectionUtils.newInstance(inputClass, conf);
        for (int i = 0; i < sourceTableColumns.size(); i++) {
            String col = sourceTableColumns.get(i);
            String v;
            if (record.get(col) == null) {
                v = "";
            } else {
                v = record.get(col).toString();
            }
            value.put(new Text(targetTableColumns.get(i).getColumnName()), new Text(v));
        }
        return true;
    }

    @Override
    public NullWritable getCurrentKey() throws IOException, InterruptedException {
        return key;
    }

    @Override
    public T getCurrentValue() throws IOException, InterruptedException {
        return value;
    }

    @Override
    public float getProgress() throws IOException, InterruptedException {
        return 0;
    }

    @Override
    public void close() throws IOException {
        reader.close();
    }
}
