/*
 * Copyright 2017 Baidu, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package com.baidu.aip.contentcensor;

import com.baidu.aip.client.BaseClient;
import com.baidu.aip.error.AipError;
import com.baidu.aip.http.AipRequest;
import com.baidu.aip.http.EBodyFormat;
import com.baidu.aip.http.Headers;
import com.baidu.aip.http.HttpContentType;
import com.baidu.aip.util.Base64Util;
import com.baidu.aip.util.ImageUtil;
import com.baidu.aip.util.Util;
import org.json.JSONArray;
import org.json.JSONObject;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

public class AipContentCensor extends BaseClient {

    public AipContentCensor(String appId, String aipKey, String aipToken) {
        super(appId, aipKey, aipToken);
    }

    /**
     * 反馈接口
     *
     * @param reportData 反馈图片识别结果好坏的json数组
     * @return JSONObject
     */
    public JSONObject report(JSONArray reportData) {
        AipRequest request = new AipRequest();
        preOperation(request);
        request.addBody("feedback", reportData);
        request.setUri(ContentCensorConsts.REPORT_URL);
        request.setBodyFormat(EBodyFormat.RAW_JSON);
        request.addHeader(Headers.CONTENT_TYPE, HttpContentType.JSON_DATA);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 图像审核接口
     * 本接口除了支持自定义配置外，还对返回结果进行了总体的包装，按照用户在控制台中配置的规则直接返回是否合规，如果不合规则指出具体不合规的内容。
     *
     * @param image   本地图片路径或图片url
     * @param type    image参数类型：FILE或URL
     * @param options 可选参数
     * @return JSONObject
     */
    public JSONObject imageCensorUserDefined(String image, EImgType type, HashMap<String, String> options) {
        if (type == EImgType.FILE) {
            try {
                byte[] imgData = Util.readFileByBytes(image);
                return imageCensorUserDefined(imgData, options);
            } catch (IOException e) {
                return AipError.IMAGE_READ_ERROR.toJsonResult();
            }
        }

        // url
        AipRequest request = new AipRequest();

        request.addBody("imgUrl", image);

        return imageCensorUserDefinedHelper(request, options);
    }

    /**
     * 图像审核接口
     * 本接口除了支持自定义配置外，还对返回结果进行了总体的包装，按照用户在控制台中配置的规则直接返回是否合规，如果不合规则指出具体不合规的内容。
     *
     * @param imgData 图片二进制数据
     * @param options 可选参数
     * @return JSONObject
     */
    public JSONObject imageCensorUserDefined(byte[] imgData, HashMap<String, String> options) {
        AipRequest request = new AipRequest();

        String base64Content = Base64Util.encode(imgData);
        request.addBody("image", base64Content);

        return imageCensorUserDefinedHelper(request, options);
    }

    private JSONObject imageCensorUserDefinedHelper(AipRequest request, HashMap<String, String> options) {
        preOperation(request);

        if (options != null) {
            for (Map.Entry<String, String> entry : options.entrySet()) {
                request.addBody(entry.getKey(), entry.getValue());
            }
        }
        request.setUri(ContentCensorConsts.USER_DEFINED_IMAGE_URL);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 文本审核接口
     * 本接口除了支持自定义配置外，还对返回结果进行了总体的包装，按照用户在控制台中配置的规则直接返回是否合规，如果不合规则指出具体不合规的内容。
     *
     * @param text 文本
     * @return JSONObject
     */
    public JSONObject textCensorUserDefined(String text) {
        AipRequest request = new AipRequest();

        request.addBody("text", text);

        return textCensorUserDefinedHelper(request, null);
    }

    private JSONObject textCensorUserDefinedHelper(AipRequest request, HashMap<String, String> options) {
        preOperation(request);

        if (options != null) {
            for (Map.Entry<String, String> entry : options.entrySet()) {
                request.addBody(entry.getKey(), entry.getValue());
            }
        }
        request.setUri(ContentCensorConsts.USER_DEFINED_TEXT_URL);
        postOperation(request);
        return requestServer(request);
    }

    private JSONObject checkParam(byte[] imgData) {
        // image format
        String format = ImageUtil.getImageFormatByBytes(imgData);
        if (!ContentCensorConsts.ANTIPORN_SUPPORT_IMAGE_FORMAT.contains(format)) {
            return AipError.UNSUPPORTED_IMAGE_FORMAT_ERROR.toJsonResult();
        }

        return AipError.SUCCESS.toJsonResult();
    }

    private JSONObject checkImgFormat(byte[] imgData, String format) {
        String realFormat = ImageUtil.getImageFormatByBytes(imgData);
        if (realFormat.equals(format)) {
            return AipError.SUCCESS.toJsonResult();
        }
        return AipError.UNSUPPORTED_IMAGE_FORMAT_ERROR.toJsonResult();
    }


    /**
     * 语音审核接口
     * 本接口除了支持自定义配置外，还对返回结果进行了总体的包装，按照用户在控制台中配置的规则直接返回是否合规，如果不合规则指出具体不合规的内容。
     *
     * @param voice   语音源
     * @param type    语音源类型
     * @param fmt     语音文件格式
     * @param options 可选参数
     * @return JSONObject
     */
    public JSONObject voiceCensorUserDefined(String voice, EImgType type, String fmt,
                                             HashMap<String, String> options) {
        if (type == EImgType.FILE) {
            try {
                return voiceFileCensorUserDefined(voice, fmt, options);
            } catch (IOException e) {
                return AipError.IMAGE_READ_ERROR.toJsonResult();
            }
        }
        AipRequest request = new AipRequest();
        request.addBody("url", voice);
        request.addBody("fmt", fmt);
        return voiceCensorUserDefinedHelper(request, options);
    }

    /**
     * 语音审核接口
     * 本接口除了支持自定义配置外，还对返回结果进行了总体的包装，按照用户在控制台中配置的规则直接返回是否合规，如果不合规则指出具体不合规的内容。
     *
     * @param voice   语音二进制数据
     * @param fmt     语音文件类型
     * @param options 可选参数
     * @return JSONObject
     */
    public JSONObject voiceFileCensorUserDefined(String voice, String fmt, HashMap<String, String> options)
            throws IOException {
        byte[] imgData = Util.readFileByBytes(voice);
        AipRequest request = new AipRequest();

        String base64Content = Base64Util.encode(imgData);
        request.addBody("base64", base64Content);
        request.addBody("fmt", fmt);
        return voiceCensorUserDefinedHelper(request, options);
    }

    private JSONObject voiceCensorUserDefinedHelper(AipRequest request, HashMap<String, String> options) {
        preOperation(request);

        if (options != null) {
            for (Map.Entry<String, String> entry : options.entrySet()) {
                request.addBody(entry.getKey(), entry.getValue());
            }
        }
        request.setUri(ContentCensorConsts.USER_DEFINED_VOICE_URL);
        postOperation(request);
        return requestServer(request);
    }

    /**
     * 视频审核接口
     * 本接口除了支持自定义配置外，还对返回结果进行了总体的包装，按照用户在控制台中配置的规则直接返回是否合规，如果不合规则指出具体不合规的内容。
     *
     * @param name     视频名字
     * @param videoUrl 视频url
     * @param extId    视频id
     * @param options  可选参数
     * @return JSONObject
     */
    public JSONObject videoCensorUserDefined(String name, String videoUrl, String extId,
                                             HashMap<String, String> options) {

        AipRequest request = new AipRequest();
        request.addBody("name", name);
        request.addBody("videoUrl", videoUrl);
        request.addBody("extId", extId);
        return videoCensorUserDefinedHelper(request, options);
    }

    private JSONObject videoCensorUserDefinedHelper(AipRequest request, HashMap<String, String> options) {
        preOperation(request);

        if (options != null) {
            for (Map.Entry<String, String> entry : options.entrySet()) {
                request.addBody(entry.getKey(), entry.getValue());
            }
        }
        request.setUri(ContentCensorConsts.USER_DEFINED_VIDEO_URL);
        postOperation(request);
        return requestServer(request);
    }
}
