/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile.bucket;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.collect.MinMaxPriorityQueue;
import com.google.common.primitives.Ints;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.collections.map.LinkedMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.io.hfile.BlockCacheKey;
import org.apache.hadoop.hbase.io.hfile.bucket.BucketAllocatorException;
import org.apache.hadoop.hbase.io.hfile.bucket.BucketCache;
import org.apache.hadoop.hbase.io.hfile.bucket.CacheFullException;
import org.codehaus.jackson.annotate.JsonIgnoreProperties;

@InterfaceAudience.Private
@JsonIgnoreProperties(value={"indexStatistics", "freeSize", "usedSize"})
public final class BucketAllocator {
    private static final Log LOG = LogFactory.getLog(BucketAllocator.class);
    private static final int[] DEFAULT_BUCKET_SIZES = new int[]{5120, 9216, 17408, 33792, 41984, 50176, 58368, 66560, 99328, 132096, 197632, 263168, 394240, 525312};
    public static final int FEWEST_ITEMS_IN_BUCKET = 4;
    private final int[] bucketSizes;
    private final int bigItemSize;
    private final long bucketCapacity;
    private Bucket[] buckets;
    private BucketSizeInfo[] bucketSizeInfos;
    private final long totalSize;
    private long usedSize = 0L;

    public BucketSizeInfo roundUpToBucketSizeInfo(int blockSize) {
        for (int i = 0; i < this.bucketSizes.length; ++i) {
            if (blockSize > this.bucketSizes[i]) continue;
            return this.bucketSizeInfos[i];
        }
        return null;
    }

    BucketAllocator(long availableSpace, int[] bucketSizes) throws BucketAllocatorException {
        int i;
        this.bucketSizes = bucketSizes == null ? DEFAULT_BUCKET_SIZES : bucketSizes;
        Arrays.sort(this.bucketSizes);
        this.bigItemSize = Ints.max((int[])this.bucketSizes);
        this.bucketCapacity = 4 * this.bigItemSize;
        this.buckets = new Bucket[(int)(availableSpace / this.bucketCapacity)];
        if (this.buckets.length < this.bucketSizes.length) {
            throw new BucketAllocatorException("Bucket allocator size too small (" + this.buckets.length + "); must have room for at least " + this.bucketSizes.length + " buckets");
        }
        this.bucketSizeInfos = new BucketSizeInfo[this.bucketSizes.length];
        for (i = 0; i < this.bucketSizes.length; ++i) {
            this.bucketSizeInfos[i] = new BucketSizeInfo(i);
        }
        for (i = 0; i < this.buckets.length; ++i) {
            this.buckets[i] = new Bucket(this.bucketCapacity * (long)i);
            this.bucketSizeInfos[i < this.bucketSizes.length ? i : this.bucketSizes.length - 1].instantiateBucket(this.buckets[i]);
        }
        this.totalSize = (long)this.buckets.length * this.bucketCapacity;
        if (LOG.isInfoEnabled()) {
            LOG.info((Object)("Cache totalSize=" + this.totalSize + ", buckets=" + this.buckets.length + ", bucket capacity=" + this.bucketCapacity + "=(" + 4 + "*" + this.bigItemSize + ")=" + "(FEWEST_ITEMS_IN_BUCKET*(largest configured bucketcache size))"));
        }
    }

    BucketAllocator(long availableSpace, int[] bucketSizes, Map<BlockCacheKey, BucketCache.BucketEntry> map, AtomicLong realCacheSize) throws BucketAllocatorException {
        this(availableSpace, bucketSizes);
        boolean[] reconfigured = new boolean[this.buckets.length];
        int sizeNotMatchedCount = 0;
        int insufficientCapacityCount = 0;
        Iterator<Map.Entry<BlockCacheKey, BucketCache.BucketEntry>> iterator = map.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<BlockCacheKey, BucketCache.BucketEntry> entry = iterator.next();
            long foundOffset = entry.getValue().offset();
            int foundLen = entry.getValue().getLength();
            int bucketSizeIndex = -1;
            for (int i = 0; i < this.bucketSizes.length; ++i) {
                if (foundLen > this.bucketSizes[i]) continue;
                bucketSizeIndex = i;
                break;
            }
            if (bucketSizeIndex == -1) {
                ++sizeNotMatchedCount;
                iterator.remove();
                continue;
            }
            int bucketNo = (int)(foundOffset / this.bucketCapacity);
            if (bucketNo < 0 || bucketNo >= this.buckets.length) {
                ++insufficientCapacityCount;
                iterator.remove();
                continue;
            }
            Bucket b = this.buckets[bucketNo];
            if (reconfigured[bucketNo]) {
                if (b.sizeIndex() != bucketSizeIndex) {
                    throw new BucketAllocatorException("Inconsistent allocation in bucket map;");
                }
            } else {
                if (!b.isCompletelyFree()) {
                    throw new BucketAllocatorException("Reconfiguring bucket " + bucketNo + " but it's already allocated; corrupt data");
                }
                BucketSizeInfo bsi = this.bucketSizeInfos[bucketSizeIndex];
                BucketSizeInfo oldbsi = this.bucketSizeInfos[b.sizeIndex()];
                oldbsi.removeBucket(b);
                bsi.instantiateBucket(b);
                reconfigured[bucketNo] = true;
            }
            realCacheSize.addAndGet(foundLen);
            this.buckets[bucketNo].addAllocation(foundOffset);
            this.usedSize += (long)this.buckets[bucketNo].getItemAllocationSize();
            this.bucketSizeInfos[bucketSizeIndex].blockAllocated(b);
        }
        if (sizeNotMatchedCount > 0) {
            LOG.warn((Object)("There are " + sizeNotMatchedCount + " blocks which can't be rebuilt because " + "there is no matching bucket size for these blocks"));
        }
        if (insufficientCapacityCount > 0) {
            LOG.warn((Object)("There are " + insufficientCapacityCount + " blocks which can't be rebuilt - " + "did you shrink the cache?"));
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder(1024);
        for (int i = 0; i < this.buckets.length; ++i) {
            Bucket b = this.buckets[i];
            if (i > 0) {
                sb.append(", ");
            }
            sb.append("bucket.").append(i).append(": size=").append(b.getItemAllocationSize());
            sb.append(", freeCount=").append(b.freeCount()).append(", used=").append(b.usedCount());
        }
        return sb.toString();
    }

    public long getUsedSize() {
        return this.usedSize;
    }

    public long getFreeSize() {
        return this.totalSize - this.getUsedSize();
    }

    public long getTotalSize() {
        return this.totalSize;
    }

    public synchronized long allocateBlock(int blockSize) throws CacheFullException, BucketAllocatorException {
        assert (blockSize > 0);
        BucketSizeInfo bsi = this.roundUpToBucketSizeInfo(blockSize);
        if (bsi == null) {
            throw new BucketAllocatorException("Allocation too big size=" + blockSize + "; adjust BucketCache sizes " + "hbase.bucketcache.bucket.sizes" + " to accomodate if size seems reasonable and you want it cached.");
        }
        long offset = bsi.allocateBlock();
        if (offset < 0L) {
            throw new CacheFullException(blockSize, bsi.sizeIndex());
        }
        this.usedSize += (long)this.bucketSizes[bsi.sizeIndex()];
        return offset;
    }

    private Bucket grabGlobalCompletelyFreeBucket() {
        for (BucketSizeInfo bsi : this.bucketSizeInfos) {
            Bucket b = bsi.findAndRemoveCompletelyFreeBucket();
            if (b == null) continue;
            return b;
        }
        return null;
    }

    public synchronized int freeBlock(long offset) {
        int bucketNo = (int)(offset / this.bucketCapacity);
        assert (bucketNo >= 0 && bucketNo < this.buckets.length);
        Bucket targetBucket = this.buckets[bucketNo];
        this.bucketSizeInfos[targetBucket.sizeIndex()].freeBlock(targetBucket, offset);
        this.usedSize -= (long)targetBucket.getItemAllocationSize();
        return targetBucket.getItemAllocationSize();
    }

    public int sizeIndexOfAllocation(long offset) {
        int bucketNo = (int)(offset / this.bucketCapacity);
        assert (bucketNo >= 0 && bucketNo < this.buckets.length);
        Bucket targetBucket = this.buckets[bucketNo];
        return targetBucket.sizeIndex();
    }

    public int sizeOfAllocation(long offset) {
        int bucketNo = (int)(offset / this.bucketCapacity);
        assert (bucketNo >= 0 && bucketNo < this.buckets.length);
        Bucket targetBucket = this.buckets[bucketNo];
        return targetBucket.getItemAllocationSize();
    }

    public Bucket[] getBuckets() {
        return this.buckets;
    }

    void logStatistics() {
        IndexStatistics total = new IndexStatistics();
        IndexStatistics[] stats = this.getIndexStatistics(total);
        LOG.info((Object)"Bucket allocator statistics follow:\n");
        LOG.info((Object)("  Free bytes=" + total.freeBytes() + "+; used bytes=" + total.usedBytes() + "; total bytes=" + total.totalBytes()));
        for (IndexStatistics s : stats) {
            LOG.info((Object)("  Object size " + s.itemSize() + " used=" + s.usedCount() + "; free=" + s.freeCount() + "; total=" + s.totalCount()));
        }
    }

    IndexStatistics[] getIndexStatistics(IndexStatistics grandTotal) {
        IndexStatistics[] stats = this.getIndexStatistics();
        long totalfree = 0L;
        long totalused = 0L;
        for (IndexStatistics stat : stats) {
            totalfree += stat.freeBytes();
            totalused += stat.usedBytes();
        }
        grandTotal.setTo(totalfree, totalused, 1L);
        return stats;
    }

    IndexStatistics[] getIndexStatistics() {
        IndexStatistics[] stats = new IndexStatistics[this.bucketSizes.length];
        for (int i = 0; i < stats.length; ++i) {
            stats[i] = this.bucketSizeInfos[i].statistics();
        }
        return stats;
    }

    public long freeBlock(long[] freeList) {
        long sz = 0L;
        for (int i = 0; i < freeList.length; ++i) {
            sz += (long)this.freeBlock(freeList[i]);
        }
        return sz;
    }

    public int getBucketIndex(long offset) {
        return (int)(offset / this.bucketCapacity);
    }

    public Set<Integer> getLeastFilledBuckets(Set<Integer> excludedBuckets, int bucketCount) {
        MinMaxPriorityQueue queue = MinMaxPriorityQueue.orderedBy((Comparator)new Comparator<Integer>(){

            @Override
            public int compare(Integer left, Integer right) {
                return Float.compare((float)BucketAllocator.this.buckets[left].usedCount / (float)BucketAllocator.this.buckets[left].itemCount, (float)BucketAllocator.this.buckets[right].usedCount / (float)BucketAllocator.this.buckets[right].itemCount);
            }
        }).maximumSize(bucketCount).create();
        for (int i = 0; i < this.buckets.length; ++i) {
            if (excludedBuckets.contains(i) || this.buckets[i].isUninstantiated() || this.bucketSizeInfos[this.buckets[i].sizeIndex()].bucketList.size() == 1) continue;
            queue.add(i);
        }
        HashSet<Integer> result = new HashSet<Integer>(bucketCount);
        result.addAll((Collection<Integer>)queue);
        return result;
    }

    static class IndexStatistics {
        private long freeCount;
        private long usedCount;
        private long itemSize;
        private long totalCount;

        public long freeCount() {
            return this.freeCount;
        }

        public long usedCount() {
            return this.usedCount;
        }

        public long totalCount() {
            return this.totalCount;
        }

        public long freeBytes() {
            return this.freeCount * this.itemSize;
        }

        public long usedBytes() {
            return this.usedCount * this.itemSize;
        }

        public long totalBytes() {
            return this.totalCount * this.itemSize;
        }

        public long itemSize() {
            return this.itemSize;
        }

        public IndexStatistics(long free, long used, long itemSize) {
            this.setTo(free, used, itemSize);
        }

        public IndexStatistics() {
            this.setTo(-1L, -1L, 0L);
        }

        public void setTo(long free, long used, long itemSize) {
            this.itemSize = itemSize;
            this.freeCount = free;
            this.usedCount = used;
            this.totalCount = free + used;
        }
    }

    final class BucketSizeInfo {
        private LinkedMap bucketList = new LinkedMap();
        private LinkedMap freeBuckets = new LinkedMap();
        private LinkedMap completelyFreeBuckets = new LinkedMap();
        private int sizeIndex;

        BucketSizeInfo(int sizeIndex) {
            this.sizeIndex = sizeIndex;
        }

        public synchronized void instantiateBucket(Bucket b) {
            assert (b.isUninstantiated() || b.isCompletelyFree());
            b.reconfigure(this.sizeIndex, BucketAllocator.this.bucketSizes, BucketAllocator.this.bucketCapacity);
            this.bucketList.put((Object)b, (Object)b);
            this.freeBuckets.put((Object)b, (Object)b);
            this.completelyFreeBuckets.put((Object)b, (Object)b);
        }

        public int sizeIndex() {
            return this.sizeIndex;
        }

        public long allocateBlock() {
            Bucket b = null;
            if (this.freeBuckets.size() > 0) {
                b = (Bucket)this.freeBuckets.lastKey();
            }
            if (b == null && (b = BucketAllocator.this.grabGlobalCompletelyFreeBucket()) != null) {
                this.instantiateBucket(b);
            }
            if (b == null) {
                return -1L;
            }
            long result = b.allocate();
            this.blockAllocated(b);
            return result;
        }

        void blockAllocated(Bucket b) {
            if (!b.isCompletelyFree()) {
                this.completelyFreeBuckets.remove((Object)b);
            }
            if (!b.hasFreeSpace()) {
                this.freeBuckets.remove((Object)b);
            }
        }

        public Bucket findAndRemoveCompletelyFreeBucket() {
            Bucket b = null;
            assert (this.bucketList.size() > 0);
            if (this.bucketList.size() == 1) {
                return null;
            }
            if (this.completelyFreeBuckets.size() > 0) {
                b = (Bucket)this.completelyFreeBuckets.firstKey();
                this.removeBucket(b);
            }
            return b;
        }

        private synchronized void removeBucket(Bucket b) {
            assert (b.isCompletelyFree());
            this.bucketList.remove((Object)b);
            this.freeBuckets.remove((Object)b);
            this.completelyFreeBuckets.remove((Object)b);
        }

        public void freeBlock(Bucket b, long offset) {
            assert (this.bucketList.containsKey((Object)b));
            assert (!this.completelyFreeBuckets.containsKey((Object)b));
            b.free(offset);
            if (!this.freeBuckets.containsKey((Object)b)) {
                this.freeBuckets.put((Object)b, (Object)b);
            }
            if (b.isCompletelyFree()) {
                this.completelyFreeBuckets.put((Object)b, (Object)b);
            }
        }

        public synchronized IndexStatistics statistics() {
            long free = 0L;
            long used = 0L;
            for (Object obj : this.bucketList.keySet()) {
                Bucket b = (Bucket)obj;
                free += (long)b.freeCount();
                used += (long)b.usedCount();
            }
            return new IndexStatistics(free, used, BucketAllocator.this.bucketSizes[this.sizeIndex]);
        }

        public String toString() {
            return Objects.toStringHelper(this.getClass()).add("sizeIndex", this.sizeIndex).add("bucketSize", BucketAllocator.this.bucketSizes[this.sizeIndex]).toString();
        }
    }

    @JsonIgnoreProperties(value={"completelyFree", "uninstantiated"})
    public static final class Bucket {
        private long baseOffset;
        private int itemAllocationSize;
        private int sizeIndex;
        private int itemCount;
        private int[] freeList;
        private int freeCount;
        private int usedCount;

        public Bucket(long offset) {
            this.baseOffset = offset;
            this.sizeIndex = -1;
        }

        void reconfigure(int sizeIndex, int[] bucketSizes, long bucketCapacity) {
            Preconditions.checkElementIndex((int)sizeIndex, (int)bucketSizes.length);
            this.sizeIndex = sizeIndex;
            this.itemAllocationSize = bucketSizes[sizeIndex];
            this.freeCount = this.itemCount = (int)(bucketCapacity / (long)this.itemAllocationSize);
            this.usedCount = 0;
            this.freeList = new int[this.itemCount];
            for (int i = 0; i < this.freeCount; ++i) {
                this.freeList[i] = i;
            }
        }

        public boolean isUninstantiated() {
            return this.sizeIndex == -1;
        }

        public int sizeIndex() {
            return this.sizeIndex;
        }

        public int getItemAllocationSize() {
            return this.itemAllocationSize;
        }

        public boolean hasFreeSpace() {
            return this.freeCount > 0;
        }

        public boolean isCompletelyFree() {
            return this.usedCount == 0;
        }

        public int freeCount() {
            return this.freeCount;
        }

        public int usedCount() {
            return this.usedCount;
        }

        public int getFreeBytes() {
            return this.freeCount * this.itemAllocationSize;
        }

        public int getUsedBytes() {
            return this.usedCount * this.itemAllocationSize;
        }

        public long getBaseOffset() {
            return this.baseOffset;
        }

        public long allocate() {
            assert (this.freeCount > 0);
            assert (this.sizeIndex != -1);
            ++this.usedCount;
            long offset = this.baseOffset + (long)(this.freeList[--this.freeCount] * this.itemAllocationSize);
            assert (offset >= 0L);
            return offset;
        }

        public void addAllocation(long offset) throws BucketAllocatorException {
            if ((offset -= this.baseOffset) < 0L || offset % (long)this.itemAllocationSize != 0L) {
                throw new BucketAllocatorException("Attempt to add allocation for bad offset: " + offset + " base=" + this.baseOffset + ", bucket size=" + this.itemAllocationSize);
            }
            int idx = (int)(offset / (long)this.itemAllocationSize);
            boolean matchFound = false;
            for (int i = 0; i < this.freeCount; ++i) {
                if (matchFound) {
                    this.freeList[i - 1] = this.freeList[i];
                    continue;
                }
                if (this.freeList[i] != idx) continue;
                matchFound = true;
            }
            if (!matchFound) {
                throw new BucketAllocatorException("Couldn't find match for index " + idx + " in free list");
            }
            ++this.usedCount;
            --this.freeCount;
        }

        private void free(long offset) {
            assert ((offset -= this.baseOffset) >= 0L);
            assert (offset < (long)(this.itemCount * this.itemAllocationSize));
            assert (offset % (long)this.itemAllocationSize == 0L);
            assert (this.usedCount > 0);
            assert (this.freeCount < this.itemCount);
            int item = (int)(offset / (long)this.itemAllocationSize);
            assert (!this.freeListContains(item));
            --this.usedCount;
            this.freeList[this.freeCount++] = item;
        }

        private boolean freeListContains(int blockNo) {
            for (int i = 0; i < this.freeCount; ++i) {
                if (this.freeList[i] != blockNo) continue;
                return true;
            }
            return false;
        }
    }
}

