/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.cleaner;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import edu.umd.cs.findbugs.annotations.SuppressWarnings;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ForkJoinTask;
import java.util.concurrent.RecursiveTask;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathIsNotEmptyDirectoryException;
import org.apache.hadoop.hbase.ScheduledChore;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.conf.ConfigurationObserver;
import org.apache.hadoop.hbase.master.cleaner.FileCleanerDelegate;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.FileStatusFilter;
import org.apache.hadoop.ipc.RemoteException;

@InterfaceAudience.Private
@SuppressWarnings(value={"ST_WRITE_TO_STATIC_FROM_INSTANCE_METHOD"}, justification="Static pool will be only updated once.")
public abstract class CleanerChore<T extends FileCleanerDelegate>
extends ScheduledChore
implements ConfigurationObserver {
    private static final Log LOG = LogFactory.getLog((String)CleanerChore.class.getName());
    private static final int AVAIL_PROCESSORS = Runtime.getRuntime().availableProcessors();
    public static final String CHORE_POOL_SIZE = "hbase.cleaner.scan.dir.concurrent.size";
    private static final String DEFAULT_CHORE_POOL_SIZE = "0.25";
    private static volatile DirScanPool POOL;
    protected final FileSystem fs;
    private final Path oldFileDir;
    private final Configuration conf;
    protected List<T> cleanersChain;
    protected Map<String, Object> params;
    private AtomicBoolean enabled = new AtomicBoolean(true);

    public static void initChorePool(Configuration conf) {
        if (POOL == null) {
            POOL = new DirScanPool(conf);
        }
    }

    public static void shutDownChorePool() {
        if (POOL != null) {
            POOL.shutDownNow();
            POOL = null;
        }
    }

    public CleanerChore(String name, int sleepPeriod, Stoppable s, Configuration conf, FileSystem fs, Path oldFileDir, String confKey) {
        this(name, sleepPeriod, s, conf, fs, oldFileDir, confKey, null);
    }

    public CleanerChore(String name, int sleepPeriod, Stoppable s, Configuration conf, FileSystem fs, Path oldFileDir, String confKey, Map<String, Object> params) {
        super(name, s, sleepPeriod);
        Preconditions.checkNotNull((Object)POOL, (Object)"Chore's pool isn't initialized, please callCleanerChore.initChorePool(Configuration) before new a cleaner chore.");
        this.fs = fs;
        this.oldFileDir = oldFileDir;
        this.conf = conf;
        this.params = params;
        this.initCleanerChain(confKey);
    }

    static int calculatePoolSize(String poolSize) {
        if (poolSize.matches("[1-9][0-9]*")) {
            int size = Math.min(Integer.parseInt(poolSize), AVAIL_PROCESSORS);
            if (size == AVAIL_PROCESSORS) {
                LOG.warn((Object)("Use full core processors to scan dir, size=" + size));
            }
            return size;
        }
        if (poolSize.matches("0.[0-9]+|1.0")) {
            int computedThreads = (int)((double)AVAIL_PROCESSORS * Double.valueOf(poolSize));
            if (computedThreads < 1) {
                LOG.debug((Object)("Computed " + computedThreads + " threads for CleanerChore, using 1 instead"));
                return 1;
            }
            return computedThreads;
        }
        LOG.error((Object)("Unrecognized value: " + poolSize + " for " + CHORE_POOL_SIZE + ", use default config: " + DEFAULT_CHORE_POOL_SIZE + " instead."));
        return CleanerChore.calculatePoolSize(DEFAULT_CHORE_POOL_SIZE);
    }

    protected abstract boolean validate(Path var1);

    private void initCleanerChain(String confKey) {
        this.cleanersChain = new LinkedList<T>();
        String[] logCleaners = this.conf.getStrings(confKey);
        if (logCleaners != null) {
            for (String className : logCleaners) {
                T logCleaner = this.newFileCleaner(className, this.conf);
                if (logCleaner == null) continue;
                LOG.debug((Object)("initialize cleaner=" + className));
                this.cleanersChain.add(logCleaner);
            }
        }
    }

    @Override
    public void onConfigurationChange(Configuration conf) {
        POOL.markUpdate(conf);
    }

    private T newFileCleaner(String className, Configuration conf) {
        try {
            Class<FileCleanerDelegate> c = Class.forName(className).asSubclass(FileCleanerDelegate.class);
            FileCleanerDelegate cleaner = c.newInstance();
            cleaner.setConf(conf);
            cleaner.init(this.params);
            return (T)cleaner;
        }
        catch (Exception e) {
            LOG.warn((Object)("Can NOT create CleanerDelegate: " + className), (Throwable)e);
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void chore() {
        if (this.getEnabled()) {
            try {
                POOL.latchCountUp();
                if (this.runCleaner().booleanValue()) {
                    if (LOG.isTraceEnabled()) {
                        LOG.trace((Object)("Cleaned all WALs under " + this.oldFileDir));
                    }
                } else if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)("WALs outstanding under " + this.oldFileDir));
                }
            }
            finally {
                POOL.latchCountDown();
            }
            if (CleanerChore.POOL.reconfigNotification.compareAndSet(true, false)) {
                POOL.updatePool((long)(0.8 * (double)this.getTimeUnit().toMillis(this.getPeriod())));
            }
        } else {
            LOG.trace((Object)"Cleaner chore disabled! Not cleaning.");
        }
    }

    public Boolean runCleaner() {
        CleanerTask task = new CleanerTask(this.oldFileDir, true);
        POOL.submit(task);
        return (Boolean)task.join();
    }

    private boolean checkAndDeleteFiles(List<FileStatus> files) {
        if (files == null) {
            return true;
        }
        ArrayList validFiles = Lists.newArrayListWithCapacity((int)files.size());
        ArrayList invalidFiles = Lists.newArrayList();
        for (FileStatus file : files) {
            if (this.validate(file.getPath())) {
                validFiles.add(file);
                continue;
            }
            LOG.warn((Object)("Found a wrongly formatted file: " + file.getPath() + " - will delete it."));
            invalidFiles.add(file);
        }
        Iterable<Object> deletableValidFiles = validFiles;
        for (FileCleanerDelegate cleaner : this.cleanersChain) {
            if (cleaner.isStopped() || this.getStopper().isStopped()) {
                LOG.warn((Object)("A file cleaner" + this.getName() + " is stopped, won't delete any more files in:" + this.oldFileDir));
                return false;
            }
            Iterable<FileStatus> filteredFiles = cleaner.getDeletableFiles((Iterable<FileStatus>)deletableValidFiles);
            if (LOG.isTraceEnabled()) {
                ImmutableSet filteredFileSet = ImmutableSet.copyOf(filteredFiles);
                for (FileStatus fileStatus : deletableValidFiles) {
                    if (filteredFileSet.contains((Object)fileStatus)) continue;
                    LOG.trace((Object)(fileStatus.getPath() + " is not deletable according to:" + cleaner));
                }
            }
            deletableValidFiles = filteredFiles;
        }
        Iterable filesToDelete = Iterables.concat((Iterable)invalidFiles, (Iterable)deletableValidFiles);
        return this.deleteFiles(filesToDelete) == files.size();
    }

    protected int deleteFiles(Iterable<FileStatus> filesToDelete) {
        int deletedFileCount = 0;
        for (FileStatus file : filesToDelete) {
            Path filePath = file.getPath();
            LOG.trace((Object)("Removing " + file + " from archive"));
            try {
                boolean success = this.fs.delete(filePath, false);
                if (success) {
                    ++deletedFileCount;
                    continue;
                }
                LOG.warn((Object)("Attempted to delete:" + filePath + ", but couldn't. Run cleaner chain and attempt to delete on next pass."));
            }
            catch (IOException e) {
                e = e instanceof RemoteException ? ((RemoteException)((Object)e)).unwrapRemoteException() : e;
                LOG.warn((Object)("Error while deleting: " + filePath), (Throwable)e);
            }
        }
        return deletedFileCount;
    }

    public void cleanup() {
        for (FileCleanerDelegate lc : this.cleanersChain) {
            try {
                lc.stop("Exiting");
            }
            catch (Throwable t) {
                LOG.warn((Object)"Stopping", t);
            }
        }
    }

    @VisibleForTesting
    int getChorePoolSize() {
        return CleanerChore.POOL.size;
    }

    public boolean setEnabled(boolean enabled) {
        return this.enabled.getAndSet(enabled);
    }

    public boolean getEnabled() {
        return this.enabled.get();
    }

    private class CleanerTask
    extends RecursiveTask<Boolean> {
        private final Path dir;
        private final boolean root;

        CleanerTask(FileStatus dir, boolean root) {
            this(dir.getPath(), root);
        }

        CleanerTask(Path dir, boolean root) {
            this.dir = dir;
            this.root = root;
        }

        @Override
        protected Boolean compute() {
            boolean result;
            List<Object> files;
            List<Object> subDirs;
            LOG.trace((Object)("Cleaning under " + this.dir));
            try {
                List<FileStatus> tmpFiles;
                subDirs = FSUtils.listStatusWithStatusFilter(CleanerChore.this.fs, this.dir, new FileStatusFilter(){

                    @Override
                    public boolean accept(FileStatus f) {
                        return f.isDirectory();
                    }
                });
                if (subDirs == null) {
                    subDirs = Collections.emptyList();
                }
                files = (tmpFiles = FSUtils.listStatusWithStatusFilter(CleanerChore.this.fs, this.dir, new FileStatusFilter(){

                    @Override
                    public boolean accept(FileStatus f) {
                        return f.isFile();
                    }
                })) == null ? Collections.emptyList() : tmpFiles;
            }
            catch (IOException ioe) {
                LOG.warn((Object)("failed to get FileStatus for contents of '" + this.dir + "'"), (Throwable)ioe);
                return false;
            }
            boolean allFilesDeleted = true;
            if (!files.isEmpty()) {
                allFilesDeleted = this.deleteAction(new Action<Boolean>(){

                    @Override
                    public Boolean act() throws IOException {
                        return CleanerChore.this.checkAndDeleteFiles(files);
                    }
                }, "files");
            }
            boolean allSubdirsDeleted = true;
            if (!subDirs.isEmpty()) {
                final ArrayList tasks = Lists.newArrayListWithCapacity((int)subDirs.size());
                for (FileStatus fileStatus : subDirs) {
                    CleanerTask task = new CleanerTask(fileStatus, false);
                    tasks.add(task);
                    task.fork();
                }
                allSubdirsDeleted = this.deleteAction(new Action<Boolean>(){

                    @Override
                    public Boolean act() throws IOException {
                        return CleanerTask.this.getCleanResult(tasks);
                    }
                }, "subdirs");
            }
            boolean bl = result = allFilesDeleted && allSubdirsDeleted;
            if (result && !this.root) {
                result &= this.deleteAction(new Action<Boolean>(){

                    @Override
                    public Boolean act() throws IOException {
                        return CleanerChore.this.fs.delete(CleanerTask.this.dir, false);
                    }
                }, "dir");
            }
            return result;
        }

        private boolean deleteAction(Action<Boolean> deletion, String type) {
            boolean deleted;
            try {
                LOG.trace((Object)("Start deleting " + type + " under " + this.dir));
                deleted = deletion.act();
            }
            catch (PathIsNotEmptyDirectoryException exception) {
                LOG.debug((Object)("Couldn't delete '" + this.dir + "' yet because it isn't empty. Probably transient. " + "exception details at TRACE."));
                LOG.trace((Object)("Couldn't delete '" + this.dir + "' yet because it isn't empty w/exception."), (Throwable)exception);
                deleted = false;
            }
            catch (IOException ioe) {
                LOG.info((Object)("Could not delete " + type + " under " + this.dir + ". might be transient; we'll " + "retry. if it keeps happening, use following exception when asking on mailing list."), (Throwable)ioe);
                deleted = false;
            }
            LOG.trace((Object)("Finish deleting " + type + " under " + this.dir + " deleted=" + deleted));
            return deleted;
        }

        private boolean getCleanResult(List<CleanerTask> tasks) throws IOException {
            boolean cleaned = true;
            try {
                for (CleanerTask task : tasks) {
                    cleaned &= ((Boolean)task.get()).booleanValue();
                }
            }
            catch (InterruptedException | ExecutionException e) {
                throw new IOException(e);
            }
            return cleaned;
        }
    }

    private static interface Action<T> {
        public T act() throws IOException;
    }

    private static class DirScanPool {
        int size;
        ForkJoinPool pool;
        int cleanerLatch;
        AtomicBoolean reconfigNotification;

        DirScanPool(Configuration conf) {
            String poolSize = conf.get(CleanerChore.CHORE_POOL_SIZE, CleanerChore.DEFAULT_CHORE_POOL_SIZE);
            this.size = CleanerChore.calculatePoolSize(poolSize);
            this.size = this.size == 0 ? CleanerChore.calculatePoolSize(CleanerChore.DEFAULT_CHORE_POOL_SIZE) : this.size;
            this.pool = new ForkJoinPool(this.size);
            LOG.info((Object)("Cleaner pool size is " + this.size));
            this.reconfigNotification = new AtomicBoolean(false);
            this.cleanerLatch = 0;
        }

        synchronized void markUpdate(Configuration conf) {
            int newSize = CleanerChore.calculatePoolSize(conf.get(CleanerChore.CHORE_POOL_SIZE, CleanerChore.DEFAULT_CHORE_POOL_SIZE));
            if (newSize == this.size) {
                LOG.trace((Object)("Size from configuration is same as previous=" + newSize + " no need to update."));
                return;
            }
            this.size = newSize;
            this.reconfigNotification.set(true);
        }

        synchronized void updatePool(long timeout) {
            long stopWaitTime = System.currentTimeMillis() + timeout;
            while (this.cleanerLatch != 0 && timeout > 0L) {
                try {
                    this.wait(timeout);
                    timeout = stopWaitTime - System.currentTimeMillis();
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    break;
                }
            }
            this.shutDownNow();
            LOG.info((Object)("Update chore's pool size from " + this.pool.getParallelism() + " to " + this.size));
            this.pool = new ForkJoinPool(this.size);
        }

        synchronized void latchCountUp() {
            ++this.cleanerLatch;
        }

        synchronized void latchCountDown() {
            --this.cleanerLatch;
            this.notifyAll();
        }

        synchronized void submit(ForkJoinTask task) {
            this.pool.submit(task);
        }

        synchronized void shutDownNow() {
            if (this.pool == null || this.pool.isShutdown()) {
                return;
            }
            this.pool.shutdownNow();
        }
    }
}

