/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.record;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Random;
import java.util.function.Supplier;
import org.apache.kafka.common.errors.UnsupportedCompressionTypeException;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.record.ControlRecordType;
import org.apache.kafka.common.record.ConvertedRecords;
import org.apache.kafka.common.record.EndTransactionMarker;
import org.apache.kafka.common.record.LegacyRecord;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.MemoryRecordsBuilder;
import org.apache.kafka.common.record.MutableRecordBatch;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.record.RecordBatch;
import org.apache.kafka.common.record.RecordConversionStats;
import org.apache.kafka.common.record.SimpleRecord;
import org.apache.kafka.common.record.TimestampType;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.test.TestUtils;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(value=Parameterized.class)
public class MemoryRecordsBuilderTest {
    private final CompressionType compressionType;
    private final int bufferOffset;
    private final Time time;

    public MemoryRecordsBuilderTest(int bufferOffset, CompressionType compressionType) {
        this.bufferOffset = bufferOffset;
        this.compressionType = compressionType;
        this.time = Time.SYSTEM;
    }

    @Test
    public void testWriteEmptyRecordSet() {
        byte magic = 0;
        this.assumeAtLeastV2OrNotZstd(magic);
        ByteBuffer buffer = ByteBuffer.allocate(128);
        buffer.position(this.bufferOffset);
        Supplier<MemoryRecordsBuilder> builderSupplier = () -> new MemoryRecordsBuilder(buffer, magic, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, -1L, -1, -1, false, false, -1, buffer.capacity());
        if (this.compressionType != CompressionType.ZSTD) {
            MemoryRecords records = builderSupplier.get().build();
            Assert.assertEquals((long)0L, (long)records.sizeInBytes());
            Assert.assertEquals((long)this.bufferOffset, (long)buffer.position());
        } else {
            Exception e = (Exception)Assert.assertThrows(IllegalArgumentException.class, () -> ((MemoryRecordsBuilder)builderSupplier.get()).build());
            Assert.assertEquals((Object)e.getMessage(), (Object)("ZStandard compression is not supported for magic " + magic));
        }
    }

    @Test
    public void testWriteTransactionalRecordSet() {
        ByteBuffer buffer = ByteBuffer.allocate(128);
        buffer.position(this.bufferOffset);
        long pid = 9809L;
        short epoch = 15;
        int sequence = 2342;
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, 2, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, pid, epoch, sequence, true, false, -1, buffer.capacity());
        builder.append(System.currentTimeMillis(), "foo".getBytes(), "bar".getBytes());
        MemoryRecords records = builder.build();
        List batches = Utils.toList(records.batches().iterator());
        Assert.assertEquals((long)1L, (long)batches.size());
        Assert.assertTrue((boolean)((MutableRecordBatch)batches.get(0)).isTransactional());
    }

    @Test(expected=IllegalArgumentException.class)
    public void testWriteTransactionalNotAllowedMagicV0() {
        ByteBuffer buffer = ByteBuffer.allocate(128);
        buffer.position(this.bufferOffset);
        long pid = 9809L;
        short epoch = 15;
        int sequence = 2342;
        new MemoryRecordsBuilder(buffer, 0, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, pid, epoch, sequence, true, false, -1, buffer.capacity());
    }

    @Test(expected=IllegalArgumentException.class)
    public void testWriteTransactionalNotAllowedMagicV1() {
        ByteBuffer buffer = ByteBuffer.allocate(128);
        buffer.position(this.bufferOffset);
        long pid = 9809L;
        short epoch = 15;
        int sequence = 2342;
        new MemoryRecordsBuilder(buffer, 1, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, pid, epoch, sequence, true, false, -1, buffer.capacity());
    }

    @Test(expected=IllegalArgumentException.class)
    public void testWriteControlBatchNotAllowedMagicV0() {
        ByteBuffer buffer = ByteBuffer.allocate(128);
        buffer.position(this.bufferOffset);
        long pid = 9809L;
        short epoch = 15;
        int sequence = 2342;
        new MemoryRecordsBuilder(buffer, 0, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, pid, epoch, sequence, false, true, -1, buffer.capacity());
    }

    @Test(expected=IllegalArgumentException.class)
    public void testWriteControlBatchNotAllowedMagicV1() {
        ByteBuffer buffer = ByteBuffer.allocate(128);
        buffer.position(this.bufferOffset);
        long pid = 9809L;
        short epoch = 15;
        int sequence = 2342;
        new MemoryRecordsBuilder(buffer, 1, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, pid, epoch, sequence, false, true, -1, buffer.capacity());
    }

    @Test(expected=IllegalArgumentException.class)
    public void testWriteTransactionalWithInvalidPID() {
        ByteBuffer buffer = ByteBuffer.allocate(128);
        buffer.position(this.bufferOffset);
        long pid = -1L;
        short epoch = 15;
        int sequence = 2342;
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, 2, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, pid, epoch, sequence, true, false, -1, buffer.capacity());
        builder.close();
    }

    @Test(expected=IllegalArgumentException.class)
    public void testWriteIdempotentWithInvalidEpoch() {
        ByteBuffer buffer = ByteBuffer.allocate(128);
        buffer.position(this.bufferOffset);
        long pid = 9809L;
        short epoch = -1;
        int sequence = 2342;
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, 2, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, pid, epoch, sequence, true, false, -1, buffer.capacity());
        builder.close();
    }

    @Test(expected=IllegalArgumentException.class)
    public void testWriteIdempotentWithInvalidBaseSequence() {
        ByteBuffer buffer = ByteBuffer.allocate(128);
        buffer.position(this.bufferOffset);
        long pid = 9809L;
        short epoch = 15;
        int sequence = -1;
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, 2, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, pid, epoch, sequence, true, false, -1, buffer.capacity());
        builder.close();
    }

    @Test(expected=IllegalArgumentException.class)
    public void testWriteEndTxnMarkerNonTransactionalBatch() {
        ByteBuffer buffer = ByteBuffer.allocate(128);
        buffer.position(this.bufferOffset);
        long pid = 9809L;
        short epoch = 15;
        int sequence = -1;
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, 2, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, pid, epoch, sequence, false, true, -1, buffer.capacity());
        builder.appendEndTxnMarker(-1L, new EndTransactionMarker(ControlRecordType.ABORT, 0));
    }

    @Test(expected=IllegalArgumentException.class)
    public void testWriteEndTxnMarkerNonControlBatch() {
        ByteBuffer buffer = ByteBuffer.allocate(128);
        buffer.position(this.bufferOffset);
        long pid = 9809L;
        short epoch = 15;
        int sequence = -1;
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, 2, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, pid, epoch, sequence, true, false, -1, buffer.capacity());
        builder.appendEndTxnMarker(-1L, new EndTransactionMarker(ControlRecordType.ABORT, 0));
    }

    @Test
    public void testCompressionRateV0() {
        byte magic = 0;
        this.assumeAtLeastV2OrNotZstd(magic);
        ByteBuffer buffer = ByteBuffer.allocate(1024);
        buffer.position(this.bufferOffset);
        LegacyRecord[] records = new LegacyRecord[]{LegacyRecord.create((byte)magic, (long)0L, (byte[])"a".getBytes(), (byte[])"1".getBytes()), LegacyRecord.create((byte)magic, (long)1L, (byte[])"b".getBytes(), (byte[])"2".getBytes()), LegacyRecord.create((byte)magic, (long)2L, (byte[])"c".getBytes(), (byte[])"3".getBytes())};
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, magic, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, -1L, -1, -1, false, false, -1, buffer.capacity());
        int uncompressedSize = 0;
        for (LegacyRecord record : records) {
            uncompressedSize += record.sizeInBytes() + 12;
            builder.append(record);
        }
        MemoryRecords built = builder.build();
        if (this.compressionType == CompressionType.NONE) {
            Assert.assertEquals((double)1.0, (double)builder.compressionRatio(), (double)1.0E-5);
        } else {
            int compressedSize = built.sizeInBytes() - 12 - 14;
            double computedCompressionRate = (double)compressedSize / (double)uncompressedSize;
            Assert.assertEquals((double)computedCompressionRate, (double)builder.compressionRatio(), (double)1.0E-5);
        }
    }

    @Test
    public void testEstimatedSizeInBytes() {
        ByteBuffer buffer = ByteBuffer.allocate(1024);
        buffer.position(this.bufferOffset);
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, 2, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, -1L, -1, -1, false, false, -1, buffer.capacity());
        int previousEstimate = 0;
        for (int i = 0; i < 10; ++i) {
            builder.append(new SimpleRecord((long)i, ("" + i).getBytes()));
            int currentEstimate = builder.estimatedSizeInBytes();
            Assert.assertTrue((currentEstimate > previousEstimate ? 1 : 0) != 0);
            previousEstimate = currentEstimate;
        }
        int bytesWrittenBeforeClose = builder.estimatedSizeInBytes();
        MemoryRecords records = builder.build();
        Assert.assertEquals((long)records.sizeInBytes(), (long)builder.estimatedSizeInBytes());
        if (this.compressionType == CompressionType.NONE) {
            Assert.assertEquals((long)records.sizeInBytes(), (long)bytesWrittenBeforeClose);
        }
    }

    @Test
    public void testCompressionRateV1() {
        byte magic = 1;
        this.assumeAtLeastV2OrNotZstd(magic);
        ByteBuffer buffer = ByteBuffer.allocate(1024);
        buffer.position(this.bufferOffset);
        LegacyRecord[] records = new LegacyRecord[]{LegacyRecord.create((byte)magic, (long)0L, (byte[])"a".getBytes(), (byte[])"1".getBytes()), LegacyRecord.create((byte)magic, (long)1L, (byte[])"b".getBytes(), (byte[])"2".getBytes()), LegacyRecord.create((byte)magic, (long)2L, (byte[])"c".getBytes(), (byte[])"3".getBytes())};
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, magic, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, -1L, -1, -1, false, false, -1, buffer.capacity());
        int uncompressedSize = 0;
        for (LegacyRecord record : records) {
            uncompressedSize += record.sizeInBytes() + 12;
            builder.append(record);
        }
        MemoryRecords built = builder.build();
        if (this.compressionType == CompressionType.NONE) {
            Assert.assertEquals((double)1.0, (double)builder.compressionRatio(), (double)1.0E-5);
        } else {
            int compressedSize = built.sizeInBytes() - 12 - 22;
            double computedCompressionRate = (double)compressedSize / (double)uncompressedSize;
            Assert.assertEquals((double)computedCompressionRate, (double)builder.compressionRatio(), (double)1.0E-5);
        }
    }

    @Test
    public void buildUsingLogAppendTime() {
        byte magic = 1;
        this.assumeAtLeastV2OrNotZstd(magic);
        ByteBuffer buffer = ByteBuffer.allocate(1024);
        buffer.position(this.bufferOffset);
        long logAppendTime = System.currentTimeMillis();
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, magic, this.compressionType, TimestampType.LOG_APPEND_TIME, 0L, logAppendTime, -1L, -1, -1, false, false, -1, buffer.capacity());
        builder.append(0L, "a".getBytes(), "1".getBytes());
        builder.append(0L, "b".getBytes(), "2".getBytes());
        builder.append(0L, "c".getBytes(), "3".getBytes());
        MemoryRecords records = builder.build();
        MemoryRecordsBuilder.RecordsInfo info = builder.info();
        Assert.assertEquals((long)logAppendTime, (long)info.maxTimestamp);
        if (this.compressionType != CompressionType.NONE) {
            Assert.assertEquals((long)2L, (long)info.shallowOffsetOfMaxTimestamp);
        } else {
            Assert.assertEquals((long)0L, (long)info.shallowOffsetOfMaxTimestamp);
        }
        for (RecordBatch batch : records.batches()) {
            Assert.assertEquals((Object)TimestampType.LOG_APPEND_TIME, (Object)batch.timestampType());
            for (Record record : batch) {
                Assert.assertEquals((long)logAppendTime, (long)record.timestamp());
            }
        }
    }

    @Test
    public void buildUsingCreateTime() {
        byte magic = 1;
        this.assumeAtLeastV2OrNotZstd(magic);
        ByteBuffer buffer = ByteBuffer.allocate(1024);
        buffer.position(this.bufferOffset);
        long logAppendTime = System.currentTimeMillis();
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, magic, this.compressionType, TimestampType.CREATE_TIME, 0L, logAppendTime, -1L, -1, -1, false, false, -1, buffer.capacity());
        builder.append(0L, "a".getBytes(), "1".getBytes());
        builder.append(2L, "b".getBytes(), "2".getBytes());
        builder.append(1L, "c".getBytes(), "3".getBytes());
        MemoryRecords records = builder.build();
        MemoryRecordsBuilder.RecordsInfo info = builder.info();
        Assert.assertEquals((long)2L, (long)info.maxTimestamp);
        if (this.compressionType == CompressionType.NONE) {
            Assert.assertEquals((long)1L, (long)info.shallowOffsetOfMaxTimestamp);
        } else {
            Assert.assertEquals((long)2L, (long)info.shallowOffsetOfMaxTimestamp);
        }
        int i = 0;
        long[] expectedTimestamps = new long[]{0L, 2L, 1L};
        for (RecordBatch batch : records.batches()) {
            Assert.assertEquals((Object)TimestampType.CREATE_TIME, (Object)batch.timestampType());
            for (Record record : batch) {
                Assert.assertEquals((long)expectedTimestamps[i++], (long)record.timestamp());
            }
        }
    }

    @Test
    public void testAppendedChecksumConsistency() {
        this.assumeAtLeastV2OrNotZstd((byte)0);
        this.assumeAtLeastV2OrNotZstd((byte)1);
        ByteBuffer buffer = ByteBuffer.allocate(512);
        for (byte magic : Arrays.asList((byte)0, (byte)1, (byte)2)) {
            MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, magic, this.compressionType, TimestampType.CREATE_TIME, 0L, -1L, -1L, -1, -1, false, false, -1, buffer.capacity());
            Long checksumOrNull = builder.append(1L, "key".getBytes(), "value".getBytes());
            MemoryRecords memoryRecords = builder.build();
            List records = TestUtils.toList(memoryRecords.records());
            Assert.assertEquals((long)1L, (long)records.size());
            Assert.assertEquals((Object)checksumOrNull, (Object)((Record)records.get(0)).checksumOrNull());
        }
    }

    @Test
    public void testSmallWriteLimit() {
        byte[] key = "foo".getBytes();
        byte[] value = "bar".getBytes();
        int writeLimit = 0;
        ByteBuffer buffer = ByteBuffer.allocate(512);
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, 2, this.compressionType, TimestampType.CREATE_TIME, 0L, -1L, -1L, -1, -1, false, false, -1, writeLimit);
        Assert.assertFalse((boolean)builder.isFull());
        Assert.assertTrue((boolean)builder.hasRoomFor(0L, key, value, Record.EMPTY_HEADERS));
        builder.append(0L, key, value);
        Assert.assertTrue((boolean)builder.isFull());
        Assert.assertFalse((boolean)builder.hasRoomFor(0L, key, value, Record.EMPTY_HEADERS));
        MemoryRecords memRecords = builder.build();
        List records = TestUtils.toList(memRecords.records());
        Assert.assertEquals((long)1L, (long)records.size());
        Record record = (Record)records.get(0);
        Assert.assertEquals((Object)ByteBuffer.wrap(key), (Object)record.key());
        Assert.assertEquals((Object)ByteBuffer.wrap(value), (Object)record.value());
    }

    @Test
    public void writePastLimit() {
        byte magic = 1;
        this.assumeAtLeastV2OrNotZstd(magic);
        ByteBuffer buffer = ByteBuffer.allocate(64);
        buffer.position(this.bufferOffset);
        long logAppendTime = System.currentTimeMillis();
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, magic, this.compressionType, TimestampType.CREATE_TIME, 0L, logAppendTime, -1L, -1, -1, false, false, -1, buffer.capacity());
        builder.setEstimatedCompressionRatio(0.5f);
        builder.append(0L, "a".getBytes(), "1".getBytes());
        builder.append(1L, "b".getBytes(), "2".getBytes());
        Assert.assertFalse((boolean)builder.hasRoomFor(2L, "c".getBytes(), "3".getBytes(), Record.EMPTY_HEADERS));
        builder.append(2L, "c".getBytes(), "3".getBytes());
        MemoryRecords records = builder.build();
        MemoryRecordsBuilder.RecordsInfo info = builder.info();
        Assert.assertEquals((long)2L, (long)info.maxTimestamp);
        Assert.assertEquals((long)2L, (long)info.shallowOffsetOfMaxTimestamp);
        long i = 0L;
        for (RecordBatch batch : records.batches()) {
            Assert.assertEquals((Object)TimestampType.CREATE_TIME, (Object)batch.timestampType());
            for (Record record : batch) {
                Assert.assertEquals((long)i++, (long)record.timestamp());
            }
        }
    }

    @Test(expected=IllegalArgumentException.class)
    public void testAppendAtInvalidOffset() {
        ByteBuffer buffer = ByteBuffer.allocate(1024);
        buffer.position(this.bufferOffset);
        long logAppendTime = System.currentTimeMillis();
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, 1, this.compressionType, TimestampType.CREATE_TIME, 0L, logAppendTime, -1L, -1, -1, false, false, -1, buffer.capacity());
        builder.appendWithOffset(0L, System.currentTimeMillis(), "a".getBytes(), null);
        builder.appendWithOffset(0L, System.currentTimeMillis(), "b".getBytes(), null);
    }

    @Test
    public void convertV2ToV1UsingMixedCreateAndLogAppendTime() {
        ByteBuffer buffer = ByteBuffer.allocate(512);
        MemoryRecordsBuilder builder = MemoryRecords.builder((ByteBuffer)buffer, (byte)2, (CompressionType)this.compressionType, (TimestampType)TimestampType.LOG_APPEND_TIME, (long)0L);
        builder.append(10L, "1".getBytes(), "a".getBytes());
        builder.close();
        int sizeExcludingTxnMarkers = buffer.position();
        MemoryRecords.writeEndTransactionalMarker((ByteBuffer)buffer, (long)1L, (long)System.currentTimeMillis(), (int)0, (long)15L, (short)0, (EndTransactionMarker)new EndTransactionMarker(ControlRecordType.ABORT, 0));
        int position = buffer.position();
        builder = MemoryRecords.builder((ByteBuffer)buffer, (byte)2, (CompressionType)this.compressionType, (TimestampType)TimestampType.CREATE_TIME, (long)1L);
        builder.append(12L, "2".getBytes(), "b".getBytes());
        builder.append(13L, "3".getBytes(), "c".getBytes());
        builder.close();
        sizeExcludingTxnMarkers += buffer.position() - position;
        MemoryRecords.writeEndTransactionalMarker((ByteBuffer)buffer, (long)14L, (long)System.currentTimeMillis(), (int)0, (long)1L, (short)0, (EndTransactionMarker)new EndTransactionMarker(ControlRecordType.COMMIT, 0));
        buffer.flip();
        Supplier<ConvertedRecords> convertedRecordsSupplier = () -> MemoryRecords.readableRecords((ByteBuffer)buffer).downConvert((byte)1, 0L, this.time);
        if (this.compressionType != CompressionType.ZSTD) {
            ConvertedRecords convertedRecords = convertedRecordsSupplier.get();
            MemoryRecords records = (MemoryRecords)convertedRecords.records();
            this.verifyRecordsProcessingStats(convertedRecords.recordConversionStats(), 3, 3, records.sizeInBytes(), sizeExcludingTxnMarkers);
            List batches = Utils.toList(records.batches().iterator());
            if (this.compressionType != CompressionType.NONE) {
                Assert.assertEquals((long)2L, (long)batches.size());
                Assert.assertEquals((Object)TimestampType.LOG_APPEND_TIME, (Object)((RecordBatch)batches.get(0)).timestampType());
                Assert.assertEquals((Object)TimestampType.CREATE_TIME, (Object)((RecordBatch)batches.get(1)).timestampType());
            } else {
                Assert.assertEquals((long)3L, (long)batches.size());
                Assert.assertEquals((Object)TimestampType.LOG_APPEND_TIME, (Object)((RecordBatch)batches.get(0)).timestampType());
                Assert.assertEquals((Object)TimestampType.CREATE_TIME, (Object)((RecordBatch)batches.get(1)).timestampType());
                Assert.assertEquals((Object)TimestampType.CREATE_TIME, (Object)((RecordBatch)batches.get(2)).timestampType());
            }
            List logRecords = Utils.toList(records.records().iterator());
            Assert.assertEquals((long)3L, (long)logRecords.size());
            Assert.assertEquals((Object)ByteBuffer.wrap("1".getBytes()), (Object)((Record)logRecords.get(0)).key());
            Assert.assertEquals((Object)ByteBuffer.wrap("2".getBytes()), (Object)((Record)logRecords.get(1)).key());
            Assert.assertEquals((Object)ByteBuffer.wrap("3".getBytes()), (Object)((Record)logRecords.get(2)).key());
        } else {
            Exception e = (Exception)Assert.assertThrows(UnsupportedCompressionTypeException.class, convertedRecordsSupplier::get);
            Assert.assertEquals((Object)"Down-conversion of zstandard-compressed batches is not supported", (Object)e.getMessage());
        }
    }

    @Test
    public void convertToV1WithMixedV0AndV2Data() {
        this.assumeAtLeastV2OrNotZstd((byte)0);
        this.assumeAtLeastV2OrNotZstd((byte)1);
        ByteBuffer buffer = ByteBuffer.allocate(512);
        MemoryRecordsBuilder builder = MemoryRecords.builder((ByteBuffer)buffer, (byte)0, (CompressionType)this.compressionType, (TimestampType)TimestampType.NO_TIMESTAMP_TYPE, (long)0L);
        builder.append(-1L, "1".getBytes(), "a".getBytes());
        builder.close();
        builder = MemoryRecords.builder((ByteBuffer)buffer, (byte)2, (CompressionType)this.compressionType, (TimestampType)TimestampType.CREATE_TIME, (long)1L);
        builder.append(11L, "2".getBytes(), "b".getBytes());
        builder.append(12L, "3".getBytes(), "c".getBytes());
        builder.close();
        buffer.flip();
        ConvertedRecords convertedRecords = MemoryRecords.readableRecords((ByteBuffer)buffer).downConvert((byte)1, 0L, this.time);
        MemoryRecords records = (MemoryRecords)convertedRecords.records();
        this.verifyRecordsProcessingStats(convertedRecords.recordConversionStats(), 3, 2, records.sizeInBytes(), buffer.limit());
        List batches = Utils.toList(records.batches().iterator());
        if (this.compressionType != CompressionType.NONE) {
            Assert.assertEquals((long)2L, (long)batches.size());
            Assert.assertEquals((long)0L, (long)((RecordBatch)batches.get(0)).magic());
            Assert.assertEquals((long)0L, (long)((RecordBatch)batches.get(0)).baseOffset());
            Assert.assertEquals((long)1L, (long)((RecordBatch)batches.get(1)).magic());
            Assert.assertEquals((long)1L, (long)((RecordBatch)batches.get(1)).baseOffset());
        } else {
            Assert.assertEquals((long)3L, (long)batches.size());
            Assert.assertEquals((long)0L, (long)((RecordBatch)batches.get(0)).magic());
            Assert.assertEquals((long)0L, (long)((RecordBatch)batches.get(0)).baseOffset());
            Assert.assertEquals((long)1L, (long)((RecordBatch)batches.get(1)).magic());
            Assert.assertEquals((long)1L, (long)((RecordBatch)batches.get(1)).baseOffset());
            Assert.assertEquals((long)1L, (long)((RecordBatch)batches.get(2)).magic());
            Assert.assertEquals((long)2L, (long)((RecordBatch)batches.get(2)).baseOffset());
        }
        List logRecords = Utils.toList(records.records().iterator());
        Assert.assertEquals((Object)"1", (Object)Utils.utf8((ByteBuffer)((Record)logRecords.get(0)).key()));
        Assert.assertEquals((Object)"2", (Object)Utils.utf8((ByteBuffer)((Record)logRecords.get(1)).key()));
        Assert.assertEquals((Object)"3", (Object)Utils.utf8((ByteBuffer)((Record)logRecords.get(2)).key()));
        convertedRecords = MemoryRecords.readableRecords((ByteBuffer)buffer).downConvert((byte)1, 2L, this.time);
        records = (MemoryRecords)convertedRecords.records();
        batches = Utils.toList(records.batches().iterator());
        logRecords = Utils.toList(records.records().iterator());
        if (this.compressionType != CompressionType.NONE) {
            Assert.assertEquals((long)2L, (long)batches.size());
            Assert.assertEquals((long)0L, (long)((RecordBatch)batches.get(0)).magic());
            Assert.assertEquals((long)0L, (long)((RecordBatch)batches.get(0)).baseOffset());
            Assert.assertEquals((long)1L, (long)((RecordBatch)batches.get(1)).magic());
            Assert.assertEquals((long)1L, (long)((RecordBatch)batches.get(1)).baseOffset());
            Assert.assertEquals((Object)"1", (Object)Utils.utf8((ByteBuffer)((Record)logRecords.get(0)).key()));
            Assert.assertEquals((Object)"2", (Object)Utils.utf8((ByteBuffer)((Record)logRecords.get(1)).key()));
            Assert.assertEquals((Object)"3", (Object)Utils.utf8((ByteBuffer)((Record)logRecords.get(2)).key()));
            this.verifyRecordsProcessingStats(convertedRecords.recordConversionStats(), 3, 2, records.sizeInBytes(), buffer.limit());
        } else {
            Assert.assertEquals((long)2L, (long)batches.size());
            Assert.assertEquals((long)0L, (long)((RecordBatch)batches.get(0)).magic());
            Assert.assertEquals((long)0L, (long)((RecordBatch)batches.get(0)).baseOffset());
            Assert.assertEquals((long)1L, (long)((RecordBatch)batches.get(1)).magic());
            Assert.assertEquals((long)2L, (long)((RecordBatch)batches.get(1)).baseOffset());
            Assert.assertEquals((Object)"1", (Object)Utils.utf8((ByteBuffer)((Record)logRecords.get(0)).key()));
            Assert.assertEquals((Object)"3", (Object)Utils.utf8((ByteBuffer)((Record)logRecords.get(1)).key()));
            this.verifyRecordsProcessingStats(convertedRecords.recordConversionStats(), 3, 1, records.sizeInBytes(), buffer.limit());
        }
    }

    @Test
    public void shouldThrowIllegalStateExceptionOnBuildWhenAborted() {
        byte magic = 0;
        this.assumeAtLeastV2OrNotZstd(magic);
        ByteBuffer buffer = ByteBuffer.allocate(128);
        buffer.position(this.bufferOffset);
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, magic, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, -1L, -1, -1, false, false, -1, buffer.capacity());
        builder.abort();
        Assert.assertThrows(IllegalStateException.class, () -> ((MemoryRecordsBuilder)builder).build());
    }

    @Test
    public void shouldResetBufferToInitialPositionOnAbort() {
        byte magic = 0;
        this.assumeAtLeastV2OrNotZstd(magic);
        ByteBuffer buffer = ByteBuffer.allocate(128);
        buffer.position(this.bufferOffset);
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, magic, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, -1L, -1, -1, false, false, -1, buffer.capacity());
        builder.append(0L, "a".getBytes(), "1".getBytes());
        builder.abort();
        Assert.assertEquals((long)this.bufferOffset, (long)builder.buffer().position());
    }

    @Test
    public void shouldThrowIllegalStateExceptionOnCloseWhenAborted() {
        byte magic = 0;
        this.assumeAtLeastV2OrNotZstd(magic);
        ByteBuffer buffer = ByteBuffer.allocate(128);
        buffer.position(this.bufferOffset);
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, magic, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, -1L, -1, -1, false, false, -1, buffer.capacity());
        builder.abort();
        try {
            builder.close();
            Assert.fail((String)"Should have thrown IllegalStateException");
        }
        catch (IllegalStateException illegalStateException) {
            // empty catch block
        }
    }

    @Test
    public void shouldThrowIllegalStateExceptionOnAppendWhenAborted() {
        byte magic = 0;
        this.assumeAtLeastV2OrNotZstd(magic);
        ByteBuffer buffer = ByteBuffer.allocate(128);
        buffer.position(this.bufferOffset);
        MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, magic, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, -1L, -1, -1, false, false, -1, buffer.capacity());
        builder.abort();
        try {
            builder.append(0L, "a".getBytes(), "1".getBytes());
            Assert.fail((String)"Should have thrown IllegalStateException");
        }
        catch (IllegalStateException illegalStateException) {
            // empty catch block
        }
    }

    @Parameterized.Parameters(name="bufferOffset={0}, compression={1}")
    public static Collection<Object[]> data() {
        ArrayList<Object[]> values = new ArrayList<Object[]>();
        for (int bufferOffset : Arrays.asList(0, 15)) {
            for (CompressionType compressionType : CompressionType.values()) {
                values.add(new Object[]{bufferOffset, compressionType});
            }
        }
        return values;
    }

    @Test
    public void testBuffersDereferencedOnClose() {
        Runtime runtime = Runtime.getRuntime();
        int payloadLen = 0x100000;
        ByteBuffer buffer = ByteBuffer.allocate(payloadLen * 2);
        byte[] key = new byte[]{};
        byte[] value = new byte[payloadLen];
        new Random().nextBytes(value);
        ArrayList<MemoryRecordsBuilder> builders = new ArrayList<MemoryRecordsBuilder>(100);
        long startMem = 0L;
        long memUsed = 0L;
        int iterations = 0;
        while (iterations++ < 100) {
            buffer.rewind();
            MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, 2, this.compressionType, TimestampType.CREATE_TIME, 0L, 0L, -1L, -1, -1, false, false, -1, 0);
            builder.append(1L, new byte[0], value);
            builder.build();
            builders.add(builder);
            System.gc();
            memUsed = runtime.totalMemory() - runtime.freeMemory() - startMem;
            if (iterations == 2) {
                startMem = memUsed;
                continue;
            }
            if (iterations <= 2 || memUsed >= (long)((iterations - 2) * 1024)) continue;
            break;
        }
        Assert.assertTrue((String)("Memory usage too high: " + memUsed), (iterations < 100 ? 1 : 0) != 0);
    }

    private void verifyRecordsProcessingStats(RecordConversionStats processingStats, int numRecords, int numRecordsConverted, long finalBytes, long preConvertedBytes) {
        Assert.assertNotNull((String)"Records processing info is null", (Object)processingStats);
        Assert.assertEquals((long)numRecordsConverted, (long)processingStats.numRecordsConverted());
        Assert.assertTrue((String)("Processing time not recorded: " + processingStats), (processingStats.conversionTimeNanos() >= 0L ? 1 : 0) != 0);
        long tempBytes = processingStats.temporaryMemoryBytes();
        if (this.compressionType == CompressionType.NONE) {
            if (numRecordsConverted == 0) {
                Assert.assertEquals((long)finalBytes, (long)tempBytes);
            } else if (numRecordsConverted == numRecords) {
                Assert.assertEquals((long)(preConvertedBytes + finalBytes), (long)tempBytes);
            } else {
                Assert.assertTrue((String)String.format("Unexpected temp bytes %d final %d pre %d", tempBytes, finalBytes, preConvertedBytes), (tempBytes > finalBytes && tempBytes < finalBytes + preConvertedBytes ? 1 : 0) != 0);
            }
        } else {
            long compressedBytes = finalBytes - 12L - 14L;
            Assert.assertTrue((String)String.format("Uncompressed size expected temp=%d, compressed=%d", tempBytes, compressedBytes), (tempBytes > compressedBytes ? 1 : 0) != 0);
        }
    }

    private void assumeAtLeastV2OrNotZstd(byte magic) {
        Assume.assumeTrue((this.compressionType != CompressionType.ZSTD || magic >= 2 ? 1 : 0) != 0);
    }
}

