/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.geometry.utils;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Collections;
import org.elasticsearch.geometry.Circle;
import org.elasticsearch.geometry.Geometry;
import org.elasticsearch.geometry.GeometryCollection;
import org.elasticsearch.geometry.GeometryVisitor;
import org.elasticsearch.geometry.Line;
import org.elasticsearch.geometry.LinearRing;
import org.elasticsearch.geometry.MultiLine;
import org.elasticsearch.geometry.MultiPoint;
import org.elasticsearch.geometry.MultiPolygon;
import org.elasticsearch.geometry.Point;
import org.elasticsearch.geometry.Polygon;
import org.elasticsearch.geometry.Rectangle;
import org.elasticsearch.geometry.ShapeType;
import org.elasticsearch.geometry.utils.GeometryValidator;

public class WellKnownBinary {
    private WellKnownBinary() {
    }

    public static byte[] toWKB(Geometry geometry, ByteOrder byteOrder) throws IOException {
        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream();){
            WellKnownBinary.toWKB(geometry, outputStream, ByteBuffer.allocate(8).order(byteOrder));
            byte[] byArray = outputStream.toByteArray();
            return byArray;
        }
    }

    private static void toWKB(Geometry geometry, final ByteArrayOutputStream out, final ByteBuffer scratch) {
        out.write(scratch.order() == ByteOrder.BIG_ENDIAN ? 0 : 1);
        geometry.visit(new GeometryVisitor<Void, RuntimeException>(){

            @Override
            public Void visit(Point point) {
                if (point.isEmpty()) {
                    throw new IllegalArgumentException("Empty " + point.type() + " cannot be represented in WKB");
                }
                WellKnownBinary.writeInt(out, scratch, point.hasZ() ? 1001 : 1);
                WellKnownBinary.writeDouble(out, scratch, point.getX());
                WellKnownBinary.writeDouble(out, scratch, point.getY());
                if (point.hasZ()) {
                    WellKnownBinary.writeDouble(out, scratch, point.getZ());
                }
                return null;
            }

            @Override
            public Void visit(Line line) {
                WellKnownBinary.writeInt(out, scratch, line.hasZ() ? 1002 : 2);
                WellKnownBinary.writeInt(out, scratch, line.length());
                for (int i = 0; i < line.length(); ++i) {
                    WellKnownBinary.writeDouble(out, scratch, line.getX(i));
                    WellKnownBinary.writeDouble(out, scratch, line.getY(i));
                    if (!line.hasZ()) continue;
                    WellKnownBinary.writeDouble(out, scratch, line.getZ(i));
                }
                return null;
            }

            @Override
            public Void visit(Polygon polygon) {
                WellKnownBinary.writeInt(out, scratch, polygon.hasZ() ? 1003 : 3);
                if (polygon.isEmpty()) {
                    WellKnownBinary.writeInt(out, scratch, 0);
                    return null;
                }
                WellKnownBinary.writeInt(out, scratch, polygon.getNumberOfHoles() + 1);
                this.visitLinearRing(polygon.getPolygon());
                for (int i = 0; i < polygon.getNumberOfHoles(); ++i) {
                    this.visitLinearRing(polygon.getHole(i));
                }
                return null;
            }

            @Override
            public Void visit(MultiPoint multiPoint) {
                WellKnownBinary.writeInt(out, scratch, multiPoint.hasZ() ? 1004 : 4);
                WellKnownBinary.writeInt(out, scratch, multiPoint.size());
                for (Point point : multiPoint) {
                    WellKnownBinary.toWKB(point, out, scratch);
                }
                return null;
            }

            @Override
            public Void visit(MultiLine multiLine) {
                WellKnownBinary.writeInt(out, scratch, multiLine.hasZ() ? 1005 : 5);
                WellKnownBinary.writeInt(out, scratch, multiLine.size());
                for (Line line : multiLine) {
                    WellKnownBinary.toWKB(line, out, scratch);
                }
                return null;
            }

            @Override
            public Void visit(MultiPolygon multiPolygon) {
                WellKnownBinary.writeInt(out, scratch, multiPolygon.hasZ() ? 1006 : 6);
                WellKnownBinary.writeInt(out, scratch, multiPolygon.size());
                for (Polygon polygon : multiPolygon) {
                    WellKnownBinary.toWKB(polygon, out, scratch);
                }
                return null;
            }

            @Override
            public Void visit(GeometryCollection<?> collection) {
                WellKnownBinary.writeInt(out, scratch, collection.hasZ() ? 1007 : 7);
                WellKnownBinary.writeInt(out, scratch, collection.size());
                for (Geometry geometry : collection) {
                    WellKnownBinary.toWKB(geometry, out, scratch);
                }
                return null;
            }

            @Override
            public Void visit(Circle circle) {
                if (circle.isEmpty()) {
                    throw new IllegalArgumentException("Empty " + circle.type() + " cannot be represented in WKB");
                }
                WellKnownBinary.writeInt(out, scratch, circle.hasZ() ? 1017 : 17);
                WellKnownBinary.writeDouble(out, scratch, circle.getX());
                WellKnownBinary.writeDouble(out, scratch, circle.getY());
                if (circle.hasZ()) {
                    WellKnownBinary.writeDouble(out, scratch, circle.getZ());
                }
                WellKnownBinary.writeDouble(out, scratch, circle.getRadiusMeters());
                return null;
            }

            @Override
            public Void visit(Rectangle rectangle) {
                if (rectangle.isEmpty()) {
                    throw new IllegalArgumentException("Empty " + rectangle.type() + " cannot be represented in WKB");
                }
                WellKnownBinary.writeInt(out, scratch, rectangle.hasZ() ? 1018 : 18);
                WellKnownBinary.writeDouble(out, scratch, rectangle.getMinX());
                WellKnownBinary.writeDouble(out, scratch, rectangle.getMaxX());
                WellKnownBinary.writeDouble(out, scratch, rectangle.getMaxY());
                WellKnownBinary.writeDouble(out, scratch, rectangle.getMinY());
                if (rectangle.hasZ()) {
                    WellKnownBinary.writeDouble(out, scratch, rectangle.getMinZ());
                    WellKnownBinary.writeDouble(out, scratch, rectangle.getMaxZ());
                }
                return null;
            }

            @Override
            public Void visit(LinearRing ring) {
                throw new IllegalArgumentException("Linear ring is not supported by WKB");
            }

            private void visitLinearRing(LinearRing ring) {
                WellKnownBinary.writeInt(out, scratch, ring.length());
                for (int i = 0; i < ring.length(); ++i) {
                    WellKnownBinary.writeDouble(out, scratch, ring.getX(i));
                    WellKnownBinary.writeDouble(out, scratch, ring.getY(i));
                    if (!ring.hasZ()) continue;
                    WellKnownBinary.writeDouble(out, scratch, ring.getZ(i));
                }
            }
        });
    }

    private static void writeInt(ByteArrayOutputStream out, ByteBuffer scratch, int i) {
        scratch.clear();
        scratch.putInt(i);
        out.write(scratch.array(), 0, 4);
    }

    private static void writeDouble(ByteArrayOutputStream out, ByteBuffer scratch, double d) {
        scratch.clear();
        scratch.putDouble(d);
        out.write(scratch.array(), 0, 8);
    }

    public static Geometry fromWKB(GeometryValidator validator, boolean coerce, byte[] wkb) {
        return WellKnownBinary.fromWKB(validator, coerce, wkb, 0, wkb.length);
    }

    public static Geometry fromWKB(GeometryValidator validator, boolean coerce, byte[] wkb, int offset, int length) {
        ByteBuffer byteBuffer = ByteBuffer.wrap(wkb, offset, length);
        Geometry geometry = WellKnownBinary.parseGeometry(byteBuffer, coerce);
        validator.validate(geometry);
        return geometry;
    }

    private static Geometry parseGeometry(ByteBuffer byteBuffer, boolean coerce) {
        byteBuffer.order(byteBuffer.get() == 0 ? ByteOrder.BIG_ENDIAN : ByteOrder.LITTLE_ENDIAN);
        int type = byteBuffer.getInt();
        return switch (type) {
            case 1 -> WellKnownBinary.parsePoint(byteBuffer, false);
            case 1001 -> WellKnownBinary.parsePoint(byteBuffer, true);
            case 2 -> WellKnownBinary.parseLine(byteBuffer, false);
            case 1002 -> WellKnownBinary.parseLine(byteBuffer, true);
            case 3 -> WellKnownBinary.parsePolygon(byteBuffer, false, coerce);
            case 1003 -> WellKnownBinary.parsePolygon(byteBuffer, true, coerce);
            case 4, 1004 -> WellKnownBinary.parseMultiPoint(byteBuffer);
            case 5, 1005 -> WellKnownBinary.parseMultiLine(byteBuffer);
            case 6, 1006 -> WellKnownBinary.parseMultiPolygon(byteBuffer, coerce);
            case 7, 1007 -> WellKnownBinary.parseGeometryCollection(byteBuffer, coerce);
            case 17 -> WellKnownBinary.parseCircle(byteBuffer, false);
            case 1017 -> WellKnownBinary.parseCircle(byteBuffer, true);
            case 18 -> WellKnownBinary.parseBBox(byteBuffer, false);
            case 1018 -> WellKnownBinary.parseBBox(byteBuffer, true);
            default -> throw new IllegalArgumentException("Unknown geometry type: " + type);
        };
    }

    private static Point parsePoint(ByteBuffer byteBuffer, boolean hasZ) {
        if (hasZ) {
            return new Point(byteBuffer.getDouble(), byteBuffer.getDouble(), byteBuffer.getDouble());
        }
        return new Point(byteBuffer.getDouble(), byteBuffer.getDouble());
    }

    private static Line parseLine(ByteBuffer byteBuffer, boolean hasZ) {
        int length = byteBuffer.getInt();
        if (length == 0) {
            return Line.EMPTY;
        }
        double[] lats = new double[length];
        double[] lons = new double[length];
        double[] alts = hasZ ? new double[length] : null;
        for (int i = 0; i < length; ++i) {
            lons[i] = byteBuffer.getDouble();
            lats[i] = byteBuffer.getDouble();
            if (!hasZ) continue;
            alts[i] = byteBuffer.getDouble();
        }
        if (hasZ) {
            return new Line(lons, lats, alts);
        }
        return new Line(lons, lats);
    }

    private static Polygon parsePolygon(ByteBuffer byteBuffer, boolean hasZ, boolean coerce) {
        int rings = byteBuffer.getInt();
        if (rings == 0) {
            return Polygon.EMPTY;
        }
        LinearRing shell = WellKnownBinary.parseLinearRing(byteBuffer, hasZ, coerce);
        ArrayList<LinearRing> holes = new ArrayList<LinearRing>();
        for (int i = 1; i < rings; ++i) {
            holes.add(WellKnownBinary.parseLinearRing(byteBuffer, hasZ, coerce));
        }
        if (holes.isEmpty()) {
            return new Polygon(shell);
        }
        return new Polygon(shell, Collections.unmodifiableList(holes));
    }

    private static MultiPoint parseMultiPoint(ByteBuffer byteBuffer) {
        int numPoints = byteBuffer.getInt();
        if (numPoints == 0) {
            return MultiPoint.EMPTY;
        }
        ArrayList<Point> points = new ArrayList<Point>(numPoints);
        for (int i = 0; i < numPoints; ++i) {
            Geometry geometry = WellKnownBinary.parseGeometry(byteBuffer, false);
            if (!(geometry instanceof Point)) {
                throw new IllegalArgumentException("Expected a " + ShapeType.POINT + ", got [" + geometry.type() + "]");
            }
            Point p = (Point)geometry;
            points.add(p);
        }
        return new MultiPoint(Collections.unmodifiableList(points));
    }

    private static MultiLine parseMultiLine(ByteBuffer byteBuffer) {
        int numLines = byteBuffer.getInt();
        if (numLines == 0) {
            return MultiLine.EMPTY;
        }
        ArrayList<Line> lines = new ArrayList<Line>(numLines);
        for (int i = 0; i < numLines; ++i) {
            Geometry geometry = WellKnownBinary.parseGeometry(byteBuffer, false);
            if (!(geometry instanceof Line)) {
                throw new IllegalArgumentException("Expected a " + ShapeType.LINESTRING + ", got [" + geometry.type() + "]");
            }
            Line l = (Line)geometry;
            lines.add(l);
        }
        return new MultiLine(Collections.unmodifiableList(lines));
    }

    private static MultiPolygon parseMultiPolygon(ByteBuffer byteBuffer, boolean coerce) {
        int numPolygons = byteBuffer.getInt();
        if (numPolygons == 0) {
            return MultiPolygon.EMPTY;
        }
        ArrayList<Polygon> polygons = new ArrayList<Polygon>(numPolygons);
        for (int i = 0; i < numPolygons; ++i) {
            Geometry geometry = WellKnownBinary.parseGeometry(byteBuffer, coerce);
            if (!(geometry instanceof Polygon)) {
                throw new IllegalArgumentException("Expected a " + ShapeType.POLYGON + ", got [" + geometry.type() + "]");
            }
            Polygon p = (Polygon)geometry;
            polygons.add(p);
        }
        return new MultiPolygon(Collections.unmodifiableList(polygons));
    }

    private static GeometryCollection<Geometry> parseGeometryCollection(ByteBuffer byteBuffer, boolean coerce) {
        int numShapes = byteBuffer.getInt();
        if (numShapes == 0) {
            return GeometryCollection.EMPTY;
        }
        ArrayList<Geometry> shapes = new ArrayList<Geometry>(numShapes);
        for (int i = 0; i < numShapes; ++i) {
            shapes.add(WellKnownBinary.parseGeometry(byteBuffer, coerce));
        }
        return new GeometryCollection<Geometry>(shapes);
    }

    private static LinearRing parseLinearRing(ByteBuffer byteBuffer, boolean hasZ, boolean coerce) {
        int length = byteBuffer.getInt();
        if (length == 0) {
            return LinearRing.EMPTY;
        }
        double[] lons = new double[length];
        double[] lats = new double[length];
        double[] alts = hasZ ? new double[length] : null;
        for (int i = 0; i < length; ++i) {
            lons[i] = byteBuffer.getDouble();
            lats[i] = byteBuffer.getDouble();
            if (!hasZ) continue;
            alts[i] = byteBuffer.getDouble();
        }
        if (WellKnownBinary.linearRingNeedsCoerced(lats, lons, alts, coerce)) {
            lons = WellKnownBinary.coerce(lons);
            lats = WellKnownBinary.coerce(lats);
            if (hasZ) {
                alts = WellKnownBinary.coerce(alts);
            }
        }
        if (hasZ) {
            return new LinearRing(lons, lats, alts);
        }
        return new LinearRing(lons, lats);
    }

    private static boolean linearRingNeedsCoerced(double[] lons, double[] lats, double[] alts, boolean coerce) {
        assert (lats.length == lons.length && (alts == null || alts.length == lats.length));
        assert (lats.length > 0);
        if (!coerce) {
            return false;
        }
        int last = lons.length - 1;
        return lons[0] != lons[last] || lats[0] != lats[last] || alts != null && alts[0] != alts[last];
    }

    private static double[] coerce(double[] array) {
        double[] copy = new double[array.length + 1];
        System.arraycopy(array, 0, copy, 0, array.length);
        copy[array.length] = copy[0];
        return copy;
    }

    private static Rectangle parseBBox(ByteBuffer byteBuffer, boolean hasZ) {
        if (hasZ) {
            return new Rectangle(byteBuffer.getDouble(), byteBuffer.getDouble(), byteBuffer.getDouble(), byteBuffer.getDouble(), byteBuffer.getDouble(), byteBuffer.getDouble());
        }
        return new Rectangle(byteBuffer.getDouble(), byteBuffer.getDouble(), byteBuffer.getDouble(), byteBuffer.getDouble());
    }

    private static Circle parseCircle(ByteBuffer byteBuffer, boolean hasZ) {
        if (hasZ) {
            return new Circle(byteBuffer.getDouble(), byteBuffer.getDouble(), byteBuffer.getDouble(), byteBuffer.getDouble());
        }
        return new Circle(byteBuffer.getDouble(), byteBuffer.getDouble(), byteBuffer.getDouble());
    }
}

