package com.ohaotian.base.log;

import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.config.ConfigurationSource;
import org.apache.logging.log4j.core.config.Configurator;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * 日志配置组件：支持-DlogConfigFile=[日志位置]方式指定日志配置位置；日志配置位置格式：classpath:[位置],file:[位置],默认file:
 * @author liuzy
 */
public class LogContextLoaderListener implements ServletContextListener {
    private static final Logger logger = Logger.getLogger(LogContextLoaderListener.class.getName());
    private static final PathMatchingResourcePatternResolver PMRPR = new PathMatchingResourcePatternResolver(LogContextLoaderListener.class.getClassLoader());
    private String logConfigFile;
    private String defaultLogPrefix;
    private LoggerContext loggerContext;
    private List<String> configFileList = new LinkedList<String>();

    public LogContextLoaderListener() {
        String configFile = "log4j2.xml";
        initDefaultLoadConfigFile(configFile);

        configFile = "log4j.xml";
        initDefaultLoadConfigFile(configFile);

        configFile = "log4j.properties";
        initDefaultLoadConfigFile(configFile);
    }

    private void initDefaultLoadConfigFile(String configFile) {
        configFileList.add("classpath:/" + configFile);
        configFileList.add("classpath:/config/" + configFile);
        configFileList.add("classpath:/../config/" + configFile);
        configFileList.add("classpath:/../../config/" + configFile);
        configFileList.add("classpath:/../../../config/" + configFile);
        configFileList.add("classpath:/../../../../config/" + configFile);
        configFileList.add("classpath:/../../../../../config/" + configFile);
        configFileList.add("classpath:/../../../../../../config/" + configFile);
        configFileList.add("classpath:/../../../../../../../config/" + configFile);
    }

    public void contextInitialized(ServletContextEvent servletContextEvent) {
        ServletContext servletContext = servletContextEvent.getServletContext();
        defaultLogPrefix = servletContext == null ? null : servletContext.getInitParameter("defaultLogPrefix");
        if (defaultLogPrefix == null || "".equals(defaultLogPrefix)) {
            defaultLogPrefix = "file:";
        }
        logConfigFile = System.getProperty("logConfigFile");
        if (logConfigFile == null || "".equals(logConfigFile)) {
            logConfigFile = servletContext == null ? null :servletContext.getInitParameter("logConfigFile");
        }
        if (logConfigFile == null || "".equals(logConfigFile)) {
            throw new RuntimeException("log config is required !!!");
        }
        if (!logConfigFile.startsWith("file:") && !logConfigFile.startsWith("classpath:")) {
            logConfigFile = defaultLogPrefix + logConfigFile;
        }
        configFileList.add(0, logConfigFile);
        for (String configFile : configFileList) {
            if (doLogConfig(configFile)) {
                this.logConfigFile = configFile;
                break;
            } else {
                logger.info("load log config[" + configFile + "] failed!");
            }
        }
    }

    private boolean doLogConfig(String configFile) {
        Resource logResource = extractLogConfigFileResource(configFile);
        if (!logResource.exists()) {
            return false;
        }
        boolean configLoaded = false;
        InputStream logInput = null;
        try {
            logInput = logResource.getInputStream();
            ConfigurationSource source = new ConfigurationSource(logInput);
            loggerContext = Configurator.initialize(this.getClass().getClassLoader(), source);
            configLoaded = true;
        } catch (IOException e) {
            logger.log(Level.SEVERE, "load log config [" + configFile + "] error:" + e.getMessage());
        } finally {
            if (logInput != null) {
                try {
                    logInput.close();
                } catch (IOException e) {
                    logger.log(Level.SEVERE, "close log config [" + configFile + "] stream error:" + e.getMessage());
                }
            }
        }
        return configLoaded;
    }

    private Resource extractLogConfigFileResource(String configFile) {
        String classpathPro = "classpath:";
        if (configFile.startsWith(classpathPro)) {
            try {
                String formatedFilePath = configFile.replace("/", File.separator).substring(classpathPro.length());
                File tmpFile = new File(PMRPR.getResource(classpathPro + "./").getFile().getCanonicalPath() + File.separator + formatedFilePath);
                formatedFilePath = "file:" + tmpFile.getCanonicalPath();
                logger.info("trans path[" + configFile + "] to path[" + formatedFilePath + "].");
                configFile = formatedFilePath;
            } catch (IOException e) {
                logger.log(Level.SEVERE, "trans path[" + configFile + "] to canonicalPath error:" + e.getMessage());
            }
        }
        logger.info("load log config. logConfigFile:[" + configFile + "]");
        return PMRPR.getResource(configFile);
    }

    public void contextDestroyed(ServletContextEvent servletContextEvent) {
        if (loggerContext != null) {
            logger.info("unload log config. logConfigFile:[" + logConfigFile + "]");
            Configurator.shutdown(loggerContext);
        }
    }
}
