package com.ohaotian.plugin.cache;

import java.util.Collections;
import java.util.List;
import java.util.Set;


import com.ohaotian.plugin.util.ConvertUtil;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisCluster;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;
import redis.clients.jedis.params.SetParams;

/** redis管理类 支持注入连接池 封装基础的增删改查功能
 *
 * @author liuce
 * @time 2016年4月13日13:57:10 */
public class RedisManager implements CacheManager {

	/** 主机名称或者ip地址 */
	private String    host      = "115.28.105.99";

	/** 端口 */
	private int       port      = 6379;

	/** 失效时间 0为永不失效 */
	private int       expire    = 0;

	/** 连接超时时间 会自动重连 */
	private int       timeout   = 0;

	/** 密码 */
	private String    password  = "BJtGXdevRedis123";

	/** 连接池 */
	private JedisPool jedisPool = null;

	private static final String LOCK_SUCCESS = "OK";
	private static final Long RELEASE_SUCCESS = 1L;

	public RedisManager() {

	}

	public RedisManager(String host, int port, String password) {
		this.host = host;
		this.port = port;
		this.password = password;
	}

	public RedisManager(String host, int port, String password, int expire, int timeout) {
		this.host = host;
		this.port = port;
		this.password = password;
		this.expire = expire;
		this.timeout = timeout;
	}

	/** 初始化方法 */
	public void init() {
		if (jedisPool == null) {
			if (password != null && !"".equals(password)) {
				jedisPool = new JedisPool(new JedisPoolConfig(), host, port, timeout, password);
			}
			else if (timeout != 0) {
				jedisPool = new JedisPool(new JedisPoolConfig(), host, port, timeout);
			}
			else {
				jedisPool = new JedisPool(new JedisPoolConfig(), host, port);
			}

		}
	}

	/** get value from redis
	 *
	 * @param key
	 * @return */
	@Override
	public byte[] get(byte[] key) {
		byte[] value = null;
		Jedis jedis = null;
		try {
			jedis = jedisPool.getResource();
			value = jedis.get(key);
		}
		finally {
			if (jedis != null) {
				jedis.close();
			}
		}
		return value;
	}

	/** set
	 *
	 * @param key
	 * @param value
	 * @return */
	@Override
	public byte[] set(byte[] key, byte[] value) {
		Jedis jedis = null;
		try {
			jedis = jedisPool.getResource();
			jedis.set(key, value);
			if (this.expire != 0) {
				jedis.expire(key, this.expire);
			}
		}
		finally {
			if (jedis != null) {
				jedis.close();
			}
		}
		return value;
	}

	/** set
	 *
	 * @param key
	 * @param value
	 * @param expire
	 * @return */
	@Override
	public byte[] set(byte[] key, byte[] value, int expire) {
		Jedis jedis = null;
		try {
			jedis = jedisPool.getResource();
			jedis.set(key, value);
			if (expire != 0) {
				jedis.expire(key, expire);
			}
		}
		finally {
			if (jedis != null) {
				jedis.close();
			}
		}
		return value;
	}

	/** del
	 *
	 * @param key */
	@Override
	public void del(byte[] key) {
		Jedis jedis = null;
		try {
			jedis = jedisPool.getResource();
			jedis.del(key);
		}
		finally {
			if (jedis != null) {
				jedis.close();
			}
		}
	}

	/** flush */
	@Override
	public void flushDB() {
		Jedis jedis = null;
		try {
			jedis = jedisPool.getResource();
			jedis.flushDB();
		}
		finally {
			if (jedis != null) {
				jedis.close();
			}
		}
	}

	/** size */
	@Override
	public Long dbSize() {
		Long dbSize = 0L;
		Jedis jedis = null;
		try {
			jedis = jedisPool.getResource();
			dbSize = jedis.dbSize();
		}
		finally {
			if (jedis != null) {
				jedis.close();
			}
		}
		return dbSize;
	}

	/** keys
	 *
	 * @param pattern
	 * @return */
	@Override
	public Set<byte[]> keys(String pattern) {
		Set<byte[]> keys = null;
		Jedis jedis = null;
		try {
			jedis = jedisPool.getResource();
			keys = jedis.keys(pattern.getBytes());
		}
		finally {
			if (jedis != null) {
				jedis.close();
			}
		}
		return keys;
	}

	public String getHost() {
		return host;
	}

	public void setHost(String host) {
		this.host = host;
	}

	public int getPort() {
		return port;
	}

	public void setPort(int port) {
		this.port = port;
	}

	public int getExpire() {
		return expire;
	}

	public void setExpire(int expire) {
		this.expire = expire;
	}

	public int getTimeout() {
		return timeout;
	}

	public void setTimeout(int timeout) {
		this.timeout = timeout;
	}

	public String getPassword() {
		return password;
	}

	public void setPassword(String password) {
		this.password = password;
	}

	public JedisPool getJedisPool() {
		return jedisPool;
	}

	public void setJedisPool(JedisPool jedisPool) {
		this.jedisPool = jedisPool;
	}
	@Override
	public JedisCluster getJedisCluster() {
		return null;
	}

	@Override
	public void setValueExpireTime(byte[] key, int expire) {
		Jedis jedis = null;
		try {
			jedis = jedisPool.getResource();
			if (expire != 0) {
				jedis.expire(key, expire);
			}
		}
		finally {
			if (jedis != null) {
				jedis.close();
			}
		}
	}

	@Override
	public Long getExpireTimeByKey(byte[] key) {
		Long expireTime = -1L;
		Jedis jedis = null;
		try {
			jedis = jedisPool.getResource();
			expireTime = jedis.ttl(key);
		}
		finally {
			if (jedis != null) {
				jedis.close();
			}
		}
		return expireTime;
	}

	@Override
	public Long incr(byte[] key) {
		Jedis jedis = null;
		Long num = null;
		try {
			jedis = jedisPool.getResource();
			num = jedis.incr(key);
		}
		finally {
			if (jedis != null) {
				jedis.close();
			}
		}
		return num;
	}

	@Override
	public Long incrExpireTime(byte[] key, int expire) {
		Jedis jedis = null;
		Long num = null;
		try {
			jedis = jedisPool.getResource();
			num = jedis.incr(key);
			jedis.expire(key, expire);
		}
		finally {
			if (jedis != null) {
				jedis.close();
			}
		}
		return num;
	}

	@Override
	public Long incrBy(byte[] key, long conut) {
		Jedis jedis = null;
		Long num = null;
		try {
			jedis = jedisPool.getResource();
			num = jedis.incrBy(key, conut);
		}
		finally {
			if (jedis != null) {
				jedis.close();
			}
		}
		return num;
	}

	@Override
	public Long incrExpireTimeBy(byte[] key, long conut, int expire) {
		Jedis jedis = null;
		Long num = null;
		try {
			jedis = jedisPool.getResource();
			num = jedis.incrBy(key, conut);
			jedis.expire(key, expire);
		}
		finally {
			if (jedis != null) {
				jedis.close();
			}
		}
		return num;
	}

	@Override
	public Long getIncr(byte[] key) {
		Jedis jedis = null;
		Long num = null;
		try {
			jedis = jedisPool.getResource();
			byte[] bnum = jedis.get(key);
			String numStr = bnum == null ? null : new String(bnum);
			num = numStr == null ? null : Long.parseLong(numStr);
		}
		finally {
			if (jedis != null) {
				jedis.close();
			}
		}
		return num;
	}


	//队列操作 begin
    /**
     * <p>通过key向list尾部添加字符串</p>
     * @param key
     * @param value 可以使一个string 也可以使string数组
     * @return 返回list的value个数
     */
	@Override
    public Long rpush(byte[] key ,byte[] value){
        Jedis jedis = null;
        Long res = null;
        try {
            jedis = jedisPool.getResource();
            res = jedis.rpush(key, value);
        } finally {
        	if (jedis != null) {
				jedis.close();
			}
        }
        return res;
    }


    /**
     * <p>通过key从list的头部删除一个value,并返回该value</p>
     * @param key
     * @return
     */
	@Override
    public String lpop(String  key){
        Jedis jedis = null;
        String res = null;
        try {
            jedis = jedisPool.getResource();
            res = jedis.lpop(key);
        } finally {
        	if (jedis != null) {
				jedis.close();
			}
        }
        return res;
    }

    /**
     * <p>通过key返回list的长度</p>
     * @param key
     * @return
     */
	@Override
    public Long llen(String key){
        Jedis jedis = null;
        Long res = null;
        try {
            jedis = jedisPool.getResource();
            res = jedis.llen(key);
        } finally {
        	if (jedis != null) {
				jedis.close();
			}
        }
        return res;
    }
	//队列操作 end

	//分布式锁 begin
	/**
     * 将 key 的值设为 value ，当且仅当 key 不存在。
     * 若给定的 key 已经存在，则 SETNX 不做任何动作。
     * SETNX 是『SET if Not eXists』(如果不存在，则 SET)的简写
     * @param key
     * @return   返回值：设置成功，返回 1 。设置失败，返回 0 。
     * */
	@Override
    public Long setnx(byte[] key,byte[] value){
        Jedis jedis = null;
        Long res = null;
        try {
            jedis = jedisPool.getResource();
            res = jedis.setnx(key, value);
        } finally {
        	if (jedis != null) {
				jedis.close();
			}
        }
        return res;
    }

	/**
     * 语法：
     * GETSET key value
     * 功能：
     * 将给定 key 的值设为 value ，并返回 key 的旧值 (old value)，当 key 存在但不是字符串类型时，返回一个错误，当key不存在时，返回nil。
     * */
	@Override
    public byte[] getSet(byte[] key,byte[] value){
        Jedis jedis = null;
        byte[] res = null;
        try {
            jedis = jedisPool.getResource();
            res = jedis.getSet(key, value);
        } finally {
        	if (jedis != null) {
				jedis.close();
			}
        }
        return res;
    }

	@Override
	public boolean tryLock(String lockKey,String lockValue,int expire) {
		Jedis jedis = null;
		try {
			jedis = jedisPool.getResource();
			String result = jedis.set(lockKey,lockValue, SetParams.setParams().nx().px(expire));
			return LOCK_SUCCESS.equals(result);
		} finally {
			if (jedis != null) {
				jedis.close();
			}
		}
	}

	@Override
	public boolean tryUnLock(String lockKey, String lockId) {
		Jedis jedis = null;
		try {
			jedis = jedisPool.getResource();
			String script = "if redis.call('get', KEYS[1]) == ARGV[1] then return redis.call('del', KEYS[1]) else return 0 end";
			Object result = jedis.eval(script, Collections.singletonList(lockKey), Collections.singletonList(lockId));

			if (RELEASE_SUCCESS.equals(result)) {
				return true;
			}
			return false;
		} finally {
			if (jedis != null) {
				jedis.close();
			}
		}
	}
	//分布式锁 end

	/**
	 * 获取数据
	 *
	 * @param key
	 * @return
	 */
	public <T> T blpop(String key, int waitSeconds, Class<T> clazz)
	{
		Jedis jedis = null;

		try
		{
			jedis = jedisPool.getResource();
			List<byte[]> values = jedis.brpop(waitSeconds, key.getBytes());

			if (values != null && values.size() > 0)
			{
				byte[] value = values.get(1);
				return ConvertUtil.unserialize(value, clazz);
			}
			else
			{
				return null;
			}
		}
		catch (Exception e)
		{
			return null;
		}
		finally
		{
			if (jedis != null) {
				jedis.close();
			}
		}
	}

	@Override
	public Long expire(String key, int secound) {
		Jedis jedis = null;
		Long expire = null;
		try {
			jedis = jedisPool.getResource();
			expire = jedis.expire(key, secound);
		} finally {
			if (jedis != null) {
				jedis.close();
			}
		}
		return expire;
	}

}
