package com.ohaotian.plugin.db.impl;

import com.ohaotian.plugin.db.OrderSequence;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

public class OrderSequenceImpl implements OrderSequence {

    private String sequenceName;
    private DataSource dataSource;

    private static final String QUERY_CURRENT_VALUE = "SELECT current_value, step FROM ohaotian_order_sequence WHERE name = ? FOR UPDATE";
    private static final String UPDATE_CURRENT_VALUE = "UPDATE ohaotian_order_sequence SET current_value = current_value + step WHERE name = ?";
    private static final String UPDATE_BATCH_CURRENT_VALUE = "UPDATE ohaotian_order_sequence SET current_value = current_value + (step * ?) WHERE name = ?";

    public OrderSequenceImpl(String sequenceName, DataSource dataSource) {
        this.sequenceName = sequenceName;
        this.dataSource = dataSource;
    }

    @Override
    public long nextId() throws SQLException {
        long id;
        try (Connection conn = dataSource.getConnection();
             PreparedStatement psUpdate = createUpdatePreparedStatement(conn);
             PreparedStatement psQuery = createPreparedStatement(conn)) {
            conn.setAutoCommit(false);
            psUpdate.execute();
            try (ResultSet rs = psQuery.executeQuery()) {
                if (rs.next()) {
                    id = rs.getLong("current_value");
                    conn.commit();
                } else {
                    throw new IllegalArgumentException("Can't find " + sequenceName + " from table ohaotian_order_sequence, please create sequence.");
                }
            }
        }
        return id;
    }

    private PreparedStatement createUpdatePreparedStatement(Connection conn) throws SQLException {
        PreparedStatement ps = conn.prepareStatement(UPDATE_CURRENT_VALUE);
        ps.setString(1, sequenceName);
        return ps;
    }

    private PreparedStatement createPreparedStatement(Connection conn) throws SQLException {
        PreparedStatement ps = conn.prepareStatement(QUERY_CURRENT_VALUE);
        ps.setString(1, sequenceName);
        return ps;
    }

    @Override
    public long[] nextIds(int idNum) throws SQLException {
        long[] ids;
        try (Connection conn = dataSource.getConnection();
             PreparedStatement psUpdate = createUpdateBatchPreparedStatement(conn, idNum);
             PreparedStatement psQuery = createPreparedStatement(conn)) {
            conn.setAutoCommit(false);
            psUpdate.execute();
            try (ResultSet rs = psQuery.executeQuery()) {
                if (rs.next()) {
                    long id = rs.getLong("current_value");
                    int step = rs.getInt("step");
                    ids = batchIds(idNum, id, step);
                    conn.commit();
                } else {
                    throw new IllegalArgumentException("Can't find " + sequenceName + " from table ohaotian_order_sequence, please create sequence.");
                }
            }
        }
        return ids;
    }

    private static long[] batchIds(int idNum, long id, int step) {
        long[] ids = new long[idNum];
        for (int i = 0; i < idNum; i++) {
            ids[i] = id - (step * i);
        }
        return ids;
    }

    private PreparedStatement createUpdateBatchPreparedStatement(Connection conn, int idNum) throws SQLException {
        PreparedStatement ps = conn.prepareStatement(UPDATE_BATCH_CURRENT_VALUE);
        ps.setInt(1, idNum);
        ps.setString(2, sequenceName);
        return ps;
    }
}