package com.tydic.dyc.oc.service.order;

import cn.hutool.core.convert.Convert;
import com.tydic.dyc.base.bo.BasePageRspBo;
import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.dyc.oc.constants.UocConstant;
import com.tydic.dyc.oc.constants.UocDicConstant;
import com.tydic.dyc.oc.constants.UocRspConstants;
import com.tydic.dyc.oc.model.order.IUocOrderModel;
import com.tydic.dyc.oc.model.order.qrybo.UocConfEffectiveQryBo;
import com.tydic.dyc.oc.model.order.sub.UocConfEffective;
import com.tydic.dyc.oc.model.sysdictionary.IUocSysDictionaryModel;
import com.tydic.dyc.oc.service.order.bo.UocProOrderEffectiveDealReqBo;
import com.tydic.dyc.oc.service.order.bo.UocProOrderEffectiveDealRspBo;
import com.tydic.dyc.oc.utils.IdUtil;
import com.tydic.dyc.oc.utils.UocRu;
import lombok.extern.ohaotian.HTServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * 标题：UocProOrderEffectiveAddAndModifyAbilityServiceImpl
 * 说明：订单时效类配置处理API 实现类
 * 时间：2021/4/2 10:07
 * 作者 何秀全
 */
@HTServiceImpl
public class UocProOrderEffectiveDealServiceImpl implements UocProOrderEffectiveDealService {

    @Autowired
    private IUocOrderModel iUocOrderModel;

    @Autowired
    private IUocSysDictionaryModel iUocSysDictionaryModel;

    @Override
    public UocProOrderEffectiveDealRspBo dealOrderEffective(UocProOrderEffectiveDealReqBo reqBo) {
        UocProOrderEffectiveDealRspBo rspBo = UocRu.success(UocProOrderEffectiveDealRspBo.class);
        validateArgs(reqBo);
        if (null != reqBo.getIsDelete() && reqBo.getIsDelete()) {
            deleteOrderEffective(reqBo);
            return rspBo;
        }
        if (null == (reqBo.getEffectiveId()) || 0L == reqBo.getEffectiveId()) {
            addOrderEffective(reqBo);
        } else {
            modifyOrderEffective(reqBo);
        }
        return null;
    }

    /**
     * 删除时效配置信息
     *
     * @param reqBo 入参
     */
    private void deleteOrderEffective(UocProOrderEffectiveDealReqBo reqBo) {
        UocConfEffective uocConfEffective = new UocConfEffective();
        uocConfEffective.setEffectiveId(reqBo.getEffectiveId());
        UocConfEffective modelBy = iUocOrderModel.getConfEffectiveBy(uocConfEffective);
        if (null == modelBy) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ORD_EFFECT_DEAL_ABILITY_ERROR, "入参[effectiveId]查询为空，无法删除");
        } else if (UocConstant.SourceCode.SYSTEM.equals(modelBy.getSourceCode())) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ORD_EFFECT_DEAL_ABILITY_ERROR, "该数据为系统预设值，不允许删除");
        }
        iUocOrderModel.deleteConfEffective(uocConfEffective);
    }

    /**
     * 修改时效配置信息
     *
     * @param reqBo 入参
     */
    private void modifyOrderEffective(UocProOrderEffectiveDealReqBo reqBo) {
        if (UocConstant.EffectiveCode.ORDER_CONFIRMATION_PAYMENT_TIME_LIMIT.equals(reqBo.getBusinessTypeCode())) {
            //订单确认付款时限， 电商采购 判断设置时限在 “电商预占库存时限”中最小时间范围内进行设置
            if (UocDicConstant.ORDER_SOURCE.E_COMMERCE_IMPORT.equals(Convert.toInt(reqBo.getBusinessTypeCode()))) {
                UocConfEffectiveQryBo uocConfEffective = new UocConfEffectiveQryBo();
                uocConfEffective.setEffectiveCode(UocConstant.EffectiveCode.E_COMMERCE_INVENTORY_TIME_LIMIT);
                uocConfEffective.setBusinessTypeCode(Convert.toStr(UocDicConstant.ORDER_SOURCE.E_COMMERCE_IMPORT));
                uocConfEffective.setOrderBy("aging");
                uocConfEffective.setPageNo(1);
                uocConfEffective.setPageSize(1);
                BasePageRspBo<UocConfEffective> uocConfEffectiveBasePageRspBo = iUocOrderModel.qryListPageConfEffective(uocConfEffective);
                if (CollectionUtils.isEmpty(uocConfEffectiveBasePageRspBo.getRows())) {
                    throw new BaseBusinessException(UocRspConstants.RSP_CODE_ORD_EFFECT_DEAL_ABILITY_ERROR, "由于电商预占库存时限没有设置，无法配置 电商采购 的 订单确认付款时限");
                } else {
                    if (0 < uocConfEffectiveBasePageRspBo.getRows().get(0).getAging()) {
                        if (reqBo.getAging() > uocConfEffectiveBasePageRspBo.getRows().get(0).getAging()) {
                            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ORD_EFFECT_DEAL_ABILITY_ERROR, "可设置的时限范围在" + formatDuring(uocConfEffectiveBasePageRspBo.getRows().get(0).getAging()) + "之内；");
                        }
                    }
                }
            }
        }
        UocConfEffective effective = new UocConfEffective();
        effective.setAging(reqBo.getAging());
        effective.setOperatingTime(new Date());
        effective.setOperatorId(String.valueOf(reqBo.getUserId()));
        effective.setOperatorName(reqBo.getUsername());
        effective.setEffectiveId(reqBo.getEffectiveId());
        int updateBy = iUocOrderModel.updateEffectiveConfById(effective);
        if (updateBy != 1) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ORD_EFFECT_DEAL_ABILITY_ERROR, "入参[effectiveId]不对，导致修改内容不对");
        }
    }

    /**
     * 增加时效配置
     *
     * @param reqBo 入参
     */
    private void addOrderEffective(UocProOrderEffectiveDealReqBo reqBo) {
        String effectiveName = checkCodeCorrect(reqBo);
        UocConfEffectiveQryBo uocConfEffectiveQryBo = new UocConfEffectiveQryBo();
        UocConfEffective uocConfEffective = new UocConfEffective();
        uocConfEffective.setEffectiveCode(reqBo.getEffectiveCode());
        uocConfEffectiveQryBo.setEffectiveCode(reqBo.getEffectiveCode());
        if (!UocConstant.EffectiveCode.ORDER_CONFIRMATION_PAYMENT_TIME_LIMIT.equals(reqBo.getEffectiveCode())) {
            if (UocConstant.EffectiveCode.E_COMMERCE_INVENTORY_TIME_LIMIT.equals(reqBo.getEffectiveCode())) {
                //电商预占库存时限 非 电商采购 不能增加配置
                if (!UocDicConstant.ORDER_SOURCE.E_COMMERCE_IMPORT.equals(Convert.toInt(reqBo.getBusinessTypeCode()))) {
                    throw new BaseBusinessException(UocRspConstants.RSP_CODE_ORD_EFFECT_DEAL_ABILITY_ERROR, "入参[businessTypeCode]不允许增加时效配置");
                }
            }
            uocConfEffective.setSupplierNo(reqBo.getSupplierNo());
            uocConfEffectiveQryBo.setSupplierNo(reqBo.getSupplierNo());
            int checkBy = iUocOrderModel.getCheckConfEffectiveBy(uocConfEffectiveQryBo);
            if (checkBy > 0) {
                throw new BaseBusinessException(UocRspConstants.RSP_CODE_ORD_EFFECT_DEAL_ABILITY_ERROR, "已存在该电商的" + effectiveName + "，请重新设置。");
            }
            uocConfEffective.setSupplierName(reqBo.getSupplierName());
        } else {
            //订单确认付款时限 不允许增加 配置
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ORD_EFFECT_DEAL_ABILITY_ERROR, "入参[effectiveCode]不允许增加时效配置");
        }

        uocConfEffective.setEffectiveId(IdUtil.nextId());
        uocConfEffective.setBusinessTypeCode(reqBo.getBusinessTypeCode());
        uocConfEffective.setAging(reqBo.getAging());
        uocConfEffective.setSourceCode(reqBo.getSourceCode());
        uocConfEffective.setOperatingTime(new Date());
        uocConfEffective.setOperatorId(String.valueOf(reqBo.getUserId()));
        uocConfEffective.setOperatorName(reqBo.getUsername());
        iUocOrderModel.addConfEffective(uocConfEffective);
    }

    /**
     * 校验编码是否正确（合法性）
     *
     * @param reqBo 入参
     * @return 配置名称
     */
    private String checkCodeCorrect(UocProOrderEffectiveDealReqBo reqBo) {
        String effectiveName;
        List<String> pCodes = new ArrayList<>();
        pCodes.add(UocConstant.EFFECTIVE_CODE);
        pCodes.add(UocDicConstant.ORDER_SOURCE.P_CODE);
        pCodes.add(UocConstant.SOURCE_CODE);
        Map<String, Map<String, String>> dictionaryMap = iUocSysDictionaryModel.getDictionaryMap(pCodes);

        Map<String, String> orderSource = dictionaryMap.get(UocDicConstant.ORDER_SOURCE.P_CODE);
        if (!orderSource.containsKey(reqBo.getBusinessTypeCode())) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参[businessTypeCode]不合法");
        }

        Map<String, String> sourceCode = dictionaryMap.get(UocConstant.SOURCE_CODE);
        if (!sourceCode.containsKey(reqBo.getSourceCode())) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参[sourceCode]不合法");
        }

        Map<String, String> effectiveCode = dictionaryMap.get(UocConstant.EFFECTIVE_CODE);
        if (!effectiveCode.containsKey(reqBo.getEffectiveCode())) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参[effectiveCode]不合法");
        } else {
            effectiveName = effectiveCode.get(reqBo.getEffectiveCode());
        }
        return effectiveName;
    }

    /**
     * 毫秒数 转 天 小时 分
     *
     * @param aging 时效
     * @return 天 小时 分
     */
    private String formatDuring(long aging) {
        long days = aging / (1000 * 60 * 60 * 24);
        long hours = (aging % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60);
        long minutes = (aging % (1000 * 60 * 60)) / (1000 * 60);
        return days + "天" + hours + "小时" + minutes + "分";
    }
    
    private void validateArgs(UocProOrderEffectiveDealReqBo reqBo) {
        if (null == reqBo) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象不能为空");
        }
        if (null != reqBo.getIsDelete() && reqBo.getIsDelete()) {
            if (null == (reqBo.getEffectiveId()) || 0L == reqBo.getEffectiveId()) {
                throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参删除[effectiveId]不能为空");
            }
            return;
        }
        if (null == (reqBo.getEffectiveId()) || 0L == reqBo.getEffectiveId()) {
            //新增
            if (!StringUtils.hasText(reqBo.getEffectiveCode())) {
                throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参新增[effectiveCode]不能为空");
            }
            if (!StringUtils.hasText(reqBo.getBusinessTypeCode())) {
                throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参新增[businessTypeCode]不能为空");
            }
            if (!StringUtils.hasText(reqBo.getSourceCode())) {
                throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参新增[sourceCode]不能为空");
            }
            if (!UocConstant.EffectiveCode.ORDER_CONFIRMATION_PAYMENT_TIME_LIMIT.equals(reqBo.getEffectiveCode())) {
                if (!StringUtils.hasText(reqBo.getSupplierNo())) {
                    throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参新增[supplierNo]不能为空");
                }
                if (!StringUtils.hasText(reqBo.getSupplierName())) {
                    throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参新增[supplierName]不能为空");
                }
            }
        }
        if (null == (reqBo.getAging()) || 0L > reqBo.getAging()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参[aging]不能为空，且大于等于0");
        }
    }
}
