package com.tydic.dyc.oc.components.es;


import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.tydic.dyc.base.constants.BaseRspConstant;
import com.tydic.dyc.base.exception.BaseBusinessException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.util.EntityUtils;
import org.elasticsearch.action.admin.indices.create.CreateIndexRequest;
import org.elasticsearch.action.admin.indices.create.CreateIndexResponse;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.client.*;
import org.elasticsearch.common.xcontent.XContentType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.IOException;


/**
 * 标题：es获取客户端工具类
 * <p>
 * 说明：es获取客户端工具类
 * <br>
 * 时间：2019/1/1616:52<br>
 *
 *
 * </p>
 *
 * @author hegy
 */
@Component
@Slf4j
public class UocElasticsearchUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(UocElasticsearchUtil.class);

    private static ObjectMapper mapper = new ObjectMapper();

    @Autowired
    private RestHighLevelClient restHighLevelClient;

//    private UocElasticsearchUtil() {
//        this.restHighLevelClient = new RestHighLevelClient(RestClient.builder(new HttpHost("39.96.39.0", 9200)));
//    }

    /**
     * 删除索引
     *
     * @param index 索引
     * @return boolean
     */
    public boolean deleteIndex(String index) {
        if (StringUtils.isBlank(index)) {
            LOGGER.info("删除索引内容为空");
            return false;
        }
        DeleteIndexRequest request = new DeleteIndexRequest(index);
        try {
//            DeleteIndexResponse indexResponse = restHighLevelClient.indices().delete(request, RequestOptions.DEFAULT); // 6.x jar包
            AcknowledgedResponse indexResponse = restHighLevelClient.indices().delete(request, RequestOptions.DEFAULT);// 升级7.10 jar包
            if (indexResponse.isAcknowledged()) {
                LOGGER.info("删除索引成功");
            } else {
                LOGGER.info("删除索引失败");
            }
            return indexResponse.isAcknowledged();
        } catch (Exception e) {
            LOGGER.error("删除索引失败", e);
            String exceptionStr = JSON.toJSONString(e.getMessage());
            if (exceptionStr.contains("index_not_found_exception")) {
                return true;
            }
        }
        return false;
    }


    /**
     * 创建索引
     *
     * @param index 索引
     * @return boolean
     */
    public boolean createIndex(String index) {
        //index名必须全小写，否则报错
        CreateIndexRequest request = new CreateIndexRequest(index);
        try {
            CreateIndexResponse indexResponse = restHighLevelClient.indices().create(request, RequestOptions.DEFAULT);
            if (indexResponse.isAcknowledged()) {
                LOGGER.info("创建索引成功");
            } else {
                LOGGER.info("创建索引失败");
            }
            return indexResponse.isAcknowledged();
        } catch (IOException e) {
            LOGGER.error("创建索引失败", e);
        }
        return false;
    }

    /**
     * 插入数据
     *
     * @param index  索引
     * @param type   类型
     * @param object 数据
     * @return String
     */
    public String addData(String index, String type, String id, JSONObject object) {
//        IndexRequest indexRequest = new IndexRequest(index, type, id);// 6.x jar包
        IndexRequest indexRequest = new IndexRequest(index);
        indexRequest.id(id);
        try {
//            indexRequest.source(mapper.writeValueAsString(object), XContentType.JSON);
            indexRequest.source(JSON.toJSONString(object), XContentType.JSON);
            IndexResponse indexResponse = restHighLevelClient.index(indexRequest, RequestOptions.DEFAULT);
            return indexResponse.getId();
        } catch (Exception e) {
            LOGGER.info("插入数据失败", e);
        }
        return null;
    }

    /**
     * 描述:保存json数据
     *
     * @param index   索引名称
     * @param id      索引数据id
     * @param jsonStr json数据
     * @return java.lang.String
     * @author tgy
     * @date 2022/3/1 13:00
     */
    public String addJsonData(String index, String id, String jsonStr) {
        IndexRequest indexRequest = new IndexRequest(index);
        indexRequest.id(id);
        try {
            indexRequest.source(jsonStr, XContentType.JSON);
            log.info("addJsonData保存数据：" + jsonStr);
            IndexResponse indexResponse = restHighLevelClient.index(indexRequest, RequestOptions.DEFAULT);
            return indexResponse.getId();
        } catch (Exception e) {
            LOGGER.error("插入数据失败", e);
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "插入数据失败：" + e.getMessage(), e);
        }
    }


    /**
     * 检查索引
     *
     * @param index 索引
     * @return boolean
     */
    public boolean checkIndexExist(String index) {
        String ok = "OK";
        boolean checkResult = false;
        try {
            Request request = new Request("HEAD", index);
            Response response = restHighLevelClient.getLowLevelClient().performRequest(request);
            if (ok.equals(response.getStatusLine().getReasonPhrase())) {
                checkResult = true;
            }
        } catch (IOException e) {
            LOGGER.error("检查索引失败", e);
        }
        return checkResult;
    }

    /**
     * 根据条件删除ES数据
     *
     * @param endPoint 接口index点
     * @param qryJson  删除数据语句
     * @return boolean
     */
    public Boolean deleteDataByCondition(String endPoint, JSONObject qryJson) {
        boolean deleteResult = false;
        HttpEntity entity = new NStringEntity(qryJson.toJSONString(), ContentType.APPLICATION_JSON);
        try {
            Request request = new Request("POST", endPoint);
            request.setEntity(entity);
            Response response = restHighLevelClient.getLowLevelClient().performRequest(request);
            String result = EntityUtils.toString(response.getEntity());
            if (StringUtils.isNotBlank(result)) {
                JSONObject deleteResultJson = JSONObject.parseObject(result);
                int deleteCount = deleteResultJson.getInteger("deleted");
                if (deleteCount > 0) {
                    deleteResult = true;
                }
            }
        } catch (IOException e) {
            LOGGER.error("根据条件删除ES数据失败", e);
        }
        return deleteResult;
    }

    /**
     * 获取低水平客户端
     *
     * @return RestClient
     */
    public RestClient getLowLevelClient() {
        return restHighLevelClient.getLowLevelClient();

    }
}
