package com.tydic.dyc.pro.base.core.encode.service.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.ohaotian.plugin.base.exception.ZTBusinessException;
import com.ohaotian.plugin.cache.CacheClient;
import com.tydic.dyc.pro.base.core.constant.DycProCoreConstant;
import com.tydic.dyc.pro.base.core.constant.DycProCoreRspConstant;
import com.tydic.dyc.pro.base.core.dao.SysSerialInstanceMapper;
import com.tydic.dyc.pro.base.core.encode.service.api.DycProEncodeSerialRuleGetService;
import com.tydic.dyc.pro.base.core.encode.service.api.DycProEncodeSerialService;
import com.tydic.dyc.pro.base.core.encode.service.bo.*;
import com.tydic.dyc.pro.base.core.po.SysSerialInstancePO;
import lombok.extern.ohaotian.HTServiceRef;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;


@Slf4j
@Service
public class DycProEncodeSerialServiceImpl implements DycProEncodeSerialService {

    @Value("${encode.write.table.max.num:100}")
    private int maxNum;

    @Autowired
    private CacheClient cacheClient;

    /**
     * 实例前缀连接符
     */
    private static final String PREFIX_INSTANCE = "PK_ENCODE_INSTANCE";

    /**
     * 规则前缀连接符
     */
    private static final String PREFIX_RULE = "PK_ENCODE_RULE";
    /**
     * 下划线连接符
     */
    private static final String CONNECTOR_UNDERLINE = "_";

    @Autowired
    private SysSerialInstanceMapper sysSerialInstanceMapper;

    @HTServiceRef
    private DycProEncodeSerialRuleGetService dycProEncodeSerialRuleGetService;

    /**
     * 流水号占位符
     */
    private static final String SERIAL_SYMBOL = "#[SERIAL_SYMBOL]#";


    @Override
    public DycProEncodeSerialRspBO getEncode(DycProEncodeSerialReqBO reqBO) {
        //参数校验
        val(reqBO);

        String encodeSerialRuleKey = PREFIX_RULE + CONNECTOR_UNDERLINE + reqBO.getCenterCode() + CONNECTOR_UNDERLINE + reqBO.getEncodeRuleCode() + CONNECTOR_UNDERLINE;
        String encodeSerialInstanceKey = PREFIX_INSTANCE + CONNECTOR_UNDERLINE + reqBO.getCenterCode() + CONNECTOR_UNDERLINE + reqBO.getEncodeRuleCode() + CONNECTOR_UNDERLINE;
        //获取编码规则
        DycProEncodeSerialRuleBO dycProEncodeSerialRuleBO = getSerialRule(encodeSerialRuleKey, encodeSerialInstanceKey, reqBO);

        //创建前缀实例
        String prefix = createPrefix(dycProEncodeSerialRuleBO);

        //初始化步长
        Long stepRange = 1L;
        if(null != dycProEncodeSerialRuleBO.getStepRange()){
            stepRange = dycProEncodeSerialRuleBO.getStepRange();
        }

        //初始化编号数量
        Long num = 1L;
        if(null != reqBO.getNum()){
            num = reqBO.getNum();
        }

        //实例编码key
        String encodeSerialGenerateKey = encodeSerialInstanceKey + CONNECTOR_UNDERLINE + prefix;

        //得到总长度
        Long numRange = num * stepRange;
        //获取最新流水号
        Long newSerial = cacheClient.incrBy(encodeSerialGenerateKey, numRange);
        //传入值与返回值一样，则是redis无生效日期编号的数据，需要清理过期编码
        if(newSerial.equals(numRange)){
            //清理过期的key
            clearExpiredKey(encodeSerialInstanceKey, encodeSerialGenerateKey);
        }

        List<String> serialNoList = new ArrayList<>();
        List<SysSerialInstancePO> serialInstancePOS = new ArrayList<>();
        //计算流水号
        for (int i = num.intValue() - 1; i >= 0; i--) {
            String serial = String.format("%0" + dycProEncodeSerialRuleBO.getSerialNum() + "d", newSerial - ( i * stepRange ));
            String serialNo = prefix.replace(SERIAL_SYMBOL, serial);
            serialNoList.add(serialNo);

            //达到次数组装实例数据对象
            if ((newSerial - ( i * stepRange ))% maxNum == 0) {
                SysSerialInstancePO serialInstancePO = new SysSerialInstancePO();
                serialInstancePO.setEncodeRuleId(dycProEncodeSerialRuleBO.getEncodeRuleId());
                serialInstancePO.setPrefix(prefix);
                serialInstancePO.setSerial(serial);
                serialInstancePO.setInstanceId(IdWorker.getId());
                serialInstancePOS.add(serialInstancePO);
            }
        }

        //记录到达次数的编号实例
        if(!CollectionUtils.isEmpty(serialInstancePOS)){
            sysSerialInstanceMapper.insertBatch(serialInstancePOS);
        }

        DycProEncodeSerialRspBO rspBO = new DycProEncodeSerialRspBO();
        rspBO.setSerialNoList(serialNoList);
        return rspBO;
    }

    private void clearExpiredKey(String encodeSerialInstanceKey, String encodeSerialGenerateKey) {
        Set<String> keys = cacheClient.getkeys(encodeSerialInstanceKey + "*");
        if (!CollectionUtils.isEmpty(keys)) {
            for (String key : keys) {
                //不是当前生成的就删除
                if(!encodeSerialGenerateKey.equals(key)) {
                    cacheClient.delete(key);
                }
            }
        }
    }

    private DycProEncodeSerialRuleBO getSerialRule(String encodeSerialRuleKey, String encodeSerialInstanceKey, DycProEncodeSerialReqBO reqBO) {
        //使用redis，获取PO数据
        Object encodeSerial = cacheClient.get(encodeSerialRuleKey);
        DycProEncodeSerialRuleBO dycProEncodeSerialRuleBO;
        if (encodeSerial == null) {

            //从数据库获取规则
            DycProEncodeSerialRuleGetRspBO dycProEncodeSerialRuleGetRspBO = getRuleByDb(reqBO);
            synchronized (this) {
                dycProEncodeSerialRuleBO = dycProEncodeSerialRuleGetRspBO.getDycProEncodeSerialRuleBO();

                if (null == dycProEncodeSerialRuleBO.getSerialNum()) {
                    throw new ZTBusinessException("序号号位数为空");
                }

                //编码规则放到redis
                cacheClient.set(encodeSerialRuleKey, JSON.toJSONString(dycProEncodeSerialRuleBO));
                //实例编码key
                String encodeSerialGenerateKey = encodeSerialInstanceKey + CONNECTOR_UNDERLINE + createPrefix(dycProEncodeSerialRuleBO);

                Long serialValue = dycProEncodeSerialRuleGetRspBO.getSerialValue();
                if (serialValue > 0) {
                    serialValue = serialValue + maxNum;
                }
                //将数据库记录达到次数的流水初始化到redis
                cacheClient.incrBy(encodeSerialGenerateKey, serialValue);
            }

        } else {
            dycProEncodeSerialRuleBO = JSON.parseObject(encodeSerial.toString(), DycProEncodeSerialRuleBO.class);
        }

        if(null == dycProEncodeSerialRuleBO){
            throw new ZTBusinessException("没有配置编码规则");
        }
        return dycProEncodeSerialRuleBO;
    }

    private DycProEncodeSerialRuleGetRspBO getRuleByDb(DycProEncodeSerialReqBO reqBO) {
        DycProEncodeSerialRuleGetReqBO dycProEncodeSerialRuleGetReqBO = JSON.parseObject(JSON.toJSONString(reqBO), DycProEncodeSerialRuleGetReqBO.class);
        DycProEncodeSerialRuleGetRspBO dycProEncodeSerialRuleGetRspBO = dycProEncodeSerialRuleGetService.getRule(dycProEncodeSerialRuleGetReqBO);
        if(!DycProCoreRspConstant.RSP_CODE_SUCCEED.equals(dycProEncodeSerialRuleGetRspBO.getRespCode())){
            throw new ZTBusinessException(dycProEncodeSerialRuleGetRspBO.getRespDesc());
        }
        return dycProEncodeSerialRuleGetRspBO;
    }


    private static void val(DycProEncodeSerialReqBO reqBO) {
        if (reqBO == null) {
            throw new ZTBusinessException("编码规则为空");
        }
        if (StringUtils.isBlank(reqBO.getCenterCode())) {
            throw new ZTBusinessException("编码规则所属模块为空");
        }
        if (StringUtils.isBlank(reqBO.getEncodeRuleCode())) {
            throw new ZTBusinessException("编码规则编码为空");
        }
    }

    private String createPrefix(DycProEncodeSerialRuleBO dycProEncodeSerialRuleBO){

        String ruleOrder = dycProEncodeSerialRuleBO.getRuleOrder();

        //空则默认顺序  1:前缀 2:一级连接符 3:日期 4:二级连接符 5:序列号
        if(null == ruleOrder){
            ruleOrder = "1-2-3-4-5";
        }
        String[] orderArray = ruleOrder.split("-");
        StringBuilder prefix = new StringBuilder();

        for (String order : orderArray) {
            if(DycProCoreConstant.SerialRuleOrder.ONE.equals(order) && !StringUtils.isBlank(dycProEncodeSerialRuleBO.getPrefix())){
                prefix.append(dycProEncodeSerialRuleBO.getPrefix());
            }
            if(DycProCoreConstant.SerialRuleOrder.TOW.equals(order) && !StringUtils.isBlank(dycProEncodeSerialRuleBO.getFirstConnectSymbol())){
                prefix.append(dycProEncodeSerialRuleBO.getFirstConnectSymbol());
            }
            if(DycProCoreConstant.SerialRuleOrder.THREE.equals(order) && !StringUtils.isBlank(dycProEncodeSerialRuleBO.getDateInfix())){
                DateTimeFormatter df = DateTimeFormatter.ofPattern(dycProEncodeSerialRuleBO.getDateInfix());
                String date = df.format(LocalDateTime.now());
                prefix.append(date);
            }
            if(DycProCoreConstant.SerialRuleOrder.FOUR.equals(order) && !StringUtils.isBlank(dycProEncodeSerialRuleBO.getSecondConnectSymbol())){
                prefix.append(dycProEncodeSerialRuleBO.getSecondConnectSymbol());
            }
            if(DycProCoreConstant.SerialRuleOrder.FIVE.equals(order)){
                prefix.append(SERIAL_SYMBOL);
            }
        }
        return prefix.toString();
    }

}
