package com.tydic.picker.advice;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.ReflectUtil;
import com.alibaba.fastjson.JSON;
import com.tydic.picker.PickerClient;
import com.tydic.picker.constant.PickerConstants;
import com.tydic.picker.dto.DataPickDTO;
import com.tydic.picker.enums.ResultCodeEnum;
import com.tydic.picker.result.PickerResult;
import lombok.extern.slf4j.Slf4j;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;

import java.lang.reflect.Field;
import java.util.*;

/**
 * @ClassName SyncDataByPointCutAdvice
 * @Description 数据同步 切面
 * @Author liugs
 * @Date 2022/8/16 10:38
 */
@Slf4j
public class SyncDataByPointCutAdvice implements MethodInterceptor {

    private PickerClient pickerClient;

    public SyncDataByPointCutAdvice(PickerClient pickerClient) {
        this.pickerClient = pickerClient;
    }

    /**
     * 获取服务全路径、方法名称，出入参，发布事件
     *
     * @param invocation
     * @return
     * @throws Throwable
     */
    @Override
    public Object invoke(MethodInvocation invocation) throws Throwable {

        Map<String, Object> fieldValueMap = new HashMap<>(16);
        Object rspObj = invocation.proceed();
        // 入参处理
        dealInArgument(invocation.getArguments(), fieldValueMap);
        if (log.isDebugEnabled()) {
            log.debug("入参加入map：{}", JSON.toJSONString(fieldValueMap));
        }

        // 判断出参同步标识字段，字段不为空且值为指定值，不进行数据同步，其他情况进行数据同步。
        Object syncFlag = ReflectUtil.getFieldValue(rspObj, PickerConstants.SYNC_FLAG);
        if (ObjectUtil.isNotEmpty(syncFlag) && PickerConstants.SYNC_FLAG_NO.equals(syncFlag)) {
            log.info("同步标识指定本次不进行数据同步操作。");
            return rspObj;
        }

        /**
         *1.获取输入输出参数和里面对应的值,出参处理,对请求和响应参数key，value封装到map中，同名参数->按照出参数覆盖入参数的原则
         */
        dealObject(rspObj, fieldValueMap);

        DataPickDTO pickDTO = DataPickDTO.builder()
                .serviceCode(invocation.getMethod().getDeclaringClass().getName())
                .methodCode(invocation.getMethod().getName())
                .conditionParam(fieldValueMap)
                .build();
        PickerResult result = pickerClient.execute(pickDTO);

        if (!ResultCodeEnum.SUCCESS.getCode().equals(result.getCode())) {
            log.error("同步数据失败：{}", result.getMessage());
        }

        return rspObj;
    }

    /**
     * 获取所有入参的集合
     *
     * @param arguments
     * @return
     */
    private void dealInArgument(Object[] arguments, Map<String, Object> map) throws Throwable {
        if (arguments != null) {
            for (Object arg : arguments) {
                dealObject(arg, map);
            }
        }
    }

    /**
     * 对单个对象进行处理
     *
     * @param dealObject
     * @throws Throwable
     */
    private void dealObject(Object dealObject, Map<String, Object> fieldValueMap) throws Throwable {
        final List<Field> fields = getAllField(dealObject.getClass());
        for (Field f : fields) {
            //设置些属性是可以访问的
            f.setAccessible(true);
            //属性名
            String fieldName = f.getName();
            if (null != f.get(dealObject)) {
                Object fieldValue = f.get(dealObject);
                fieldValueMap.put(fieldName, fieldValue);
            }
        }
    }

    /**
     * 获取当前类的所有字段
     *
     * @param clazz
     * @return
     */
    private static List<Field> getAllField(Class clazz) {
        List<Field> fields = new ArrayList<>();
        while (clazz != null) {
            fields.addAll(new ArrayList<>(Arrays.asList(clazz.getDeclaredFields())));
            clazz = clazz.getSuperclass();
        }
        return fields;
    }
}
