package com.tydic.picker.config;

import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpHost;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.nio.client.HttpAsyncClientBuilder;
import org.apache.http.impl.nio.reactor.IOReactorConfig;
import org.elasticsearch.client.Node;
import org.elasticsearch.client.RestClient;
import org.elasticsearch.client.RestClientBuilder;
import org.elasticsearch.client.RestHighLevelClient;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * @ClassName ElasticSearchConfiguration
 * @Description ES RestClient 配置
 * @Author liugs
 * @Date 2021/3/19 17:31:56
 */
@Slf4j
@Order(-1)
@Configuration
@ConditionalOnProperty(prefix = "picker.client", name = "enable", havingValue = "true")
public class ElasticSearchConfiguration {

    /**
     * 地址
     */
    @Value("${es.cluster.address}")
    private String clusterNodes;

    @Value("${es.source}")
    private String esSource;

    @Value(("${es.client.username}"))
    private String esClientUsername;

    @Value(("${es.client.password}"))
    private String esClientPassword;
    /**
     * 连接数
     */
    @Value(("${es.pool.maxTotal}"))
    private int maxTotal;
    /**
     * 获取连接超时时间
     */
    @Value(("${es.pool.connectionTimeout}"))
    private int connectionTimeout;
    /**
     * 套接字超时超时时间
     */
    @Value(("${es.pool.socketTimeout}"))
    private int socketTimeout;

    @Value(("${es.pool.connectionRequestTimeout}"))
    private int connectionRequestTimeout;

    @Bean(value = "restHighLevelClient", destroyMethod = "close")
    @ConditionalOnMissingBean(RestHighLevelClient.class)
    public RestHighLevelClient buildRestClient() {
        RestClientBuilder clientBuilder = null;
        try {
            if (StringUtils.isEmpty(clusterNodes)) {
                return null;
            }
            List<String> ipAddress = new ArrayList<>(Arrays.asList(clusterNodes.split(",")));
            HttpHost httpHost;
            HttpHost[] httpHosts = new HttpHost[ipAddress.size()];
            for (int i = 0; i < ipAddress.size(); i++) {
                httpHost = new HttpHost(ipAddress.get(i).split(":")[0],
                        Integer.valueOf(ipAddress.get(i).split(":")[1]),
                        "http");
                httpHosts[i] = (httpHost);
            }
            clientBuilder = RestClient.builder(
                    httpHosts);

            /*clientBuilder =RestClient.builder(
                new HttpHost(
                        clusterNodes.split(":")[0],
                        9200,
                        "http"));*/
        } catch (Exception e) {
            e.printStackTrace();
        }
        // 设置超时时间
        //clientBuilder.(maxRetryTimeoutMillis);


        // 设置监听器，每次节点失败都可以监听到，可以作额外处理
        clientBuilder.setFailureListener(new RestClient.FailureListener() {
            public void onFailure(HttpHost host) {
                super.onFailure(new Node(host));
                System.out.println(host.getHostName() + "==节点失败了");
            }
        });

        /* 配置异步请求的线程数量，Apache Http Async Client默认启动一个调度程序线程，以及由连接管理器使用的许多工作线程
        （与本地检测到的处理器数量一样多，取决于Runtime.getRuntime().availableProcessors()返回的数量）。线程数可以修改如下,
        这里是修改为1个线程，即默认情况
        */
        clientBuilder.setHttpClientConfigCallback(new RestClientBuilder.HttpClientConfigCallback() {
            @Override
            public HttpAsyncClientBuilder customizeHttpClient(HttpAsyncClientBuilder httpAsyncClientBuilder) {
                return httpAsyncClientBuilder.setDefaultIOReactorConfig(
                        IOReactorConfig.custom().setIoThreadCount(maxTotal).build()
                );
            }
        });

        /*
        配置请求超时，将连接超时（默认为1秒）和套接字超时（默认为30秒）增加，
        这里配置完应该相应地调整最大重试超时（默认为30秒），即上面的setMaxRetryTimeoutMillis，一般于最大的那个值一致即60000
        */
        clientBuilder.setRequestConfigCallback(new RestClientBuilder.RequestConfigCallback() {
            @Override
            public RequestConfig.Builder customizeRequestConfig(RequestConfig.Builder requestConfigBuilder) {
                // 连接5秒超时，套接字连接60s超时
                return requestConfigBuilder.setConnectTimeout(connectionTimeout).setSocketTimeout(socketTimeout).setConnectionRequestTimeout(connectionRequestTimeout);
            }
        });

        /*
        如果ES设置了密码，那这里也提供了一个基本的认证机制，下面设置了ES需要基本身份验证的默认凭据提供程序
        */
        if ("AliYun".equals(esSource)) {
            final CredentialsProvider credentialsProvider = new BasicCredentialsProvider();
            credentialsProvider.setCredentials(AuthScope.ANY,
                    new UsernamePasswordCredentials(esClientUsername, esClientPassword));
            clientBuilder.setHttpClientConfigCallback(httpClientBuilder -> httpClientBuilder.setDefaultCredentialsProvider(credentialsProvider));
        }


        /*
          上面采用异步机制实现抢先认证，这个功能也可以禁用，这意味着每个请求都将在没有授权标头的情况下发送，然后查看它是否被接受，
          并且在收到HTTP 401响应后，它再使用基本认证头重新发送完全相同的请求，这个可能是基于安全、性能的考虑
         */
        /*clientBuilder.setHttpClientConfigCallback(new RestClientBuilder.HttpClientConfigCallback() {
            @Override
            public HttpAsyncClientBuilder customizeHttpClient(HttpAsyncClientBuilder httpClientBuilder) {
                // 禁用抢先认证的方式
                httpClientBuilder.disableAuthCaching();
                return httpClientBuilder.setDefaultCredentialsProvider(credentialsProvider);
            }
        });*/

        return new RestHighLevelClient(clientBuilder);
    }
}

