package com.tydic.picker.service.imp;

import cn.hutool.core.codec.Base64Encoder;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.tydic.picker.dto.DataPickDTO;
import com.tydic.picker.dto.DataSyncConfigSubQueryDTO;
import com.tydic.picker.dto.PickerRecordDTO;
import com.tydic.picker.enums.ConstantEnum;
import com.tydic.picker.enums.ResultCodeEnum;
import com.tydic.picker.result.PickerResult;
import com.tydic.picker.service.DataSyncService;
import com.tydic.picker.utils.DynamicUtil;
import com.tydic.picker.utils.ElasticsearchUtil;
import com.tydic.picker.utils.Sequence;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.jdbc.core.JdbcTemplate;

import java.util.List;
import java.util.Map;

/**
 * @ClassName DataSyncServiceImpl
 * @Description 数据同步服务
 * @Author liugs
 * @Date 2022/8/15 16:48
 */
@Slf4j
public class DataSyncServiceImpl implements DataSyncService {

    /** 文档ID取值字段 */
    private static final String DOCUMENT_ID = "DOCUMENT_ID";
    /** 最大深度 */
    private static final int MAXIMUM_DEPTH = 2;

    private JdbcTemplate jdbcTemplate;

    private ElasticsearchUtil elasticsearchUtil;

    public DataSyncServiceImpl(JdbcTemplate jdbcTemplate,
                               ElasticsearchUtil elasticsearchUtil) {
        this.jdbcTemplate = jdbcTemplate;
        this.elasticsearchUtil = elasticsearchUtil;
    }

    @Override
    public PickerResult doSync(DataPickDTO dto) {
        log.info("==开始采集数据并同步ES");
        PickerResult result = new PickerResult(ResultCodeEnum.SUCCESS);
        // 记录同步信息
        Long recordId = null;
        try {
            recordId = createSyncRecord(dto);
        } catch (Exception e) {
            log.error("数据同步记录入库异常：{}", e);
        }

        String errorMessage = null;
        try {
            // 执行动态SQL并根据映射配置填充ES对象
            JSONObject esData = getMappedData(dto);
            log.debug("根据映射关系的到的数据对象：{}", JSON.toJSONString(esData, SerializerFeature.WriteMapNullValue));
            if (ObjectUtil.isNotEmpty(esData.get(DOCUMENT_ID))) {
                switch (dto.getEventTypeEnum()) {
                    case CREATE:
                        createDoc(esData, dto);
                        break;
                    case UPDATE:
                        updateDoc(esData, dto);
                        break;
                    case DELETE:
                        deleteDoc(esData, dto);
                        break;
                    default:
                        break;
                }
            } else {
                errorMessage = String.format("根据配置的文档ID取值字段[%s]未获取到文档ID", dto.getDocIdField());
            }
        } catch (Exception e) {
            log.error("同步ES数据异常：");
            e.printStackTrace();
            errorMessage = "同步ES数据异常：" + e;
        }

        // 更新同步信息
        try {
            if (StringUtils.isEmpty(errorMessage)) {
                updateRecord(recordId, ConstantEnum.PickerRecordStatusEnum.SUCCESS, ConstantEnum.PickerRecordStatusEnum.SUCCESS.getDesc());
            } else {
                updateRecord(recordId, ConstantEnum.PickerRecordStatusEnum.FAIL, errorMessage);
            }
        } catch (Exception e) {
            log.error("数据同步记录状态更新异常：{}", e);
        }

        if (StringUtils.isNotEmpty(errorMessage)) {
            result.buildFail(errorMessage);
        }
        log.info("==采集数据并同步ES完成：{}", JSON.toJSONString(result));
        return result;
    }

    /**
     * 创建文档
     * @param esData
     * @param dto
     */
    private void createDoc(JSONObject esData, DataPickDTO dto) {
        elasticsearchUtil.addDocument(dto.getIndexName(), esData.remove(DOCUMENT_ID).toString(), esData);
    }

    /**
     * 更新文档
     * @param esData
     * @param dto
     */
    private void updateDoc(JSONObject esData, DataPickDTO dto) {
        elasticsearchUtil.updateDoc(dto.getIndexName(), esData.remove(DOCUMENT_ID).toString(), esData);
    }

    /**
     * 删除文档
     * @param esData
     * @param dto
     */
    private void deleteDoc(JSONObject esData, DataPickDTO dto) {
        elasticsearchUtil.deleteDocById(dto.getIndexName(), esData.getString(DOCUMENT_ID));
    }


    /**
     * 处理同步记录
     * @param dto
     * @return recordId
     */
    private Long createSyncRecord(DataPickDTO dto) {
        Long id = Sequence.nextId();
        PickerRecordDTO recordPo = new PickerRecordDTO();
        BeanUtils.copyProperties(dto, recordPo);
        recordPo.setId(id);
        recordPo.setEventType(dto.getEventTypeEnum().getEventType());
        recordPo.setStartTime(new DateTime().toString("yyyy-MM-dd HH:mm:ss"));
        recordPo.setStatusDesc(ConstantEnum.PickerRecordStatusEnum.INIT.getDesc());
        recordPo.setDynamicSql(StringUtils.isEmpty(dto.getDynamicSql()) ? null : Base64Encoder.encode(dto.getDynamicSql()));
        String dynamicSql = ConstantEnum.DynamicSqlEnum.INSERT.getDynamicSql(JSON.parseObject(JSON.toJSONString(recordPo), Map.class));
        log.debug("记录数据同步SQL：{}", dynamicSql);
        if (1 > jdbcTemplate.update(dynamicSql)) {
            log.error("记录数据同步记录失败");
            return null;
        }
        return id;
    }

    /**
     * 更新记录状态
     * @param recordId
     * @param statusEnum
     * @param desc
     * @return
     */
    private void updateRecord(Long recordId, ConstantEnum.PickerRecordStatusEnum statusEnum, String desc) {
        PickerRecordDTO recordPo = new PickerRecordDTO();
        recordPo.setId(recordId);
        recordPo.setStatus(statusEnum.getState());
        recordPo.setStatusDesc(desc);
        recordPo.setFinishTime(new DateTime().toString("yyyy-MM-dd HH:mm:ss"));
        String dynamicSql = ConstantEnum.DynamicSqlEnum.UPDATE.getDynamicSql(JSON.parseObject(JSON.toJSONString(recordPo), Map.class));
        log.debug("记录数据同步SQL：{}", dynamicSql);
        if ( 1 > jdbcTemplate.update(dynamicSql)) {
            log.error("更新数据同步记录失败");
        }
    }


    /**
     * 描述 获取数据并做映射
     * @param dto
     * @return com.alibaba.fastjson.JSONObject
     * @author liugs
     * @date 2022/8/17 11:38
     */
    public JSONObject getMappedData(DataPickDTO dto) {
        // 字段映射
        JSONObject mappingRelJson = JSON.parseObject(dto.getMappingInfo());
        // 将ES文档取值字段加入到映射
        mappingRelJson.put(dto.getDocIdField(), DOCUMENT_ID);
        JSONObject mappedDataJson = new JSONObject(true);
        // 出入参参数合集
        Map<String, Object> conditionParam = JSON.parseObject(dto.getConditionParam());
        // 删除操作的条件数据从出入参来
        if (StringUtils.isEmpty(dto.getDynamicSql())) {
            for (Map.Entry<String, Object> entry : conditionParam.entrySet()) {
                mappedDataJson.put(entry.getValue().toString(), conditionParam.get(entry.getKey()));
            }
            // 根据配置的文档ID字段取值文档ID
            mappedDataJson.put(DOCUMENT_ID, conditionParam.get(dto.getDocIdField()));
            return mappedDataJson;
        }
        // 新增、更新，查询数据库
        String dynamicSql = DynamicUtil.buildDynamicSql(dto.getDynamicSql(), conditionParam);
        log.info("执行的取值动态语句为：{}", dynamicSql);

        Map<String, Object> dynamicSqlResult =  buildSingleData(mappedDataJson, mappingRelJson, dynamicSql);

        // 构建子查询
        buildSubField(mappedDataJson, dynamicSqlResult, dto.getSubQuery(), 0);
        return mappedDataJson;
    }

    /**
     * 描述 构建子查询
     * @param parentData            父级数据对象
     * @param conditionMap          条件map
     * @param subQuery              子查询列表
     * @param depth                 子查询纵向深度
     * @author liugs
     * @date 2022/8/31 15:41
     */
    private void buildSubField(JSONObject parentData, Map<String, Object> conditionMap, List<DataSyncConfigSubQueryDTO> subQuery, int depth) {
        // 超过最大深度，子查询为空，直接返回
        if (MAXIMUM_DEPTH < depth || CollectionUtils.isEmpty(subQuery)) {
            return;
        }
        // 字段映射
        JSONObject fieldMapping;
        // 构建动态查询语句
        String dynamicSql;
        // 遍历子查询列表
        for (DataSyncConfigSubQueryDTO subQueryItem : subQuery) {
            fieldMapping = JSON.parseObject(subQueryItem.getMappingInfo());
            dynamicSql = DynamicUtil.buildDynamicSql(subQueryItem.getDataFilter(), conditionMap);
            log.info("[{}]级子查询取值动态语句为：{}", depth, dynamicSql);

            // 本层级查询是列表
            if (ConstantEnum.DataType.LIST.equal(subQueryItem.getDataType())) {
                parentData.put(subQueryItem.getParentField(), buildMultipleData(fieldMapping, dynamicSql));
                continue;
            }
            // 本层级查询是单个字段
            if (ConstantEnum.DataType.FIELD.equal(subQueryItem.getDataType())) {
                parentData.put(subQueryItem.getParentField(), buildFieldData(fieldMapping, dynamicSql));
                continue;
            }

            // 本层级的查询不是列表，子查询配置不为空
            JSONObject subData = new JSONObject();
            // 构建单一数据，并将SQL结果返回，作为子查询的条件参数
            Map<String, Object> dynamicSqlResult = buildSingleData(subData, fieldMapping, dynamicSql);
            if (CollectionUtils.isNotEmpty(subQueryItem.getSubQuery())) {
                buildSubField(subData, dynamicSqlResult, subQueryItem.getSubQuery(), ++ depth);
                -- depth;
            }
            parentData.put(subQueryItem.getParentField(), subData);
        }
    }

    /**
     * 描述  构建字段型数据
     * @param fieldMapping
     * @param dynamicSql
     * @return java.lang.Object
     * @author liugs
     * @date 2022/9/1 16:14
     */
    private String buildFieldData(JSONObject fieldMapping, String dynamicSql) {
        String fieldValue = null;
        Map<String, Object> resultDataMap = jdbcTemplate.queryForMap(dynamicSql);
        log.debug("执行动态语句获取到的数据：{}", JSON.toJSONString(resultDataMap));
        // 根据配置完成字段映射
        for (Map.Entry<String, Object> entry : fieldMapping.entrySet()) {
            Object value = resultDataMap.get(entry.getKey());
            if (!ObjectUtil.isNotEmpty(value)) {
                fieldValue = value.toString();
                break;
            }
        }
        return fieldValue;
    }

    /**
     * 描述 构建多数据
     * @param fieldMapping
     * @param dynamicSql
     * @return void
     * @author liugs
     * @date 2022/8/31 14:52
     */
    private JSONArray buildMultipleData(JSONObject fieldMapping, String dynamicSql) {
        JSONArray subArrayData = new JSONArray();
        // 查询数据库
        List<Map<String, Object>> queryForList = jdbcTemplate.queryForList(dynamicSql);
        if (CollectionUtils.isEmpty(queryForList)) {
            log.error("语句[{}]未获取到数据", dynamicSql);
            return subArrayData;
        }
        // 遍历查询结果
        queryForList.forEach(item -> {
            // 根据映射关系构建数据对象
            JSONObject subDateItem = new JSONObject();
            for (Map.Entry<String, Object> entry : fieldMapping.entrySet()) {
                subDateItem.put(entry.getValue().toString(), item.get(entry.getKey()));
            }
            subArrayData.add(subDateItem);
        });
        return subArrayData;
    }

    /**
     * 构建单一数据
     * @param targetData
     * @param fieldMapping
     * @param dynamicSql
     */
    private Map<String, Object> buildSingleData(JSONObject targetData, JSONObject fieldMapping, String dynamicSql) {
        // 查询单一数据
        Map<String, Object> resultDataMap = jdbcTemplate.queryForMap(dynamicSql);
        log.debug("执行动态语句获取到的数据：{}", JSON.toJSONString(resultDataMap));

        // 根据配置完成字段映射
        for (Map.Entry<String, Object> entry : fieldMapping.entrySet()) {
            targetData.put(entry.getValue().toString(), resultDataMap.get(entry.getKey()));
        }
        return resultDataMap;
    }


}
