package com.tydic.picker;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.ohaotian.plugin.cache.CacheClient;
import com.tydic.picker.dto.DataPickDTO;
import com.tydic.picker.dto.DataSyncConfigDTO;
import com.tydic.picker.enums.ConstantEnum;
import com.tydic.picker.enums.DataEventTypeEnum;
import com.tydic.picker.enums.ResultCodeEnum;
import com.tydic.picker.publisher.PickerClientPublisher;
import com.tydic.picker.result.PickerResult;
import com.tydic.picker.service.DataSyncService;
import lombok.extern.slf4j.Slf4j;

import java.util.concurrent.atomic.AtomicReference;

/**
 * @ClassName PickerClient
 * @Description 数据同步客户端
 * @Author liugs
 * @Date 2022/8/15 16:58
 */
@Slf4j
public class PickerClient {

    /** 缓存key连接符 */
    private static final String POINT = ".";

    /** 配置key编码连接符 */
    private static final String UNDERSCORE = "_";

    /** 缓存配置编号字段 */
    private static final String PICKER_CONFIG_NO = "pickerConfigNo";

    /** 同步事件发布者 */
    private PickerClientPublisher publisher;
    /** 缓存客户端 */
    private CacheClient cacheClient;
    /** 数据同步服务 */
    private DataSyncService dataSyncService;

    /**
     * init
     * @param publisher
     * @param cacheClient
     * @param dataSyncService
     */
    public void init(PickerClientPublisher publisher, CacheClient cacheClient, DataSyncService dataSyncService) {
        this.publisher = publisher;
        this.cacheClient = cacheClient;
        this.dataSyncService = dataSyncService;
    }

    /**
     * Close
     */
    public void close() {
        publisher.close();
    }

    /**
     * 描述 执行
     * @param dto
     * @return com.tydic.picker.result.PickerResult
     * @author liugs
     * @date 2022/8/16 11:05
     */
    public PickerResult execute(DataPickDTO dto) {
        AtomicReference<PickerResult> pickerResult = new AtomicReference<>(new PickerResult(ResultCodeEnum.SUCCESS));
        // 根据服务编码+方法编码确定缓存KEY
        StringBuilder cacheKeyBuilder = new StringBuilder();
        cacheKeyBuilder.append(dto.getServiceCode())
                .append(POINT)
                .append(dto.getMethodCode());
        // 判断参数集合中是否有指定的配置编码，有加上，作为缓存key
        Object pickerConfigNo = dto.getConditionParam().get(PICKER_CONFIG_NO);
        if (ObjectUtil.isNotEmpty(pickerConfigNo)) {
            cacheKeyBuilder.append(UNDERSCORE).append(pickerConfigNo);
        }

        Object cacheInfo = cacheClient.get(cacheKeyBuilder.toString());
        if (ObjectUtil.isEmpty(cacheInfo)) {
            log.info("方法：[{}]没有配置数据同步信息", cacheKeyBuilder);
            return pickerResult.get();
        }
        JSONArray syncConfigArray = JSON.parseArray(String.valueOf(cacheInfo));
        syncConfigArray.forEach(item -> {
            // 配置信息转对象
            DataSyncConfigDTO configDTO = JSON.parseObject(item.toString(), DataSyncConfigDTO.class);
            // 构建同步信息
            dto.setBusiCode(configDTO.getBusiCode());
            dto.setIndexName(configDTO.getIndexName());
            dto.setDynamicSql(configDTO.getDataFilter());
            dto.setMappingInfo(configDTO.getMappingInfo());
            dto.setEventTypeEnum(DataEventTypeEnum.acquireByValue(configDTO.getEventType()));
            dto.setDocIdField(configDTO.getDocValueFiled());
            dto.setSyncType(Integer.valueOf(configDTO.getSyncType()));
            dto.setSubQuery(configDTO.getSubQuery());

            // 判断同步还是异步
            if (ConstantEnum.SyncTypeEnum.SYNC.getSyncType().equals(configDTO.getSyncType())) {
                pickerResult.set(dataSyncService.doSync(dto));
            } else {
                publisher.publish(dto);
            }
        });

        return pickerResult.get();
    }
}
