package com.tydic.picker.task;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.tydic.picker.config.FastJsonConfig;
import com.tydic.picker.constant.PickerConstants;
import com.tydic.picker.dto.DataPickDTO;
import com.tydic.picker.enums.ResultCodeEnum;
import com.tydic.picker.utils.DynamicSqlUtil;
import com.tydic.picker.utils.ElasticsearchUtil;
import lombok.Builder;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;

/**
 * @ClassName PartitionTask
 * @Description 分片任务
 * @Author liugs
 * @Date 2023/1/31 11:12
 */
@Slf4j
@Builder
public class PartitionTask implements Callable<Object> {

    /** 记录ID */
    private Long recordId;

    /** DataPickDTO */
    private DataPickDTO dto;

    /** ES 客户端*/
    private ElasticsearchUtil elasticsearchUtil;

    /** 动态 */
    private DynamicSqlUtil dynamicSqlUtil;

    /** 主数据查询结果列表 */
    private List<Map<String, Object>> mainDataList;

    /** 当前分片查询页 */
    private Integer pageNo;


    @Override
    public Object call() {
        PartitionResult result = new PartitionResult();
        result.setCode(ResultCodeEnum.FAIL.getCode());
        try {
            for (Map<String, Object> mainData : mainDataList) {
                // 执行动态SQL并根据映射配置填充ES对象
                JSONObject esData = dynamicSqlUtil.getMappedDataForBatch(dto, mainData);
                log.debug("根据映射关系的到的数据对象：{}", JSON.toJSONString(esData, FastJsonConfig.getSerializerFeature()));

                if (ObjectUtil.isNotEmpty(esData.get(PickerConstants.DOCUMENT_ID))) {
                    elasticsearchUtil.doSyncByEventType(dto.getEventTypeEnum(), dto.getIndexName(), esData);
                } else {
                    result.setMessage(String.format("根据配置的文档ID取值字段[%s]未获取到文档ID", dto.getDocIdField()));
                }
            }
        } catch (Exception e) {
            log.error("同步ES数据异常：");
            e.printStackTrace();
            result.setMessage(String.format("同步任务ID【%d】，第【%d】分片同步发生异常：【%s】", recordId, pageNo, e.getMessage()));
            return result;
        }

        result.setCode(ResultCodeEnum.SUCCESS.getCode());
        return result;
    }
}
