package com.tydic.picker.service.imp;

import cn.hutool.core.codec.Base64Encoder;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.text.StrBuilder;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.tydic.picker.constant.PickerConstants;
import com.tydic.picker.dto.DataPickDTO;
import com.tydic.picker.dto.PickerRecordDTO;
import com.tydic.picker.enums.ConstantEnum;
import com.tydic.picker.enums.DataEventTypeEnum;
import com.tydic.picker.enums.ResultCodeEnum;
import com.tydic.picker.result.PickerResult;
import com.tydic.picker.service.DataSyncService;
import com.tydic.picker.task.PartitionResult;
import com.tydic.picker.task.PartitionTask;
import com.tydic.picker.utils.DynamicSqlUtil;
import com.tydic.picker.utils.ElasticsearchUtil;
import com.tydic.picker.utils.Sequence;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.jdbc.core.JdbcTemplate;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @ClassName DataSyncServiceImpl
 * @Description 数据同步服务
 * @Author liugs
 * @Date 2022/8/15 16:48
 */
@Slf4j
public class DataSyncServiceImpl implements DataSyncService {

    @Resource
    private DynamicSqlUtil dynamicSqlUtil;

    private JdbcTemplate jdbcTemplate;

    private ElasticsearchUtil elasticsearchUtil;

    private ExecutorService partitionRequestPool;

    private Integer pageSize;

    public DataSyncServiceImpl(JdbcTemplate jdbcTemplate, ElasticsearchUtil elasticsearchUtil,
                               ExecutorService partitionRequestPool, Integer pageSize) {
        this.jdbcTemplate = jdbcTemplate;
        this.elasticsearchUtil = elasticsearchUtil;
        this.partitionRequestPool = partitionRequestPool;
        this.pageSize = pageSize;
    }

    @Override
    public PickerResult doSync(DataPickDTO dto) {
        log.info("==开始采集数据并同步ES");

        // 记录同步信息
        Long recordId = null;
        try {
            recordId = createSyncRecord(dto);
        } catch (Exception e) {
            log.error("数据同步记录入库异常：{}", e);
        }

        StrBuilder errorMessageBuilder = StrUtil.strBuilder();

        // 直接使用列表查询，执行分片。
        String mainSql = dto.getDynamicSql();

        if (StringUtils.isEmpty(mainSql)) {
            JSONObject esData = dynamicSqlUtil.getMappedData(dto);
            log.debug("根据映射关系的到的数据对象：{}", JSON.toJSONString(esData, SerializerFeature.WriteMapNullValue));
            if (PickerConstants.BATCH_FLAG.equals(dto.getBatch())){
                dto.setEventTypeEnum(DataEventTypeEnum.BATCH_DELETE);
            }

            elasticsearchUtil.doSyncByEventType(dto.getEventTypeEnum(), dto.getIndexName(), esData);
        } else {
            mainSql = dynamicSqlUtil.getDynamicSqlPage(dto, pageSize);

            Integer startPageNo = 0;
            int mainDataResultSize;

            List<Future<Object>> futureResults = new ArrayList<>();
            // 只要查询结果数量等于分页大小，说明后面可能还有数据，继续分片。
            do {
                mainDataResultSize = 0;
                List<Map<String, Object>> mainDataList = dynamicSqlUtil.getDataPage(mainSql, startPageNo, pageSize);
                if (CollectionUtil.isNotEmpty(mainDataList)) {
                    mainDataResultSize = mainDataList.size();
                    if (0 < mainDataResultSize) {
                        // 分片任务
                        PartitionTask partitionTask = PartitionTask.builder()
                                .recordId(recordId)
                                .dynamicSqlUtil(dynamicSqlUtil)
                                .elasticsearchUtil(elasticsearchUtil)
                                .dto(dto)
                                // 页码在这里自增，页码从0开始的，这里加了刚好。
                                .pageNo(startPageNo++)
                                .mainDataList(mainDataList).build();
                        // 提交分片任务
                        Future<Object> future = partitionRequestPool.submit(partitionTask);
                        futureResults.add(future);
                    }
                }
            } while (pageSize == mainDataResultSize);

            // 获取分片执行结果
            List<PartitionResult> resultList = new ArrayList<>(futureResults.size());
            for (Future<Object> future : futureResults) {
                try {
                    resultList.add((PartitionResult) future.get(5, TimeUnit.MINUTES));
                } catch (Exception  e) {
                    log.error("获取分片任务结果异常：{}", e.getMessage());
                    e.printStackTrace();
                    errorMessageBuilder.append("获取分片任务结果异常：{}" + e.getMessage());
                }
            }
            resultList = resultList.stream().filter(item -> ResultCodeEnum.FAIL.getCode().equals(item.getCode())).collect(Collectors.toList());
            resultList.forEach(item -> errorMessageBuilder.append(item.getMessage()).append(StrUtil.COMMA));
        }

        return updateRecordAndBuildResult(recordId, errorMessageBuilder.toString());
    }

    /**
     * 描述 更新记录并构建出参
     * @param errMessage
     * @return void
     * @author liugs
     * @date 2023/2/6 15:15
     */
    private PickerResult updateRecordAndBuildResult(Long recordId, String errMessage) {
        PickerResult result = new PickerResult(ResultCodeEnum.SUCCESS);
        // 更新同步信息
        try {
            if (StringUtils.isEmpty(errMessage)) {
                updateRecord(recordId, ConstantEnum.PickerRecordStatusEnum.SUCCESS, ConstantEnum.PickerRecordStatusEnum.SUCCESS.getDesc());
            } else {
                updateRecord(recordId, ConstantEnum.PickerRecordStatusEnum.FAIL, errMessage);
            }
        } catch (Exception e) {
            log.error("数据同步记录状态更新异常：{}", e);
        }
        if (StringUtils.isNotEmpty(errMessage)) {
            result.buildFail(errMessage);
        }
        log.info("==采集数据并同步ES完成：{}", JSON.toJSONString(result));
        return result;
    }

    /**
     * 处理同步记录
     * @param dto
     * @return recordId
     */
    private Long createSyncRecord(DataPickDTO dto) {
        Long id = Sequence.nextId();
        PickerRecordDTO recordPo = new PickerRecordDTO();
        BeanUtils.copyProperties(dto, recordPo);
        recordPo.setId(id);
        recordPo.setConditionParam(JSON.toJSONString(dto.getConditionParam()));
        recordPo.setEventType(dto.getEventTypeEnum().getEventType());
        recordPo.setStartTime(new DateTime().toString("yyyy-MM-dd HH:mm:ss"));
        recordPo.setStatusDesc(ConstantEnum.PickerRecordStatusEnum.INIT.getDesc());
        recordPo.setDynamicSql(StringUtils.isEmpty(dto.getDynamicSql()) ? null : Base64Encoder.encode(dto.getDynamicSql()));
        String dynamicSql = ConstantEnum.DynamicSqlEnum.INSERT.getDynamicSql(JSON.parseObject(JSON.toJSONString(recordPo), Map.class));
        log.debug("记录数据同步SQL：{}", dynamicSql);
        if (1 > jdbcTemplate.update(dynamicSql)) {
            log.error("记录数据同步记录失败");
            return null;
        }
        return id;
    }

    /**
     * 更新记录状态
     * @param recordId
     * @param statusEnum
     * @param desc
     * @return
     */
    private void updateRecord(Long recordId, ConstantEnum.PickerRecordStatusEnum statusEnum, String desc) {
        PickerRecordDTO recordPo = new PickerRecordDTO();
        recordPo.setId(recordId);
        recordPo.setStatus(statusEnum.getState());
        recordPo.setStatusDesc(desc);
        recordPo.setFinishTime(new DateTime().toString("yyyy-MM-dd HH:mm:ss"));
        String dynamicSql = ConstantEnum.DynamicSqlEnum.UPDATE.getDynamicSql(JSON.parseObject(JSON.toJSONString(recordPo), Map.class));
        log.debug("记录数据同步SQL：{}", dynamicSql);
        if ( 1 > jdbcTemplate.update(dynamicSql)) {
            log.error("更新数据同步记录失败");
        }
    }
}
