package com.tydic.picker.task;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.tydic.picker.constant.PickerConstants;
import com.tydic.picker.dto.DataPickDTO;
import com.tydic.picker.enums.ResultCodeEnum;
import com.tydic.picker.utils.DynamicSqlUtil;
import com.tydic.picker.utils.ElasticsearchUtil;
import lombok.Builder;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.stream.Collectors;

/**
 * @ClassName PartitionTask
 * @Description 分片任务
 * @Author liugs
 * @Date 2023/1/31 11:12
 */
@Slf4j
@Builder
public class PartitionTask implements Callable<Object> {

    /** 记录ID */
    private Long recordId;

    /** DataPickDTO */
    private DataPickDTO dto;

    /** ES 客户端*/
    private ElasticsearchUtil elasticsearchUtil;

    /** 动态 */
    private DynamicSqlUtil dynamicSqlUtil;

    /** 主数据查询结果列表 */
    private List<Map<String, Object>> mainDataList;

    /** 当前分片查询页 */
    private Integer pageNo;


    @Override
    public Object call() {
        PartitionResult result = new PartitionResult();
        result.setCode(ResultCodeEnum.FAIL.getCode());
        try {
            if (PickerConstants.BATCH_FLAG.equals(dto.getBatch())) {

                String docKey = getDocKey();

                List<Object> docIdList = mainDataList.stream().map(item -> item.get(docKey)).collect(Collectors.toList());
                // 根据配置完成字段映射
                // 映射结果
                // 字段映射
                JSONObject mappingRelJson = JSON.parseObject(dto.getMappingInfo());
                JSONObject mappedDataJson = new JSONObject(true);
                Map<String, Object> objMap = mainDataList.get(0);
                for (Map.Entry<String, Object> entry : mappingRelJson.entrySet()) {
                    mappedDataJson.put(entry.getValue().toString(), objMap.get(entry.getKey()));
                }

                mappedDataJson.put(PickerConstants.DOCUMENT_ID, docIdList);

                elasticsearchUtil.doSyncByEventType(dto.getEventTypeEnum(), dto.getIndexName(), mappedDataJson);
            } else {
                for (Map<String, Object> mainData : mainDataList) {
                    // 执行动态SQL并根据映射配置填充ES对象
                    JSONObject esData = dynamicSqlUtil.getMappedDataForBatch(dto, mainData);
                    log.debug("根据映射关系的到的数据对象：{}", JSON.toJSONString(esData, SerializerFeature.WriteMapNullValue));

                    if (ObjectUtil.isNotEmpty(esData.get(PickerConstants.DOCUMENT_ID))) {
                        elasticsearchUtil.doSyncByEventType(dto.getEventTypeEnum(), dto.getIndexName(), esData);
                    } else {
                        result.setMessage(String.format("根据配置的文档ID取值字段[%s]未获取到文档ID", dto.getDocIdField()));
                    }
                }
            }
        } catch (Exception e) {
            log.error("同步ES数据异常：");
            e.printStackTrace();
            result.setMessage(String.format("同步任务ID【%d】，第【%d】分片同步发生异常：【%s】", recordId, pageNo, e.getMessage()));
            return result;
        }

        result.setCode(ResultCodeEnum.SUCCESS.getCode());
        return result;
    }

    private String getDocKey(){
        String docIdField = dto.getDocIdField();
        String docKey = "";
        JSONObject jsonObject = JSON.parseObject(dto.getMappingInfo());
        Set<Map.Entry<String, Object>> entries = jsonObject.entrySet();
        for (Map.Entry<String, Object> entry : entries){
            if (entry.getValue().equals(docIdField)){
                docKey = entry.getKey();
                break;
            }
        }
        return docKey;
    }
}
