package com.tydic.picker.utils;

import cn.hutool.core.text.StrBuilder;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.tydic.picker.constant.PickerConstants;
import com.tydic.picker.dto.DataPickDTO;
import com.tydic.picker.dto.DataSyncConfigSubQueryDTO;
import com.tydic.picker.enums.ConstantEnum;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * @ClassName DynamicUtil
 * @Description
 * @Author liugs
 * @Date 2022/8/25 11:52
 */
@Slf4j
@Component
public class DynamicSqlUtil {
    /** 最大深度 */
    private static final int MAXIMUM_DEPTH = 2;
    /** 占位符匹配正则 */
    public static final Pattern PLACE_PATTERN = Pattern.compile("(\\$\\{)([\\w]+)(})");
    /** 表名字段 */
    public static final String TABLE_NAME = "tableName";

    private JdbcTemplate jdbcTemplate;

    public DynamicSqlUtil(JdbcTemplate jdbcTemplate) {
        this.jdbcTemplate = jdbcTemplate;
    }

    /**
     * 描述 获取数据并做映射
     * @param dto
     * @return com.alibaba.fastjson.JSONObject
     * @author liugs
     * @date 2022/8/17 11:38
     */
    public JSONObject getMappedData(DataPickDTO dto) {
        // 字段映射
        JSONObject mappingRelJson = JSON.parseObject(dto.getMappingInfo());
        // 映射结果
        JSONObject mappedDataJson = new JSONObject(true);
        // 出入参参数合集
        Map<String, Object> conditionParam = dto.getConditionParam();
        // 直接使用出入参合集的参数、删除操作的条件
        if (StringUtils.isEmpty(dto.getDynamicSql())) {
            for (Map.Entry<String, Object> entry : conditionParam.entrySet()) {
                mappedDataJson.put(entry.getValue().toString(), conditionParam.get(entry.getKey()));
            }
            // 根据配置的文档ID字段取值文档ID
            mappedDataJson.put(PickerConstants.DOCUMENT_ID, conditionParam.get(dto.getDocIdField()));
            return mappedDataJson;
        }
        // 新增、更新，查询数据库
        String dynamicSql = DynamicSqlUtil.buildDynamicSql(dto.getDynamicSql(), conditionParam);
        log.info("顶层数据执行的取值动态语句为：{}", dynamicSql);

        Map<String, Object> dynamicSqlResult =  buildSingleData(mappedDataJson, mappingRelJson, dynamicSql, conditionParam);

        // 构建子查询
        buildSubField(mappedDataJson, dynamicSqlResult, conditionParam, dto.getSubQuery(), 0);
        mappedDataJson.put(PickerConstants.DOCUMENT_ID, mappedDataJson.get(dto.getDocIdField()));
        return mappedDataJson;
    }

    /**
     * 描述 获取数据并做映射
     * @param dto
     * @param resultDataMap
     * @return com.alibaba.fastjson.JSONObject
     * @author liugs
     * @date 2023/2/1 16:11
     */
    public JSONObject getMappedDataForBatch(DataPickDTO dto, Map<String, Object> resultDataMap) {
        // 字段映射
        JSONObject mappingRelJson = JSON.parseObject(dto.getMappingInfo());
        // 映射结果
        JSONObject mappedDataJson = new JSONObject(true);
        // 出入参参数合集
        Map<String, Object> conditionParam = dto.getConditionParam();
        // 直接使用出入参合集的参数、删除操作的条件
        if (StringUtils.isEmpty(dto.getDynamicSql())) {
            for (Map.Entry<String, Object> entry : conditionParam.entrySet()) {
                mappedDataJson.put(entry.getValue().toString(), conditionParam.get(entry.getKey()));
            }
            // 根据配置的文档ID字段取值文档ID
            mappedDataJson.put(PickerConstants.DOCUMENT_ID, conditionParam.get(dto.getDocIdField()));
            return mappedDataJson;
        }

        Map<String, Object> tempDataMap = new HashMap<>(16);
        tempDataMap.putAll(conditionParam);
        tempDataMap.putAll(resultDataMap);

        // 根据配置完成字段映射
        for (Map.Entry<String, Object> entry : mappingRelJson.entrySet()) {
            mappedDataJson.put(entry.getValue().toString(), tempDataMap.get(entry.getKey()));
        }

        // 构建子查询
        buildSubField(mappedDataJson, resultDataMap, conditionParam, dto.getSubQuery(), 0);
        mappedDataJson.put(PickerConstants.DOCUMENT_ID, mappedDataJson.get(dto.getDocIdField()));
        return mappedDataJson;
    }

    /**
     * 描述 查询指定页码的数据
     * @param dynamicSql
     * @param pageNo
     * @param pageSize
     * @return java.util.List<java.util.Map<java.lang.String,java.lang.Object>>
     * @author liugs
     * @date 2023/2/1 15:25
     */
    public List<Map<String, Object>> getDataPage(String dynamicSql, Integer pageNo, Integer pageSize) {
        Integer pageStart = pageNo * pageSize;
        dynamicSql = StrUtil.replace(dynamicSql, PickerConstants.PICKER_PAGE_START, pageStart.toString());
        if (log.isDebugEnabled()) {
            log.debug("动态查询语句为：{}", dynamicSql);
        }
        return executeDynamicSqlForList(dynamicSql);
    }

    /**
     * 描述 构建子查询
     * @param parentData            父级数据对象
     * @param parentQryResult       父级查询结果
     * @param conditionParam        出入参MAP
     * @param subQuery              子查询列表
     * @param depth                 子查询纵向深度
     * @author liugs
     * @date 2022/8/31 15:41
     */
    private void buildSubField(JSONObject parentData, Map<String, Object> parentQryResult, Map<String, Object> conditionParam,
                               List<DataSyncConfigSubQueryDTO> subQuery, int depth) {
        // 超过最大深度，子查询为空，直接返回
        if (MAXIMUM_DEPTH < depth || CollectionUtils.isEmpty(subQuery)) {
            return;
        }
        Map<String, Object> dynamicConditionMap = new HashMap<>(16);
        dynamicConditionMap.putAll(conditionParam);
        dynamicConditionMap.putAll(parentQryResult);
        // 字段映射
        JSONObject fieldMapping;
        // 构建动态查询语句
        String dynamicSql;
        // 遍历子查询列表
        for (DataSyncConfigSubQueryDTO subQueryItem : subQuery) {
            fieldMapping = JSON.parseObject(subQueryItem.getMappingInfo());
            dynamicSql = DynamicSqlUtil.buildDynamicSql(subQueryItem.getDataFilter(), dynamicConditionMap);

            ConstantEnum.DataType typeEnum = ConstantEnum.DataType.getType(subQueryItem.getDataType());

            switch (typeEnum) {
                case FIELD:
                    log.info("[{}]级字段类型子查询取值动态语句为：{}", depth, dynamicSql);
                    buildFieldData(parentData, fieldMapping, conditionParam, dynamicSql);
                    break;
                case  EXTEND_FIELD:
                    log.info("[{}]级扩展字段子查询取值动态语句为：{}", depth, dynamicSql);
                    buildExtFieldData(parentData, fieldMapping, dynamicSql);
                    break;
                case MULTIPLE:
                    log.info("[{}]级列表子查询取值动态语句为：{}", depth, dynamicSql);
                    JSONArray multipleData = buildMultipleData(fieldMapping, dynamicSql, conditionParam, subQueryItem.getSubQuery(), ++ depth);
                    -- depth;
                    parentData.put(subQueryItem.getParentField(), multipleData);
                    break;
                case MULTIPLE_SIMPLE:
                    log.info("[{}]级简单对象列表子查询取值动态语句为：{}", depth, dynamicSql);
                    parentData.put(subQueryItem.getParentField(), buildMultipleSimple(fieldMapping, dynamicSql, conditionParam));
                    break;
                case SINGLE:
                    // 本层级查询是单一对象
                    log.info("[{}]级单一对象子查询取值动态语句为：{}", depth, dynamicSql);
                    JSONObject subData = new JSONObject();
                    // 构建单一数据，并将SQL结果返回，作为子查询的条件参数，如果查询异常，将其值字段值置为null;
                    Map<String, Object> dynamicSqlResult;
                    try {
                        dynamicSqlResult = buildSingleData(subData, fieldMapping, dynamicSql, conditionParam);
                    } catch (DataAccessException e) {
                        log.error("[{}]级子执行单一对象型取值动态语句异常：{}", depth, e.getMessage());
                        parentData.put(subQueryItem.getParentField(), subData);
                        continue;
                    }
                    if (CollectionUtils.isNotEmpty(subQueryItem.getSubQuery())) {
                        buildSubField(subData, dynamicSqlResult, conditionParam, subQueryItem.getSubQuery(), ++ depth);
                        -- depth;
                    }
                    parentData.put(subQueryItem.getParentField(), subData);
                    break;
                default:
                    break;
            }
        }
    }

    /**
     * 描述 构建字段型数据
     * @param parentData        父级数据对象
     * @param fieldMapping      映射关系
     * @param conditionParam    出入参合集
     * @param dynamicSql        动态语句
     * @author liugs
     * @date 2022/9/19 15:34
     */
    private void buildFieldData(JSONObject parentData, JSONObject fieldMapping, Map<String, Object> conditionParam, String dynamicSql) {
        Map<String, Object> resultDataMap;
        try {
            resultDataMap = jdbcTemplate.queryForMap(dynamicSql);
        } catch (DataAccessException e) {
            log.error("执行动态子查询语句获取字段类型数据异常：{}", e.getMessage());
            return;
        }
        log.debug("执行动态子查询语句获取到的数据：{}", JSON.toJSONString(resultDataMap));

        resultDataMap.putAll(conditionParam);
        // 根据配置完成字段映射
        for (Map.Entry<String, Object> entry : fieldMapping.entrySet()) {
            parentData.put(entry.getValue().toString(), resultDataMap.get(entry.getKey()));
        }
    }

    /**
     * 描述 构建扩展字段数据
     * @param parentData        父级数据对象
     * @param fieldMapping      映射关系
     * @param dynamicSql        动态语句
     * @return void
     * @author liugs
     * @date 2022/9/16 10:45
     */
    private void buildExtFieldData(JSONObject parentData, JSONObject fieldMapping, String dynamicSql) {
        // 执行查询语句
        List<Map<String, Object>> queryForList = executeDynamicSqlForList(dynamicSql) ;
        if (CollectionUtils.isEmpty(queryForList)) {
            log.error("执行动态子查询语句[{}]未获取到数据", dynamicSql);
            return;
        }

        log.debug("执行动态子查询语句获取到的数据：{}", JSON.toJSONString(queryForList));
        // 遍历查询结果，根据配置完成字段映射
        queryForList.forEach(item -> {
            for (Map.Entry<String, Object> entry : fieldMapping.entrySet()) {
                Object esKey = item.get(entry.getKey());
                if (ObjectUtil.isNotEmpty(esKey)) {
                    parentData.put(StrUtil.toCamelCase(esKey.toString()), item.get(entry.getValue()));
                }
            }
        });
    }

    /**
     * 描述 构建简单java对象列表
     * @param fieldMapping      映射关系
     * @param dynamicSql        动态语句
     * @param conditionParam    出入参合集
     * @return java.lang.Object
     * @author liugs
     * @date 2022/9/19 16:13
     */
    private Object buildMultipleSimple(JSONObject fieldMapping, String dynamicSql, Map<String, Object> conditionParam) {
        Set<Object> subArrayData = new HashSet<>(16);
        // 查询数据库
        List<Map<String, Object>> queryForList = executeDynamicSqlForList(dynamicSql);
        if (CollectionUtils.isEmpty(queryForList)) {
            log.error("执行动态子查询语句[{}]未获取到数据", dynamicSql);
            return subArrayData;
        }
        // 映射map只有一条映射关系，直接用流
        for (Map.Entry<String, Object> entry : fieldMapping.entrySet()) {
            subArrayData = queryForList.stream().map(item -> item.get(entry.getKey())).collect(Collectors.toSet());
            if (CollectionUtils.isNotEmpty(subArrayData)) {
                break;
            }
        }

        return subArrayData.toArray();
    }

    /**
     * 描述 构建多对象类型
     * @param fieldMapping          映射关系
     * @param dynamicSql            动态语句
     * @param conditionParam        出入参合集
     * @param subQuery              子查询配置
     * @param depth                 深度
     * @return com.alibaba.fastjson.JSONArray
     * @author liugs
     * @date 2022/9/19 15:46
     */
    private JSONArray buildMultipleData(JSONObject fieldMapping, String dynamicSql, Map<String, Object> conditionParam,
                                        List<DataSyncConfigSubQueryDTO> subQuery, int depth) {
        JSONArray subArrayData = new JSONArray();
        // 查询数据库
        List<Map<String, Object>> queryForList = executeDynamicSqlForList(dynamicSql);
        if (CollectionUtils.isEmpty(queryForList)) {
            log.error("执行动态子查询语句[{}]未获取到数据", dynamicSql);
            return subArrayData;
        }
        // 遍历查询结果
        queryForList.forEach(item -> {
            item.putAll(conditionParam);
            // 根据映射关系构建数据对象
            JSONObject subDateItem = new JSONObject();
            for (Map.Entry<String, Object> entry : fieldMapping.entrySet()) {
                subDateItem.put(entry.getValue().toString(), item.get(entry.getKey()));
            }
            // 如果子查询不为空，查询。
            if (CollectionUtils.isNotEmpty(subQuery)) {
                buildSubField(subDateItem, item, conditionParam, subQuery, depth);
            }
            subArrayData.add(subDateItem);
        });
        return subArrayData;
    }

    /**
     * 描述 构建单对象类型
     * @param targetData        目标对象
     * @param fieldMapping      映射关系
     * @param dynamicSql        动态语句
     * @param conditionParam    出入参合集
     * @return java.util.Map<java.lang.String,java.lang.Object>
     * @author liugs
     * @date 2022/9/19 15:47
     */
    private Map<String, Object> buildSingleData(JSONObject targetData, JSONObject fieldMapping, String dynamicSql, Map<String, Object> conditionParam) {
        // 查询单一数据
        Map<String, Object> resultDataMap = jdbcTemplate.queryForMap(dynamicSql);
        log.debug("执行动态语句获取到的数据：{}", JSON.toJSONString(resultDataMap));
        Map<String, Object> tempDataMap = new HashMap<>(16);
        tempDataMap.putAll(conditionParam);
        tempDataMap.putAll(resultDataMap);

        // 根据配置完成字段映射
        for (Map.Entry<String, Object> entry : fieldMapping.entrySet()) {
            targetData.put(entry.getValue().toString(), tempDataMap.get(entry.getKey()));
        }
        return resultDataMap;
    }

    /**
     * 执行列表查询语句
     * @param dynamicSql
     * @return
     */
    private List<Map<String, Object>> executeDynamicSqlForList(String dynamicSql) {
        List<Map<String, Object>> queryForList;
        try {
            queryForList = jdbcTemplate.queryForList(dynamicSql);
        } catch (Exception e) {
            log.error("执行动态查询语句列表类型数据异常：{}", e.getMessage());
            return null;
        }
        return queryForList;
    }

    /**
     * 描述 构建动态 Sql
     * @param dynamicSql
     * @param paramMap
     * @return java.lang.String
     * @author liugs
     * @date 2022/8/25 11:54
     */
    private static String buildDynamicSql(String dynamicSql, Map<String, Object> paramMap) {
        StringBuffer stringBuffer = new StringBuffer();
        Matcher matcher = PLACE_PATTERN.matcher(dynamicSql);
        while (matcher.find()) {
            String group = matcher.group(2);
            if (paramMap.get(group) == null) {
                matcher.appendReplacement(stringBuffer, "null");
                continue;
            }
            if (TABLE_NAME.equals(group)) {
                matcher.appendReplacement(stringBuffer, paramMap.get(group).toString());
            } else {
                matcher.appendReplacement(stringBuffer, "'" + paramMap.get(group).toString() + "'");
            }
        }
        matcher.appendTail(stringBuffer);

        return stringBuffer.toString().trim();
    }

    /**
     * 描述 获取可分页的动态SQL
     * @param dto
     * @return java.lang.String
     * @author liugs
     * @date 2023/2/1 15:00
     */
    public String getDynamicSqlPage(DataPickDTO dto, Integer pageSize) {
        String dynamicSql = buildDynamicSql(dto.getDynamicSql(), dto.getConditionParam());
        dynamicSql = StrUtil.removeAll(dynamicSql, PickerConstants.SEMICOLON);
        dynamicSql = StrBuilder.create(dynamicSql).append(StrUtil.SPACE)
                .append(PickerConstants.LIMIT).append(StrUtil.SPACE).append(pageSize).append(StrUtil.SPACE)
                .append(PickerConstants.OFFSET).append(StrUtil.SPACE).append(PickerConstants.PICKER_PAGE_START)
                .append(PickerConstants.SEMICOLON).toString();
        if (log.isDebugEnabled()) {
            log.debug("拼接好的分页SQL：{}", dynamicSql);
        }
        return dynamicSql;
    }
}
