package com.tydic.picker.utils;


import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.tydic.dyc.base.constants.BaseRspConstant;
import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.picker.constant.PickerConstants;
import com.tydic.picker.enums.DataEventTypeEnum;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.util.EntityUtils;
import org.elasticsearch.action.admin.indices.create.CreateIndexRequest;
import org.elasticsearch.action.admin.indices.create.CreateIndexResponse;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequest;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.action.support.replication.ReplicationResponse;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.client.*;
import org.elasticsearch.client.core.GetSourceRequest;
import org.elasticsearch.client.core.GetSourceResponse;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.query.TermQueryBuilder;
import org.elasticsearch.index.reindex.BulkByScrollResponse;
import org.elasticsearch.index.reindex.DeleteByQueryRequest;
import org.elasticsearch.xcontent.XContentType;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;


/**
 * 标题：es获取客户端工具类
 * <p>
 * 说明：es获取客户端工具类
 * <br>
 * 时间：2019/1/1616:52<br>
 *
 *
 * </p>
 *
 * @author hegy
 */
@Slf4j
public class ElasticsearchUtil {

    private RestHighLevelClient restHighLevelClient;

    public ElasticsearchUtil(RestHighLevelClient restHighLevelClient) {
        this.restHighLevelClient = restHighLevelClient;
    }

    public void doSyncByEventType(DataEventTypeEnum eventTypeEnum, String indexName, JSONObject esData) {
        switch (eventTypeEnum) {
            case CREATE:
                addDocument(indexName, esData.remove(PickerConstants.DOCUMENT_ID).toString(), esData);
                break;
            case UPDATE:
                updateDoc(indexName, esData.remove(PickerConstants.DOCUMENT_ID).toString(), esData);
                break;
            case DELETE:
                deleteDocById(indexName, esData.getString(PickerConstants.DOCUMENT_ID));
                break;
            case DELETE_BY_QUERY:
                // 移除文档ID
                esData.remove(PickerConstants.DOCUMENT_ID);
                deleteDocByQuery(indexName, esData);
            default:
                break;
        }
    }

    /**
     * 删除索引
     *
     * @param index 索引
     * @return boolean
     */
    public boolean deleteIndex(String index) {
        if (StringUtils.isBlank(index)) {
            log.info("删除索引内容为空");
            return false;
        }
        DeleteIndexRequest request = new DeleteIndexRequest(index);
        try {
//            DeleteIndexResponse indexResponse = restHighLevelClient.indices().delete(request, RequestOptions.DEFAULT); // 6.x jar包
            // 升级7.10 jar包
            AcknowledgedResponse indexResponse = restHighLevelClient.indices().delete(request, RequestOptions.DEFAULT);
            if (indexResponse.isAcknowledged()) {
                log.info("删除索引成功");
            } else {
                log.info("删除索引失败");
            }
            return indexResponse.isAcknowledged();
        } catch (Exception e) {
            log.error("删除索引失败", e);
            String exceptionStr = JSON.toJSONString(e.getMessage());
            if (exceptionStr.contains("index_not_found_exception")) {
                return true;
            }
        }
        return false;
    }


    /**
     * 创建索引
     *
     * @param index 索引
     * @return boolean
     */
    public boolean createIndex(String index) {
        //index名必须全小写，否则报错
        CreateIndexRequest request = new CreateIndexRequest(index);
        try {
            CreateIndexResponse indexResponse = restHighLevelClient.indices().create(request, RequestOptions.DEFAULT);
            if (indexResponse.isAcknowledged()) {
                log.info("创建索引成功");
            } else {
                log.info("创建索引失败");
            }
            return indexResponse.isAcknowledged();
        } catch (IOException e) {
            log.error("创建索引失败", e);
        }
        return false;
    }

    /**
     * 插入数据
     *
     * @param index  索引
     * @param type   类型
     * @param object 数据
     * @return String
     */
    public String addData(String index, String type, String id, JSONObject object) {
        // 注意：在Elasticsearch 7.x及以上版本中，type参数已经被废弃
        // 索引操作通常不再需要type

        // 假设restClientBuilder已经配置好
        try {
            // 构建JSON字符串
            String jsonString = JSON.toJSONString(object);

            // 创建请求
            Request request = new Request("PUT", "/" + index + "/_doc/" + (id != null ? id : ""));
            request.setJsonEntity(jsonString);

            // 发送请求
            Response response = restHighLevelClient.getLowLevelClient().performRequest(request);

            // 处理响应
            if (response.getStatusLine().getStatusCode() == 200 || response.getStatusLine().getStatusCode() == 201) {
                // 索引成功，返回ID（注意：如果指定了ID，则直接返回该ID；否则Elasticsearch会生成一个）
                String responseBody = EntityUtils.toString(response.getEntity());
                // 解析responseBody以获取返回的ID（实际取决于Elasticsearch的版本和响应格式）
                // 这里只是一个示例，你可能需要根据你的具体响应来调整
                // 注意：在Elasticsearch 7.x及以后，直接通过URL的ID部分即可知道索引的ID
                return id != null ? id : responseBody; // 假设responseBody包含ID，这取决于你的Elasticsearch版本和配置
            } else {
                log.info("插入数据失败，状态码: " + response.getStatusLine().getStatusCode());
            }
        } catch (IOException e) {
            log.info("插入数据失败", e);
        }
        return null;
    }

    public Response addData(String index, String id, JSONObject object) {
        // 注意：在Elasticsearch 7.x及以上版本中，type参数已经被废弃
        // 索引操作通常不再需要type

        // 假设restClientBuilder已经配置好
        try {
            // 构建JSON字符串
            String jsonString = JSON.toJSONString(object);

            // 创建请求
            Request request = new Request("PUT", "/" + index + "/_doc/" + (id != null ? id : ""));
            request.setJsonEntity(jsonString);

            // 发送请求
            Response response = restHighLevelClient.getLowLevelClient().performRequest(request);

            return response;
        } catch (IOException e) {
            log.info("插入数据失败", e);
        }
        return null;
    }

    /**
     * 描述:保存json数据
     *
     * @param index   索引名称
     * @param id      索引数据id
     * @param jsonStr json数据
     * @return java.lang.String
     * @author tgy
     * @date 2022/3/1 13:00
     */
    public String addJsonData(String index, String type, String id, String jsonStr) {
        // 注意：在Elasticsearch 7.x及以上版本中，type参数已经被废弃
        // 索引操作通常不再需要type

        // 假设restClientBuilder已经配置好
        try {
            // 创建请求
            Request request = new Request("PUT", "/" + index + "/_doc/" + (id != null ? id : ""));
            request.setJsonEntity(jsonStr);

            // 发送请求
            Response response = restHighLevelClient.getLowLevelClient().performRequest(request);

            // 处理响应
            if (response.getStatusLine().getStatusCode() == 200 || response.getStatusLine().getStatusCode() == 201) {
                // 索引成功，返回ID（注意：如果指定了ID，则直接返回该ID；否则Elasticsearch会生成一个）
                String responseBody = EntityUtils.toString(response.getEntity());
                // 解析responseBody以获取返回的ID（实际取决于Elasticsearch的版本和响应格式）
                // 这里只是一个示例，你可能需要根据你的具体响应来调整
                // 注意：在Elasticsearch 7.x及以后，直接通过URL的ID部分即可知道索引的ID
                return id != null ? id : responseBody; // 假设responseBody包含ID，这取决于你的Elasticsearch版本和配置
            } else {
                log.info("插入数据失败，状态码: " + response.getStatusLine().getStatusCode());
            }
        } catch (IOException e) {
            log.info("插入数据失败", e);
        }
        return null;
    }

    /**
     * 检查索引
     *
     * @param index 索引
     * @return boolean
     */
    public boolean checkIndexExist(String index) {
        String ok = "OK";
        boolean checkResult = false;
        try {
            Request request = new Request("HEAD", index);
            Response response = restHighLevelClient.getLowLevelClient().performRequest(request);
            if (ok.equals(response.getStatusLine().getReasonPhrase())) {
                checkResult = true;
            }
        } catch (IOException e) {
            log.error("检查索引失败", e);
        }
        return checkResult;
    }

    /**
     * 根据条件删除ES数据
     *
     * @param endPoint   接口index点
     * @param qryJsonStr 删除数据语句
     * @return boolean
     */
    public Boolean deleteDataByCondition(String endPoint, String qryJsonStr) {
        boolean deleteResult = false;
        HttpEntity entity = new NStringEntity(qryJsonStr, ContentType.APPLICATION_JSON);
        try {
            Request request = new Request("POST", endPoint);
            request.setEntity(entity);
            Response response = restHighLevelClient.getLowLevelClient().performRequest(request);
            String result = EntityUtils.toString(response.getEntity());
            if (StringUtils.isNotBlank(result)) {
                JSONObject deleteResultJson = JSONObject.parseObject(result);
                int deleteCount = deleteResultJson.getInteger("deleted");
                if (deleteCount > 0) {
                    deleteResult = true;
                }
            }
        } catch (IOException e) {
            log.error("根据条件删除ES数据失败", e);
        }
        return deleteResult;
    }

    public Boolean updateDataById(String endPoint, String id, JSONObject updateJson) {
        boolean updateREsult = false;
        //组装更新语句
        String esUpdateStr = toEsUpdateStr(id, updateJson);
        log.info("esUpdate URL:{}", endPoint);
        log.info("esUpdateStrReq:{}", esUpdateStr);
        HttpEntity entity = new NStringEntity(esUpdateStr, ContentType.APPLICATION_JSON);
        try {
            Request request = new Request("POST", endPoint);
            request.setEntity(entity);
            Response response = restHighLevelClient.getLowLevelClient().performRequest(request);
            String result = EntityUtils.toString(response.getEntity());
            log.info("esUpdateStrRsp:{}", result);
            if (StringUtils.isNotBlank(result)) {
                JSONObject rspJsonObject = JSON.parseObject(result);
                if ("1".equals(rspJsonObject.getString("updated"))) {
                    updateREsult = true;
                }
            }
        } catch (IOException e) {
            log.error("根据条件更新ES数据失败:{},更新语句:{}", e, esUpdateStr);
        }
        return updateREsult;
    }

    /**
     * 获取低水平客户端
     *
     * @return RestClient
     */
    public RestClient getLowLevelClient() {
        return restHighLevelClient.getLowLevelClient();

    }

    /**
     * 组装更新es语句
     *
     * @param id           主键ID
     * @param updateParams 更新的字段
     * @return 更新es语句
     */
    public String toEsUpdateStr(String id, JSONObject updateParams) {
        //组装查询条件
        JSONObject term = new JSONObject();
        term.put("_id", id);

        JSONObject query = new JSONObject();
        query.put("term", term);

        JSONObject script = new JSONObject();
        script.put("source", getSource(updateParams));
        script.put("lang", "painless");
        script.put("params", updateParams);

        JSONObject esJsonObj = new JSONObject();
        esJsonObj.put("query", query);
        esJsonObj.put("script", script);

        return esJsonObj.toJSONString();
    }


    /**
     * 组装source映射字段
     *
     * @param updateParams 更新的字段
     * @return source值
     */
    private String getSource(JSONObject updateParams) {
        StringBuilder stringBuilder = new StringBuilder();
        updateParams.keySet().forEach(key -> {
            stringBuilder.append("ctx._source.")
                    .append(key)
                    .append("=")
                    .append("params.")
                    .append(key)
                    .append(";");
        });
        return stringBuilder.toString();
    }

    /**
     * 描述 新增文章
     *
     * @param indexName 索引名称
     * @param docId     文章ID
     * @param source    文章源数据
     * @return java.lang.String
     * @author liugs
     * @date 2021/3/22 14:44:29
     */
    public String addDocument(String indexName, String docId, JSONObject source) {
        log.debug("新增文章入参：indexName='{}'，id='{}'，source='{}'", indexName, docId, source);
        if (StringUtils.isEmpty(indexName) || source.isEmpty()) {
            log.info("新增文章，[indexName]，[source]不能为空");
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "新增文章，[indexName]，[source]不能为空");
        }

        IndexRequest request = new IndexRequest(indexName);
        if (!StringUtils.isEmpty(docId)) {
            request.id(docId);
        }
        request.source(JSON.toJSONString(source), XContentType.JSON);
        try {
            String id = this.addData(indexName, null, docId, source);
            if (StringUtils.isNotEmpty(id)) {
                log.info("新增文章成功");
                return id;
            } else {
                String failReason = "ES调用失败";
                log.info("新增文章失败：{}", failReason);
                throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "新增文章失败：" + failReason);
            }
        } catch (Exception e) {
            log.error("新增文章异常：" + e);
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "新增文章异常：" + e.getMessage());
        }
    }

    /**
     * 描述 更新文章
     *
     * @param indexName 索引名称
     * @param docId     文章ID
     * @param source    文章数据
     * @return java.lang.Boolean
     * @author liugs
     * @date 2021/3/24 9:39:58
     */
    public void updateDoc(String indexName, String docId, JSONObject source) {
        if (log.isDebugEnabled()) {
            log.debug("根据文章ID更新文章，入参：indexName = '{}', docId = '{}'", indexName, docId);
        }
        UpdateRequest request = new UpdateRequest(indexName, docId);
        // 在版本冲突，更新失败时，重试3次。
        request.retryOnConflict(3);
        request.doc(JSON.toJSONString(source), XContentType.JSON);
        try {
            String id = this.addData(indexName, null, docId, source);
            if (StringUtils.isNotEmpty(id)) {
                log.info("文章更新成功");
                return;
            }
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "根据文章ID更新文章失败：" + id);
        } catch (Exception e) {
            log.error("根据文章ID更新文章异常：" + e);
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "根据文章ID更新文章异常：" + e.getMessage());
        }
    }

    /**
     * 描述 根据文章ID删除文章
     *
     * @param indexName 索引名称
     * @param docId     文章ID
     * @return java.lang.Boolean
     * @author liugs
     * @date 2021/3/22 16:58:57
     */
    public void deleteDocById(String indexName, String docId) {
        log.debug("根据文章ID删除文章，入参：indexName={}，docId={}", indexName, docId);
        if (StringUtils.isEmpty(indexName) || StringUtils.isEmpty(docId)) {
            log.info("根据文章ID删除文章，入参[indexName]和[docId]为空，执行结束！");
            return;
        }
        DeleteRequest request = new DeleteRequest(indexName, docId);
        try {
            DeleteResponse response = restHighLevelClient.delete(request, RequestOptions.DEFAULT);
            if (response.getShardInfo().getFailed() <= 0) {
                log.info("删除索引[{}]文章[{}]成功", indexName, docId);
                return;
            }

            ReplicationResponse.ShardInfo.Failure[] failures = response.getShardInfo().getFailures();
            log.info("删除文章失败：{}", failures[0].reason());
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "根据文章ID删除文章失败：" + failures[0].reason());
        } catch (IOException e) {
            log.error("根据文章ID删除文章异常：" + e);
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "根据文章ID删除文章异常：" + e.getMessage());
        }
    }

    /**
     * 描述 根据条件删除文章
     *
     * @param indexName 索引名称
     * @param condition 删除条件
     * @return void
     * @author liugs
     * @date 2023/11/29 0029 17:43
     */
    public void deleteDocByQuery(String indexName, JSONObject condition) {
        log.debug("根据条件删除文章，入参：indexName={}，condition={}", indexName, condition.toJSONString());
        if (StringUtils.isEmpty(indexName) || ObjectUtil.isEmpty(condition)) {
            log.info("根据条件删除文章，入参[indexName]和[condition]为空，执行结束！");
            return;
        }
        //通过QueryBuilders中的搜索逻辑
        BoolQueryBuilder queryBuilder = QueryBuilders.boolQuery();
        // 遍历构建条件
        condition.forEach((key, value) -> {
            if (ObjectUtil.isNotEmpty(value)) {
                TermQueryBuilder termQueryBuilder = QueryBuilders.termQuery(key, value);
                queryBuilder.must(termQueryBuilder);
            }
        });

        DeleteByQueryRequest deleteByQueryRequest = new DeleteByQueryRequest(indexName);
        deleteByQueryRequest.setQuery(queryBuilder);
        try {
            BulkByScrollResponse deleteResponse = restHighLevelClient.deleteByQuery(deleteByQueryRequest, RequestOptions.DEFAULT);
            if (1 > deleteResponse.getDeleted()) {
                log.info("根据条件{}删除索引{}文章成功，影响条数：{}", indexName, condition.toJSONString(), deleteResponse.getDeleted());
            }
        } catch (IOException e) {
            log.error("根据条件删除文章发生异常：" + e);
            e.printStackTrace();
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "根据条件删除文章发生异常：" + e.getMessage());
        }
    }

    /**
     * 描述 根据文章ID获取文章
     *
     * @param indexName 索引名称
     * @param docId     文章ID
     * @return com.alibaba.fastjson.JSONObject
     * @author liugs
     * @date 2021/3/25 11:39:24
     */
    public JSONObject getDocById(String indexName, String docId) {
        if (log.isDebugEnabled()) {
            log.debug("根据文章ID获取文章：入参：indexName='{}'，docId='{}'", indexName, docId);
        }
        GetSourceRequest request = new GetSourceRequest(indexName, docId);
        Map<String, Object> source = new HashMap<>(16);
        try {
            GetSourceResponse response = restHighLevelClient.getSource(request, RequestOptions.DEFAULT);
            source = response.getSource();
        } catch (IOException e) {
            log.error("根据文章ID获取文章异常：" + e);
        }
        if (source.isEmpty()) {
            log.info("根据入参：入参：indexName='{}'，docId='{}'，未查询到文章", indexName, docId);
        }

        return new JSONObject(source);
    }
}
