package com.tydic.picker.utils;


import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.tydic.dyc.base.constants.BaseRspConstant;
import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.picker.constant.PickerConstants;
import com.tydic.picker.enums.DataEventTypeEnum;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.util.EntityUtils;
import org.elasticsearch.action.DocWriteResponse;
import org.elasticsearch.action.admin.indices.create.CreateIndexRequest;
import org.elasticsearch.action.admin.indices.create.CreateIndexResponse;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequest;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.action.support.replication.ReplicationResponse;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.action.update.UpdateResponse;
import org.elasticsearch.client.*;
import org.elasticsearch.client.core.GetSourceRequest;
import org.elasticsearch.client.core.GetSourceResponse;
import org.elasticsearch.common.xcontent.XContentType;
import org.springframework.util.ObjectUtils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;


/**
 * 标题：es获取客户端工具类
 * <p>
 * 说明：es获取客户端工具类
 * <br>
 * 时间：2019/1/1616:52<br>
 *
 *
 * </p>
 *
 * @author hegy
 */
@Slf4j
public class ElasticsearchUtil {

    private RestHighLevelClient restHighLevelClient;

    public ElasticsearchUtil(RestHighLevelClient restHighLevelClient) {
        this.restHighLevelClient = restHighLevelClient;
    }

    public void doSyncByEventType(DataEventTypeEnum eventTypeEnum, String indexName, JSONObject esData) {
        switch (eventTypeEnum) {
            case CREATE:
                addDocument(indexName, esData.remove(PickerConstants.DOCUMENT_ID).toString(), esData);
                break;
            case UPDATE:
                updateDoc(indexName, esData.remove(PickerConstants.DOCUMENT_ID).toString(), esData);
                break;
            case BATCH_UPDATE:
                updateBatch(indexName, esData.remove(PickerConstants.DOCUMENT_ID), esData);
                break;
            case DELETE:
                deleteDocById(indexName, esData.getString(PickerConstants.DOCUMENT_ID));
                break;
            case BATCH_DELETE:
                deleteBatch(indexName, esData.get(PickerConstants.DOCUMENT_ID));
                break;
            default:
                break;
        }
    }

    /**
     * 删除索引
     *
     * @param index 索引
     * @return boolean
     */
    public boolean deleteIndex(String index) {
        if (StringUtils.isBlank(index)) {
            log.info("删除索引内容为空");
            return false;
        }
        DeleteIndexRequest request = new DeleteIndexRequest(index);
        try {
//            DeleteIndexResponse indexResponse = restHighLevelClient.indices().delete(request, RequestOptions.DEFAULT); // 6.x jar包
            // 升级7.10 jar包
            AcknowledgedResponse indexResponse = restHighLevelClient.indices().delete(request, RequestOptions.DEFAULT);
            if (indexResponse.isAcknowledged()) {
                log.info("删除索引成功");
            } else {
                log.info("删除索引失败");
            }
            return indexResponse.isAcknowledged();
        } catch (Exception e) {
            log.error("删除索引失败", e);
            String exceptionStr = JSON.toJSONString(e.getMessage());
            if (exceptionStr.contains("index_not_found_exception")) {
                return true;
            }
        }
        return false;
    }


    /**
     * 创建索引
     *
     * @param index 索引
     * @return boolean
     */
    public boolean createIndex(String index) {
        //index名必须全小写，否则报错
        CreateIndexRequest request = new CreateIndexRequest(index);
        try {
            CreateIndexResponse indexResponse = restHighLevelClient.indices().create(request, RequestOptions.DEFAULT);
            if (indexResponse.isAcknowledged()) {
                log.info("创建索引成功");
            } else {
                log.info("创建索引失败");
            }
            return indexResponse.isAcknowledged();
        } catch (IOException e) {
            log.error("创建索引失败", e);
        }
        return false;
    }

    /**
     * 插入数据
     *
     * @param index  索引
     * @param type   类型
     * @param object 数据
     * @return String
     */
    public String addData(String index, String type, String id, JSONObject object) {
//        IndexRequest indexRequest = new IndexRequest(index, type, id);// 6.x jar包
        IndexRequest indexRequest = new IndexRequest(index);
        indexRequest.id(id);
        try {
//            indexRequest.source(mapper.writeValueAsString(object), XContentType.JSON);
            indexRequest.source(JSON.toJSONString(object), XContentType.JSON);
            IndexResponse indexResponse = restHighLevelClient.index(indexRequest, RequestOptions.DEFAULT);
            return indexResponse.getId();
        } catch (Exception e) {
            log.info("插入数据失败", e);
        }
        return null;
    }

    /**
     * 描述:保存json数据
     *
     * @param index   索引名称
     * @param id      索引数据id
     * @param jsonStr json数据
     * @return java.lang.String
     * @author tgy
     * @date 2022/3/1 13:00
     */
    public String addJsonData(String index, String id, String jsonStr) {
        IndexRequest indexRequest = new IndexRequest(index);
        indexRequest.id(id);
        try {
            indexRequest.source(jsonStr, XContentType.JSON);
            log.info("addJsonData保存数据：" + jsonStr);
            IndexResponse indexResponse = restHighLevelClient.index(indexRequest, RequestOptions.DEFAULT);
            return indexResponse.getId();
        } catch (Exception e) {
            log.error("插入数据失败", e);
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "插入数据失败：" + e.getMessage(), e);
        }
    }

    /**
     * 检查索引
     *
     * @param index 索引
     * @return boolean
     */
    public boolean checkIndexExist(String index) {
        String ok = "OK";
        boolean checkResult = false;
        try {
            Request request = new Request("HEAD", index);
            Response response = restHighLevelClient.getLowLevelClient().performRequest(request);
            if (ok.equals(response.getStatusLine().getReasonPhrase())) {
                checkResult = true;
            }
        } catch (IOException e) {
            log.error("检查索引失败", e);
        }
        return checkResult;
    }

    /**
     * 根据条件删除ES数据
     *
     * @param endPoint 接口index点
     * @param qryJsonStr  删除数据语句
     * @return boolean
     */
    public Boolean deleteDataByCondition(String endPoint, String qryJsonStr) {
        boolean deleteResult = false;
        HttpEntity entity = new NStringEntity(qryJsonStr, ContentType.APPLICATION_JSON);
        try {
            Request request = new Request("POST", endPoint);
            request.setEntity(entity);
            Response response = restHighLevelClient.getLowLevelClient().performRequest(request);
            String result = EntityUtils.toString(response.getEntity());
            if (StringUtils.isNotBlank(result)) {
                JSONObject deleteResultJson = JSONObject.parseObject(result);
                int deleteCount = deleteResultJson.getInteger("deleted");
                if (deleteCount > 0) {
                    deleteResult = true;
                }
            }
        } catch (IOException e) {
            log.error("根据条件删除ES数据失败", e);
        }
        return deleteResult;
    }


    /**
     * 描述 新增文章
     *
     * @param indexName 索引名称
     * @param docId     文章ID
     * @param source    文章源数据
     * @return java.lang.String
     * @author liugs
     * @date 2021/3/22 14:44:29
     */
    public String addDocument(String indexName, String docId, JSONObject source) {
        log.debug("新增文章入参：indexName='{}'，id='{}'，source='{}'", indexName, docId, source);
        if (StringUtils.isEmpty(indexName) || source.isEmpty()) {
            log.info("新增文章，[indexName]，[source]不能为空");
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "新增文章，[indexName]，[source]不能为空");
        }

        IndexRequest request = new IndexRequest(indexName);
        if (!StringUtils.isEmpty(docId)) {
            request.id(docId);
        }
        request.source(JSON.toJSONString(source), XContentType.JSON);
        try {
            IndexResponse response = restHighLevelClient.index(request, RequestOptions.DEFAULT);
            // 结果返回CRETED或者文档ID非空都视为成功
            if (DocWriteResponse.Result.CREATED.equals(response.getResult()) || StringUtils.isNotEmpty(response.getId())) {
                log.info("新增文章成功");
                return response.getId();
            }
            String failReason = null;
            if (response.getShardInfo().getFailed() > 0) {
                ReplicationResponse.ShardInfo.Failure[] failures = response.getShardInfo().getFailures();
                failReason = failures[0].reason();
            }
            log.info("新增文章失败：{}", failReason);
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "新增文章失败：" + failReason);
        } catch (IOException e) {
            log.error("新增文章异常：" + e);
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "新增文章异常：" + e.getMessage());
        }
    }

    /**
     * 描述 更新文章
     *
     * @param indexName 索引名称
     * @param docId     文章ID
     * @param source    文章数据
     * @return java.lang.Boolean
     * @author liugs
     * @date 2021/3/24 9:39:58
     */
    public void updateDoc(String indexName, String docId, JSONObject source) {
        if (log.isDebugEnabled()) {
            log.debug("根据文章ID更新文章，入参：indexName = '{}', docId = '{}'", indexName, docId);
        }
        UpdateRequest request = new UpdateRequest(indexName, "_doc",docId);
        // 在版本冲突，更新失败时，重试3次。
        request.retryOnConflict(1);
        request.doc(JSON.toJSONString(source), XContentType.JSON);
        try {
            UpdateResponse response = restHighLevelClient.update(request, RequestOptions.DEFAULT);
            // UPDATED和NOOP都视为成功
            if (DocWriteResponse.Result.UPDATED.equals(response.getResult())
                    || DocWriteResponse.Result.NOOP.equals(response.getResult())) {
                log.info("文章更新成功");
                return;
            }
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "根据文章ID更新文章失败：" + response.getResult());
        } catch (Exception e) {
            log.info("发生异常， 尝试使用ES6更新文章");
            updateForEs6(indexName, source, docId);
        }
    }

    /**
     * 描述 根据文章ID删除文章
     *
     * @param indexName 索引名称
     * @param docId     文章ID
     * @return java.lang.Boolean
     * @author liugs
     * @date 2021/3/22 16:58:57
     */
    public void deleteDocById(String indexName, String docId) {
        log.debug("根据文章ID删除文章，入参：indexName={}，docId={}", indexName, docId);
        if (StringUtils.isEmpty(indexName) || StringUtils.isEmpty(docId)) {
            log.info("根据文章ID删除文章，入参[indexName]和[docId]不能为空");
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "根据文章ID删除文章，入参[indexName]和[docId]不能为空");
        }
        DeleteRequest request = new DeleteRequest(indexName, docId);
        try {
            DeleteResponse response = restHighLevelClient.delete(request, RequestOptions.DEFAULT);
            if (response.getShardInfo().getFailed() <= 0) {
                log.info("删除索引[{}]文章[{}]成功", indexName, docId);
                return;
            }

            ReplicationResponse.ShardInfo.Failure[] failures = response.getShardInfo().getFailures();
            log.info("删除文章失败：{}", failures[0].reason());
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "根据文章ID删除文章失败：" + failures[0].reason());
        } catch (IOException e) {
            log.error("根据文章ID删除文章异常：" + e);
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "根据文章ID删除文章异常：" + e.getMessage());
        }
    }

    public void deleteBatch(String indexName, Object docIds) {
        String endPoint = "/" + indexName + "/_delete_by_query";
        //组装更新语句
        String esUpdateStr = toEsBatchDeleteStr(docIds);
        log.info("esDelete URL:{}", endPoint);
        log.info("esDeleteStrReq:{}", esUpdateStr);
        HttpEntity entity = new NStringEntity(esUpdateStr, ContentType.APPLICATION_JSON);
        try {
            Request request = new Request("POST", endPoint);
            request.setEntity(entity);
            Response response = restHighLevelClient.getLowLevelClient().performRequest(request);
            String result = EntityUtils.toString(response.getEntity());
            log.info("esUpdateStrRsp:{}", result);
            if (StringUtils.isNotBlank(result)) {
                JSONObject rspJsonObject = JSON.parseObject(result);
                Integer deleted = (Integer) rspJsonObject.get("deleted");
                if (deleted > 0) {
                    log.info("批量删除成功");
                }
            }
        } catch (IOException e) {
            log.error("批量删除ES数据失败:{},删除语句:{}", e, esUpdateStr);
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "批量删除ES数据失败(ES6)：" + e.getMessage());
        }
    }

    /**
     * 描述 根据文章ID获取文章
     *
     * @param indexName 索引名称
     * @param docId     文章ID
     * @return com.alibaba.fastjson.JSONObject
     * @author liugs
     * @date 2021/3/25 11:39:24
     */
    public JSONObject getDocById(String indexName, String docId) {
        if (log.isDebugEnabled()) {
            log.debug("根据文章ID获取文章：入参：indexName='{}'，docId='{}'", indexName, docId);
        }
        GetSourceRequest request = new GetSourceRequest(indexName, docId);
        Map<String, Object> source = new HashMap<>(16);
        try {
            GetSourceResponse response = restHighLevelClient.getSource(request, RequestOptions.DEFAULT);
            source = response.getSource();
        } catch (IOException e) {
            log.error("根据文章ID获取文章异常：" + e);
        }
        if (source.isEmpty()) {
            log.info("根据入参：入参：indexName='{}'，docId='{}'，未查询到文章", indexName, docId);
        }

        return new JSONObject(source);
    }



    public void updateForEs6(String indexName, JSONObject source, String docId) {

        int retry = 0;

        while (retry++ < 3){
            if (retry > 0){
                log.info("updateForEs6, 发起第(" + retry + ")次重试");
            }
            String endPoint = "/" + indexName + "/_doc/" + docId + "/_update";

            //组装更新语句
            String esUpdateStr = toEsUpdateStr(docId, source);
            log.info("========执行ES更新语句======= :{}", endPoint);
            log.info("========ES请求报文==========:{}", esUpdateStr);
            HttpEntity entity = new NStringEntity(esUpdateStr, ContentType.APPLICATION_JSON);
            try {
                Request request = new Request("POST", endPoint);
                request.setEntity(entity);
                Response response = restHighLevelClient.getLowLevelClient().performRequest(request);
                String result = EntityUtils.toString(response.getEntity());
                log.info("=======ES响应报文==========:{}", result);
                if (StringUtils.isNotBlank(result)) {
                    JSONObject rspJsonObject = JSON.parseObject(result);
                    JSONObject shardsJson = JSON.parseObject(JSON.toJSONString(rspJsonObject.get("_shards")));
                    if (!ObjectUtils.isEmpty(shardsJson) && (Integer)shardsJson.get("failed") == 0) {
                        log.info("更新成功，rspJsonObject：{} ", rspJsonObject);
                       break;
                    }else if ((Integer)shardsJson.get("failed") != 0 && retry >= 3){
                        log.info("更新重试完成, 更新状态：失败");
                        break;
                    }else {
                        log.info("更新失败，准备发起重试。。。。。。。");
                    }
                }
                log.info("更新失败，准备发起重试。。。。。。。");
            } catch (Exception e) {
                log.error("根据条件更新ES数据失败:{},更新语句:{}", e, esUpdateStr);
                log.info("更新失败，准备发起重试。。。。。。。");
            }
        }

        if (retry <= 3){
            log.info("本次更新成功");
        }
    }


    /**
     * 组装更新es语句
     *
     * @param id           主键ID
     * @param updateParams 更新的字段
     * @return 更新es语句
     */
    public String toEsUpdateStr(String id, JSONObject updateParams) {
        JSONObject esJsonObj = new JSONObject();
        esJsonObj.put("doc", updateParams);

        return esJsonObj.toJSONString();
    }

    public JSONObject getDocByDocId(String indexName, String docId) {
        String endPoint = "/" + indexName + "/_doc/" + docId;

        //组装更新语句
        try {
            Request request = new Request("GET", endPoint);

            Response response = restHighLevelClient.getLowLevelClient().performRequest(request);
            String result = EntityUtils.toString(response.getEntity());
            log.info("esGetStrRsp:{}", result);
            if (StringUtils.isNotBlank(result)) {
                JSONObject rspJsonObject = JSON.parseObject(result);
                if (!ObjectUtils.isEmpty("_seq_no") && !ObjectUtils.isEmpty("_primary_term")){
                    return rspJsonObject;
                }
                throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "查询ES文档信息(_seq_no, _primary_term)失败");
            }
        } catch (Exception e) {
            log.error("查询ES文档失败:{},查询语句:{}", e, endPoint);
            log.error("查询ES文档异常：" + e);
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "查询ES文档版本异常：" + e.getMessage());
        }
        return null;
    }

    /**
     * 组装source映射字段
     *
     * @param updateParams 更新的字段
     * @return source值
     */
    private String getSource(JSONObject updateParams) {
        StringBuilder stringBuilder = new StringBuilder();
        updateParams.keySet().forEach(key -> {
            stringBuilder.append("ctx._source.")
                    .append(key)
                    .append("=")
                    .append("params.")
                    .append(key)
                    .append(";");
        });
        return stringBuilder.toString();
    }

    public void updateBatch(String indexName, Object docIds, JSONObject source) {

        String endPoint = "/" + indexName + "/_update_by_query";
        //组装更新语句
        String esUpdateStr = toEsBatchUpdateStr(docIds, source);
        log.info("esUpdate URL:{}", endPoint);
        log.info("esUpdateStrReq:{}", esUpdateStr);
        HttpEntity entity = new NStringEntity(esUpdateStr, ContentType.APPLICATION_JSON);
        try {
            Request request = new Request("POST", endPoint);
            request.setEntity(entity);
            Response response = restHighLevelClient.getLowLevelClient().performRequest(request);
            String result = EntityUtils.toString(response.getEntity());
            log.info("esUpdateStrRsp:{}", result);
            if (StringUtils.isNotBlank(result)) {
                JSONObject rspJsonObject = JSON.parseObject(result);
                Integer updated = (Integer) rspJsonObject.get("updated");
                if (updated > 0) {
                    log.info("批量更新成功");
                }
            }
        } catch (IOException e) {
            log.error("批量更新ES数据失败:{},更新语句:{}", e, esUpdateStr);
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "批量更新ES数据失败(ES6)：" + e.getMessage());
        }
    }

    public String toEsBatchUpdateStr(Object docIds, JSONObject updateParams) {

        List<Object> docList = JSON.parseArray(JSON.toJSONString(docIds));

        //组装查询条件
        List<String> idsList = docList.stream().map(Object::toString).collect(Collectors.toList());

        JSONObject ids = new JSONObject();
        ids.put("values", idsList);

        JSONObject query = new JSONObject();
        query.put("ids", ids);

        JSONObject script = new JSONObject();
        script.put("source", getSource(updateParams));
        script.put("lang", "painless");
        script.put("params", updateParams);

        JSONObject esJsonObj = new JSONObject();
        esJsonObj.put("query", query);
        esJsonObj.put("script", script);

        return esJsonObj.toJSONString();
    }

    public String toEsBatchDeleteStr(Object docIds) {

        List<Object> docList = JSON.parseArray(JSON.toJSONString(docIds));

        //组装查询条件
        List<String> idsList = docList.stream().map(Object::toString).collect(Collectors.toList());

        JSONObject terms = new JSONObject();
        terms.put("_id", idsList);

        JSONObject query = new JSONObject();
        query.put("terms", terms);

        JSONObject esJsonObj = new JSONObject();
        esJsonObj.put("query", query);

        return esJsonObj.toJSONString();
    }


//    public void updateForEs6New(String indexName, JSONObject source, String docId) {
//        int retry = 0;
//
//        while (retry++ < 3){
//            if (retry > 0){
//                log.info("updateForEs6, 发起第(" + retry + ")次重试");
//            }
//            JSONObject docJSON = getDocByDocId(indexName, docId);
//
//            String endPoint = "/" + indexName + "/_update_by_query?if_seq_no=" + docJSON.get("_seq_no") +
//                    "&if_primary_term=" + docJSON.get("_primary_term");
//
//            //组装更新语句
//            String esUpdateStr = toEsUpdateStr(docId, source);
//            log.info("========执行ES更新语句======= :{}", endPoint);
//            log.info("========ES请求报文==========:{}", esUpdateStr);
//            HttpEntity entity = new NStringEntity(esUpdateStr, ContentType.APPLICATION_JSON);
//            try {
//                Request request = new Request("POST", endPoint);
//                request.setEntity(entity);
//                Response response = restHighLevelClient.getLowLevelClient().performRequest(request);
//                String result = EntityUtils.toString(response.getEntity());
//                log.info("=======ES响应报文==========:{}", result);
//                if (StringUtils.isNotBlank(result)) {
//                    JSONObject rspJsonObject = JSON.parseObject(result);
//                    if (1 == (Integer) rspJsonObject.get("updated")) {
//                        log.info("更新成功，rspJsonObject：{} ", rspJsonObject);
//                        return;
//                    }else if (!StringUtils.isEmpty("updated") && retry >= 3){
//                        log.info("更新重试完成, 更新状态：失败");
//                        return;
//                    }else {
//                        log.info("更新失败，准备发起重试。。。。。。。");
//                    }
//                }
//                log.info("更新失败，准备发起重试。。。。。。。");
//            } catch (Exception e) {
//                log.error("根据条件更新ES数据失败:{},更新语句:{}", e, esUpdateStr);
//                log.info("更新失败，准备发起重试。。。。。。。");
//            }
//        }
//
//        throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "重试次数超出限制， 本次更新结束， 更新状态：失败");
//    }
}
