package com.tydic.picker.utils;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.tydic.dyc.base.constants.BaseRspConstant;
import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.picker.constant.PickerConstants;
import com.tydic.picker.enums.DataEventTypeEnum;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.util.EntityUtils;
import org.elasticsearch.client.Request;
import org.elasticsearch.client.Response;
import org.elasticsearch.client.RestClient;

import java.io.IOException;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @ClassName EsToolCompatibleEdition
 * @Description ES工具兼容版
 * @Author liugs
 * @Date 2024/12/9 星期一 14:10
 */
@Slf4j
public class EsToolCompatibleEdition {

    private final RestClient restClient;

    public EsToolCompatibleEdition(RestClient restClient) {
        this.restClient = restClient;
    }

    public void doSyncByEventType(DataEventTypeEnum eventTypeEnum, String indexName, JSONObject esData) {
        switch (eventTypeEnum) {
            case CREATE:
                addDocument(indexName, esData.remove(PickerConstants.DOCUMENT_ID).toString(), esData);
                break;
            case UPDATE:
                updateDoc(indexName, esData.remove(PickerConstants.DOCUMENT_ID).toString(), esData);
                break;
            case BATCH_UPDATE:
                updateBatch(indexName, esData.remove(PickerConstants.DOCUMENT_ID), esData);
                break;
            case DELETE:
                deleteDocById(indexName, esData.getString(PickerConstants.DOCUMENT_ID));
                break;
            case DELETE_BY_QUERY:
                // 移除文档ID
                esData.remove(PickerConstants.DOCUMENT_ID);
                deleteDocByQuery(indexName, esData);
            default:
                break;
        }
    }

    /**
     * 描述 新增文章
     *
     * @param indexName 索引名称
     * @param docId     文章ID
     * @param source    文章源数据
     * @author liugs
     * @date 2021/3/22 14:44:29
     */
    public void addDocument(String indexName, String docId, JSONObject source) {
        log.debug("新增文章入参：indexName='{}'，id='{}'，source='{}'", indexName, docId, source);
        if (StringUtils.isEmpty(indexName) || source.isEmpty()) {
            log.info("新增文章，[indexName]，[source]不能为空");
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "新增文章，[indexName]，[source]不能为空");
        }

        String endPoint = StrUtil.format("/{}/_doc/{}", indexName, docId);
        HttpEntity entity = new NStringEntity(JSON.toJSONString(source, SerializerFeature.WriteMapNullValue), ContentType.APPLICATION_JSON);
        Request request = new Request("POST", endPoint);
        request.setEntity(entity);
        request.addParameter("refresh", "true");

        try {
            Response response = restClient.performRequest(request);
            String result = EntityUtils.toString(response.getEntity());
            log.info("新增文章返回结果：{}", result);
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode == 201 || statusCode == 200) {
                String responseBody = EntityUtils.toString(response.getEntity());
                log.info("新增文章成功，响应参数：{}", responseBody);
            } else {
                String failReason = EntityUtils.toString(response.getEntity());
                log.info("新增文章失败：{}", failReason);
                throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "新增文章失败：" + failReason);
            }
        } catch (IOException e) {
            log.error("新增文章异常：", e);
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "新增文章异常：" + e.getMessage());
        }
    }

    /**
     * 描述 更新文章
     *
     * @param indexName 索引名称
     * @param docId     文章ID
     * @param source    文章数据
     * @return java.lang.Boolean
     * @author liugs
     * @date 2021/3/24 9:39:58
     */
    public void updateDoc(String indexName, String docId, JSONObject source) {
        if (log.isDebugEnabled()) {
            log.debug("根据文章ID更新文章，入参：indexName = '{}', docId = '{}'", indexName, docId);
        }
        String endPoint = StrUtil.format("/{}/_update/{}", indexName, docId);
        HttpEntity entity = new NStringEntity("{\"doc\": " + JSON.toJSONString(source, SerializerFeature.WriteMapNullValue) + "}", ContentType.APPLICATION_JSON);

        Request request = new Request("POST", endPoint);
        request.setEntity(entity);
        request.addParameter("retry_on_conflict", "3");

        try {
            Response response = restClient.performRequest(request);
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode == 200) {
                String responseBody = EntityUtils.toString(response.getEntity());
                JSONObject jsonResponse = JSONObject.parseObject(responseBody);
                String result = jsonResponse.getString("result");
                if ("updated".equals(result) || "noop".equals(result)) {
                    log.info("文章更新成功");
                    return;
                }
                throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "根据文章ID更新文章失败：" + result);
            } else {
                String failReason = EntityUtils.toString(response.getEntity());
                log.info("根据文章ID更新文章失败：{}", failReason);
                throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "根据文章ID更新文章失败：" + failReason);
            }
        } catch (IOException e) {
            log.error("根据文章ID更新文章异常：", e);
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "根据文章ID更新文章异常：" + e.getMessage());
        }
    }

    /**
     * 根据文档ID列表，批量更新
     * @param indexName
     * @param docIds
     * @param source
     */
    public void updateBatch(String indexName, Object docIds, JSONObject source) {
        String endPoint = "/" + indexName + "/_update_by_query";
        //组装更新语句
        String esUpdateStr = toEsBatchUpdateStr(docIds, source);
        log.info("esUpdate URL:{}", endPoint);
        log.info("esUpdateStrReq:{}", esUpdateStr);
        HttpEntity entity = new NStringEntity(esUpdateStr, ContentType.APPLICATION_JSON);
        try {
            Request request = new Request("POST", endPoint);
            request.setEntity(entity);
            Response response = restClient.performRequest(request);
            String result = EntityUtils.toString(response.getEntity());
            log.info("esUpdateStrRsp:{}", result);
            if (StringUtils.isNotBlank(result)) {
                JSONObject rspJsonObject = JSON.parseObject(result);
                Integer updated = (Integer) rspJsonObject.get("updated");
                if (updated > 0) {
                    log.info("批量更新成功");
                }
            }
        } catch (IOException e) {
            log.error("批量更新ES数据失败:{},更新语句:{}", e, esUpdateStr);
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "批量更新ES数据失败(ES6)：" + e.getMessage());
        }
    }

    /**
     * 描述 根据文章ID删除文章
     *
     * @param indexName 索引名称
     * @param docId     文章ID
     * @return java.lang.Boolean
     * @author liugs
     * @date 2021/3/22 16:58:57
     */
    public void deleteDocById(String indexName, String docId) {
        log.debug("根据文章ID删除文章，入参：indexName={}，docId={}", indexName, docId);
        if (StringUtils.isEmpty(indexName) || StringUtils.isEmpty(docId)) {
            log.info("根据文章ID删除文章，入参[indexName]和[docId]为空，执行结束！");
            return;
        }
        Request request = new Request("DELETE", StrUtil.format("/{}/_doc/{}", indexName, docId));
        request.addParameter("refresh", "true");

        try {
            Response response = restClient.performRequest(request);
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode == 200) {
                log.info("删除索引[{}]文章[{}]成功", indexName, docId);
            } else {
                HttpEntity entity = response.getEntity();
                String responseString = EntityUtils.toString(entity);
                log.error("删除文章失败，状态码：{}，响应：{}", statusCode, responseString);
                throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "根据文章ID删除文章失败：" + responseString);
            }
        } catch (IOException e) {
            log.error("根据文章ID删除文章异常：" + e);
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "根据文章ID删除文章异常：" + e.getMessage());
        }
    }

    /**
     * 描述 根据条件删除文章
     * @param indexName 索引名称
     * @param condition 删除条件
     * @return void
     * @author liugs
     * @date 2023/11/29 0029 17:43
     */
    public void deleteDocByQuery(String indexName, JSONObject condition) {
        log.debug("根据条件删除文章，入参：indexName={}，condition={}", indexName, condition.toJSONString());
        if (StringUtils.isEmpty(indexName) || ObjectUtil.isEmpty(condition)) {
            log.info("根据条件删除文章，入参[indexName]和[condition]为空，执行结束！");
            return;
        }
        // 构建查询条件
        StringBuilder queryBuilder = new StringBuilder("{\"query\":{\"bool\":{\"must\":[");
        Iterator<Map.Entry<String, Object>> iterator = condition.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<String, Object> entry = iterator.next();
            queryBuilder.append("{\"term\":{\"").append(entry.getKey()).append("\":\"").append(entry.getValue()).append("\"}}");
            if (iterator.hasNext()) {
                queryBuilder.append(",");
            }
        }
        queryBuilder.append("]}}}");

        Request request = new Request("POST", "/" + indexName + "/_delete_by_query");
        request.setEntity(new NStringEntity(queryBuilder.toString(), ContentType.APPLICATION_JSON));
        request.addParameter("refresh", "true");

        try {
            Response response = restClient.performRequest(request);
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode == 200) {
                HttpEntity entity = response.getEntity();
                String responseString = EntityUtils.toString(entity);
                log.info("根据条件{}删除索引{}文章调用成功，响应：{}", condition.toJSONString(), indexName, responseString);
                Integer deleted = JSON.parseObject(responseString).getInteger("deleted");
                if (deleted > 1) {
                    log.info("根据条件删除索引文章成功。");
                }
            } else {
                HttpEntity entity = response.getEntity();
                String responseString = EntityUtils.toString(entity);
                log.error("删除文章失败，状态码：{}，响应：{}", statusCode, responseString);
                throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "根据条件删除文章失败：" + responseString);
            }
        } catch (IOException e) {
            log.error("根据条件删除文章异常：" + e);
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "根据条件删除文章异常：" + e.getMessage());
        }
    }


    public String toEsBatchUpdateStr(Object docIds, JSONObject updateParams) {

        List<Object> docList = JSON.parseArray(JSON.toJSONString(docIds));

        //组装查询条件
        List<String> idsList = docList.stream().map(Object::toString).collect(Collectors.toList());

        JSONObject ids = new JSONObject();
        ids.put("values", idsList);

        JSONObject query = new JSONObject();
        query.put("ids", ids);

        JSONObject script = new JSONObject();
        script.put("source", getSource(updateParams));
        script.put("lang", "painless");
        script.put("params", updateParams);

        JSONObject esJsonObj = new JSONObject();
        esJsonObj.put("query", query);
        esJsonObj.put("script", script);

        return esJsonObj.toJSONString();
    }

    private String getSource(JSONObject updateParams) {
        StringBuilder stringBuilder = new StringBuilder();
        updateParams.keySet().forEach(key -> {
            stringBuilder.append("ctx._source.")
                    .append(key)
                    .append("=")
                    .append("params.")
                    .append(key)
                    .append(";");
        });
        return stringBuilder.toString();
    }

}
