/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service;

import java.net.InetAddress;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.$internal.com.google.common.collect.Iterables;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.concurrent.StageManager;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.ReadRepairDecision;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.ConsistencyLevel;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.ReadCommand;
import org.apache.cassandra.db.ReadResponse;
import org.apache.cassandra.db.Row;
import org.apache.cassandra.exceptions.ReadTimeoutException;
import org.apache.cassandra.exceptions.UnavailableException;
import org.apache.cassandra.metrics.ReadRepairMetrics;
import org.apache.cassandra.net.MessageOut;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.service.DigestMismatchException;
import org.apache.cassandra.service.ReadCallback;
import org.apache.cassandra.service.RowDigestResolver;
import org.apache.cassandra.service.StorageProxy;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.tracing.TraceState;
import org.apache.cassandra.tracing.Tracing;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractReadExecutor {
    private static final Logger logger = LoggerFactory.getLogger(AbstractReadExecutor.class);
    protected final ReadCommand command;
    protected final List<InetAddress> targetReplicas;
    protected final RowDigestResolver resolver;
    protected final ReadCallback<ReadResponse, Row> handler;
    protected final TraceState traceState;

    AbstractReadExecutor(ReadCommand command, ConsistencyLevel consistencyLevel, List<InetAddress> targetReplicas) {
        this.command = command;
        this.targetReplicas = targetReplicas;
        this.resolver = new RowDigestResolver(command.ksName, command.key);
        this.traceState = Tracing.instance.get();
        this.handler = new ReadCallback<ReadResponse, Row>(this.resolver, consistencyLevel, command, targetReplicas);
    }

    private static boolean isLocalRequest(InetAddress replica) {
        return replica.equals(FBUtilities.getBroadcastAddress());
    }

    protected void makeDataRequests(Iterable<InetAddress> endpoints) {
        this.makeRequests(this.command, endpoints);
    }

    protected void makeDigestRequests(Iterable<InetAddress> endpoints) {
        this.makeRequests(this.command.copy().setIsDigestQuery(true), endpoints);
    }

    private void makeRequests(ReadCommand readCommand, Iterable<InetAddress> endpoints) {
        MessageOut<ReadCommand> message = null;
        boolean hasLocalEndpoint = false;
        for (InetAddress endpoint : endpoints) {
            if (AbstractReadExecutor.isLocalRequest(endpoint)) {
                hasLocalEndpoint = true;
                continue;
            }
            if (this.traceState != null) {
                this.traceState.trace("reading {} from {}", readCommand.isDigestQuery() ? "digest" : "data", endpoint);
            }
            logger.trace("reading {} from {}", (Object)(readCommand.isDigestQuery() ? "digest" : "data"), (Object)endpoint);
            if (message == null) {
                message = readCommand.createMessage();
            }
            MessagingService.instance().sendRR(message, endpoint, this.handler);
        }
        if (hasLocalEndpoint) {
            logger.trace("reading {} locally", (Object)(readCommand.isDigestQuery() ? "digest" : "data"));
            StageManager.getStage(Stage.READ).maybeExecuteImmediately(new StorageProxy.LocalReadRunnable(this.command, this.handler));
        }
    }

    public abstract void maybeTryAdditionalReplicas();

    public abstract Collection<InetAddress> getContactedReplicas();

    public abstract void executeAsync();

    public Row get() throws ReadTimeoutException, DigestMismatchException {
        return this.handler.get();
    }

    public static AbstractReadExecutor getReadExecutor(ReadCommand command, ConsistencyLevel consistencyLevel) throws UnavailableException {
        Keyspace keyspace = Keyspace.open(command.ksName);
        List<InetAddress> allReplicas = StorageProxy.getLiveSortedEndpoints(keyspace, command.key);
        ReadRepairDecision repairDecision = Schema.instance.getCFMetaData(command.ksName, command.cfName).newReadRepairDecision();
        List<InetAddress> targetReplicas = consistencyLevel.filterForQuery(keyspace, allReplicas, repairDecision);
        consistencyLevel.assureSufficientLiveNodes(keyspace, targetReplicas);
        if (StorageService.instance.isClientMode()) {
            return new NeverSpeculatingReadExecutor(command, consistencyLevel, targetReplicas);
        }
        if (repairDecision != ReadRepairDecision.NONE) {
            Tracing.trace("Read-repair {}", (Object)repairDecision);
            ReadRepairMetrics.attempted.mark();
        }
        ColumnFamilyStore cfs = keyspace.getColumnFamilyStore(command.cfName);
        CFMetaData.SpeculativeRetry.RetryType retryType = cfs.metadata.getSpeculativeRetry().type;
        if (retryType == CFMetaData.SpeculativeRetry.RetryType.NONE || consistencyLevel.blockFor(keyspace) == allReplicas.size()) {
            return new NeverSpeculatingReadExecutor(command, consistencyLevel, targetReplicas);
        }
        if (targetReplicas.size() == allReplicas.size()) {
            return new AlwaysSpeculatingReadExecutor(cfs, command, consistencyLevel, targetReplicas);
        }
        InetAddress extraReplica = allReplicas.get(targetReplicas.size());
        if (repairDecision == ReadRepairDecision.DC_LOCAL && targetReplicas.contains(extraReplica)) {
            for (InetAddress address : allReplicas) {
                if (targetReplicas.contains(address)) continue;
                extraReplica = address;
                break;
            }
        }
        targetReplicas.add(extraReplica);
        if (retryType == CFMetaData.SpeculativeRetry.RetryType.ALWAYS) {
            return new AlwaysSpeculatingReadExecutor(cfs, command, consistencyLevel, targetReplicas);
        }
        return new SpeculatingReadExecutor(cfs, command, consistencyLevel, targetReplicas);
    }

    private static class AlwaysSpeculatingReadExecutor
    extends AbstractReadExecutor {
        private final ColumnFamilyStore cfs;

        public AlwaysSpeculatingReadExecutor(ColumnFamilyStore cfs, ReadCommand command, ConsistencyLevel consistencyLevel, List<InetAddress> targetReplicas) {
            super(command, consistencyLevel, targetReplicas);
            this.cfs = cfs;
        }

        @Override
        public void maybeTryAdditionalReplicas() {
        }

        @Override
        public Collection<InetAddress> getContactedReplicas() {
            return this.targetReplicas;
        }

        @Override
        public void executeAsync() {
            this.makeDataRequests(this.targetReplicas.subList(0, this.targetReplicas.size() > 1 ? 2 : 1));
            if (this.targetReplicas.size() > 2) {
                this.makeDigestRequests(this.targetReplicas.subList(2, this.targetReplicas.size()));
            }
            this.cfs.metric.speculativeRetries.inc();
        }
    }

    private static class SpeculatingReadExecutor
    extends AbstractReadExecutor {
        private final ColumnFamilyStore cfs;
        private volatile boolean speculated = false;

        public SpeculatingReadExecutor(ColumnFamilyStore cfs, ReadCommand command, ConsistencyLevel consistencyLevel, List<InetAddress> targetReplicas) {
            super(command, consistencyLevel, targetReplicas);
            this.cfs = cfs;
        }

        @Override
        public void executeAsync() {
            List initialReplicas = this.targetReplicas.subList(0, this.targetReplicas.size() - 1);
            if (this.handler.blockfor < initialReplicas.size()) {
                this.makeDataRequests(initialReplicas.subList(0, 2));
                if (initialReplicas.size() > 2) {
                    this.makeDigestRequests(initialReplicas.subList(2, initialReplicas.size()));
                }
            } else {
                this.makeDataRequests(initialReplicas.subList(0, 1));
                if (initialReplicas.size() > 1) {
                    this.makeDigestRequests(initialReplicas.subList(1, initialReplicas.size()));
                }
            }
        }

        @Override
        public void maybeTryAdditionalReplicas() {
            if (this.cfs.sampleLatencyNanos > TimeUnit.MILLISECONDS.toNanos(this.command.getTimeout())) {
                return;
            }
            if (!this.handler.await(this.cfs.sampleLatencyNanos, TimeUnit.NANOSECONDS)) {
                ReadCommand retryCommand = this.command;
                if (this.resolver.getData() != null) {
                    retryCommand = this.command.copy().setIsDigestQuery(true);
                }
                InetAddress extraReplica = (InetAddress)Iterables.getLast(this.targetReplicas);
                if (this.traceState != null) {
                    this.traceState.trace("speculating read retry on {}", extraReplica);
                }
                logger.trace("speculating read retry on {}", (Object)extraReplica);
                MessagingService.instance().sendRR(retryCommand.createMessage(), extraReplica, this.handler);
                this.speculated = true;
                this.cfs.metric.speculativeRetries.inc();
            }
        }

        @Override
        public Collection<InetAddress> getContactedReplicas() {
            return this.speculated ? this.targetReplicas : this.targetReplicas.subList(0, this.targetReplicas.size() - 1);
        }
    }

    private static class NeverSpeculatingReadExecutor
    extends AbstractReadExecutor {
        public NeverSpeculatingReadExecutor(ReadCommand command, ConsistencyLevel consistencyLevel, List<InetAddress> targetReplicas) {
            super(command, consistencyLevel, targetReplicas);
        }

        @Override
        public void executeAsync() {
            this.makeDataRequests(this.targetReplicas.subList(0, 1));
            if (this.targetReplicas.size() > 1) {
                this.makeDigestRequests(this.targetReplicas.subList(1, this.targetReplicas.size()));
            }
        }

        @Override
        public void maybeTryAdditionalReplicas() {
        }

        @Override
        public Collection<InetAddress> getContactedReplicas() {
            return this.targetReplicas;
        }
    }
}

