package com.taobao.hsf.tbremoting.invoke;

/**
 * High-Speed Service Framework (HSF)
 *
 * www.taobao.com
 *  (C) 淘宝(中国) 2003-2014
 */

import com.taobao.hsf.domain.HSFResponse;
import com.taobao.hsf.exception.HSFException;
import com.taobao.hsf.exception.HSFTimeOutException;
import com.taobao.hsf.remoting.exception.RemotingUncheckedException;
import com.taobao.middleware.logger.support.LoggerHelper;
import com.taobao.remoting.ResponseFuture;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * Future方式来获取远程返回的结果
 * 
 * 可满足并行调用的需求：每次调用后，使用getFuture获取对应的future。
 */
public class HSFResponseFuture {

    public static ThreadLocal<Future<Object>> future = new ThreadLocal<Future<Object>>();

    public static ThreadLocal<HSFFutureListener> listener = new ThreadLocal<HSFFutureListener>();

    /**
     * 获取远程对象的结果。 如果通信层抛出异常，把异常外包装HSFException。
     * <p>
     * 如果指定的时间用完但还没有收到响应，则返回{@link ResponseFuture#ASYN_NOT_DONE}
     * 
     * @param timeout
     *            超时时间
     * @throws HSFException
     * @throws InterruptedException
     */
    public static Object getResponse(long timeout) throws HSFException, InterruptedException {
        if (null == future.get()) {
            throw new HSFException("Thread [" + Thread.currentThread() + "] have not set the response future!");
        }

        try {
            HSFResponse hsfResp = (HSFResponse) (future.get().get(timeout, TimeUnit.MILLISECONDS));
            if (hsfResp.isError()) {
                throw new HSFException(hsfResp.getErrorMsg(), hsfResp.getErrorMsg());
            }
            return hsfResp.getAppResponse();
        } catch (ExecutionException e) {
            throw new HSFTimeOutException(getErrorLog(), e);
        } catch (TimeoutException e) {
            throw new HSFTimeOutException(getErrorLog(), e);
        } catch (RemotingUncheckedException e) {
            if (e.getErrorCode().getCode() == 103) {
                throw new HSFTimeOutException(getErrorLog(), e);
            } else {
                throw e;
            }
        }
    }

    private static String getErrorLog() {
        return LoggerHelper.getErrorCodeStr("hsf", "HSF-0002", "Business problems", "");
    }

    public static HSFFuture getFuture() throws HSFException {
        if (null == future.get()) {
            throw new HSFException("Thread [" + Thread.currentThread() + "] have not set the response future!");
        }

        HSFFuture hsfFuture = new HSFFuture(HSFResponseFuture.future.get());
        future.remove();
        return hsfFuture;
    }

    /**
     * 设置Future方式的调用,不需要应用调用
     * 
     */
    public static void setFuture(Future<Object> future) {
        HSFResponseFuture.future.set(future);

        if (HSFResponseFuture.listener.get() != null) {
            HSFResponseFuture.listener.get().onSetFuture();
        }
    }

    public static void setFutureListener(HSFFutureListener listener) {
        HSFResponseFuture.listener.set(listener);
    }
}
