package com.taobao.hsf.lightapi;

import com.alibaba.dubbo.rpc.service.GenericService;
import com.taobao.hsf.app.spring.util.HSFSpringConsumerBean;
import com.taobao.hsf.lightapi.util.AsynType;
import com.taobao.hsf.lightapi.util.GenericMethod;
import com.taobao.hsf.lightapi.util.LightConstant;
import com.taobao.hsf.lightapi.util.Partner;
import com.taobao.hsf.model.metadata.MethodSpecial;
import com.taobao.hsf.standalone.HSFStarter;
import com.taobao.hsf.standalone.util.Constant;
import com.taobao.hsf.tbremoting.invoke.HSFResponseCallback;
import com.taobao.hsf.util.HSFConstants;
import com.taobao.hsf.util.HSFServiceTargetUtil;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

/**
 * HSF消费端api类
 * Created by huangsheng.hs on 2014/10/25.
 */
public class ConsumerService {
    private HSFSpringConsumerBean consumer;
    private AtomicReference<MockProxy> mockProxy = new AtomicReference<MockProxy>(null);
    private AtomicBoolean inited = new AtomicBoolean(false);
    private AtomicBoolean consumed = new AtomicBoolean(false);
    private List<MethodSpecial> methodSpecials = new LinkedList<MethodSpecial>();
    private Map<String, AsynType> asynMethods = new ConcurrentHashMap<String, AsynType>();
    private ConcurrentHashMap<Method, Object> mockContext = new ConcurrentHashMap<Method, Object>();
    private AtomicBoolean mockAll = new AtomicBoolean(false);
    private String serviceInterface;
    private String version = "1.0.0";
    private String uniqueName;
    private GenericService genericService;

    public ConsumerService newConsumer() {
        if (inited.compareAndSet(false, true)) {
            Class<?> consumerClazz = LightConstant.pandoraInited ? initConsumerClass() : HSFStarter.getHSFSpringConsumerBean();
            if (consumerClazz != null) {
                try {
                    consumer = (HSFSpringConsumerBean) consumerClazz.newInstance();
                } catch (Exception e) {
                    throw new RuntimeException(LightConstant.INIT_C_FAIL, e);
                }
            } else {
                throw new RuntimeException(LightConstant.INIT_C_FAIL + "because there is no HSFSpringConsumerBean's Class found.");
            }
            return this;
        } else
            throw new RuntimeException(LightConstant.MULTI_INIT);
    }


    /**
     * 设置要提供的HSF服务名
     *
     * @param service HSF服务名
     * @return
     */
    public ConsumerService service(String service) {
        checkInited();
        consumer.setInterfaceName(service);
        this.serviceInterface = service;
        return this;
    }

    /**
     * 设置要提供的HSF服务的版本号
     *
     * @param version
     * @return
     */
    public ConsumerService version(String version) {
        checkInited();
        consumer.setVersion(version);
        this.version = version;
        return this;
    }

    /**
     * 设置要提供的HSF服务的组别
     *
     * @param group
     * @return
     */
    public ConsumerService group(String group) {
        checkInited();
        consumer.setGroup(group);
        return this;
    }

    public ConsumerService generic() {
        checkInited();
        consumer.setGeneric(HSFConstants.GENERIC_SERIALIZATION_DEFAULT);
        return this;
    }

    /**
     * @param method GenericMethod
     * @return
     * @throws Exception
     */
    public Object genericInvoke(GenericMethod method) throws Exception {
        checkInited();
        if (genericService == null) {
            consumer.setGeneric(HSFConstants.GENERIC_SERIALIZATION_DEFAULT);
            genericService = (GenericService) consumer.getObject();
        }
        return genericService.$invoke(method.getMethodName(), method.getArgsType(), method.getArgsVal());
    }

    /**
     * 统一设置所有服务的超时时间ͳ
     *
     * @param timeout 超时时间
     * @return
     */
    public ConsumerService timeout(int timeout) {
        checkInited();
        consumer.setClientTimeout(timeout);
        return this;
    }


    /**
     * 针对某个方法设置单独的超时时间
     *
     * @param methodName 方法名
     * @param timeout    对应方法的超时时间
     * @return
     */
    public ConsumerService methodTimeout(String methodName, int timeout) {
        checkInited();
        MethodSpecial ms = new MethodSpecial();
        ms.setMethodName(methodName);
        ms.setClientTimeout(timeout);
        synchronized (methodSpecials) {
            methodSpecials.add(ms);
        }
        return this;
    }

    /**
     * 获取消费服务的接口，可以进行强制转换。
     *
     * @return
     */
    public Object subscribe() {
        checkInited();
        try {
            if (consumed.compareAndSet(false, true)) {
                putMethodSpecials();
                putAsynMethods();
                consumer.setMaxWaitAddressTimeMS(3000);
                consumer.init();
                this.uniqueName = this.serviceInterface + ":" + this.version;
            }
            return consumer.getObject();
        } catch (Exception e) {
            throw new RuntimeException(LightConstant.CONS_FAIL, e);
        }
    }


    /**
     * 获取针对服务接口的Mock对象，可以直接进行调用
     *
     * @return
     */
    public Object getMockService() {
        checkInited();
        if (mockProxy.get() == null) {
            Class<?> clazz = consumer.getObjectType();
            if (clazz != null)
                mockProxy.compareAndSet(null, new MockProxy(new Class<?>[]{clazz}));
        }
        return mockProxy.get().getProxyInstance();
    }


    /**
     * 传入需要mock的Method对象和返回值，在调用这个方法的时候会返回对应的expectValue
     *
     * @param method
     * @param expectValue
     */
    public void mock(Method method, Object expectValue) {
        if (method == null)
            return;
        mockContext.put(method, expectValue);
    }


    /**
     * 传入需要mock的Method对象的List和返回值，在调用这个方法的时候会返回对应的expectValue
     *
     * @param methods
     * @param expectValue
     */
    public void mock(List<Method> methods, Object expectValue) {
        if (methods == null)
            return;
        for (Method method : methods) {
            mockContext.put(method, expectValue);
        }
    }

    /**
     * 传入需要取消mock的Method对象
     *
     * @param method
     */
    public void cancelMock(Method method) {
        if (method == null)
            return;
        mockContext.remove(method);
    }

    /**
     * 取消所有方法的mock
     */
    public void cancelAllMock() {
        mockContext.clear();
    }

    public void cancelMock(List<Method> methods) {
        if (methods == null)
            return;
        for (Method method : methods) {
            mockContext.remove(method);
        }
    }

    private void putAsynMethods() {
        checkInited();
        List<String> asynDesc = new ArrayList<String>();
        for (Map.Entry<String, AsynType> entry : asynMethods.entrySet()) {
            if (entry.getValue().getType().equals("future")) {
                asynDesc.add(MessageFormat.format("name:{0};type:{1}", new Object[]{entry.getKey(), entry.getValue().getType()}).toString());
            } else if (entry.getValue().getType().equals("callback")) {
                asynDesc.add(MessageFormat.format("name:{0};type:{1};listener:{2}", new Object[]{entry.getKey(), entry.getValue().getType(), entry.getValue().getListenerClass()}).toString());
            }
        }
        consumer.setAsyncallMethods(asynDesc);
    }

    /**
     * 已默认等待地址，不需要再调用此方法
     */
    @Deprecated
    public void sync() throws Exception {
        return;
    }

    /**
     * 指定IP调用
     *
     * @param ip 指定的IP
     * @return
     */
    public ConsumerService targetIp(String ip) throws Exception {
        checkInited();
        if (Constant.main_version != 1)
            Partner.setRunModeToTest();
        consumer.setTarget(HSFServiceTargetUtil.formatTargetURL(ip));
        return this;
    }

    /**
     * 获取所有可用地址列表
     *
     *  2.1 和 2.2取法不一样，因为改成了默认等待地址，不需要再调用此方法
     * @return
     * @throws Exception
     */
    @Deprecated
    public List<String> addresses() throws Exception {
        checkInited();
        return Partner.allAddresses(uniqueName);
    }

    public boolean isConsumed() {
        return consumed.get();
    }

    /**
     * 增加一个通过future方式调用的方法
     *
     * @param methodName
     */
    public ConsumerService futureMethod(String methodName) {
        checkInited();
        asynMethods.put(methodName, new AsynType(AsynType.Type.future));
        return this;
    }

    /**
     * 增加一个通过callback方式调用的方法
     *
     * @param methodName
     * @param listener   HSF回调的listener，要求必须实现HSFResponseCallback接口。
     */
    public ConsumerService callbackMethod(String methodName, Object listener) {
        checkInited();
        if (HSFResponseCallback.class.isAssignableFrom(listener.getClass())) {
            asynMethods.put(methodName, new AsynType(AsynType.Type.callback, listener.getClass()));
        } else {
            throw new RuntimeException("The callback listener : " + listener + ",doesn't implements HSFResponseCallback interface.");
        }
        return this;
    }

    private void putMethodSpecials() {
        synchronized (methodSpecials) {
            if (methodSpecials.size() > 0) {
                /**
                 * 坑爹啊，直接toArray会丢失类型
                 */
                MethodSpecial[] specials = new MethodSpecial[methodSpecials.size()];
                for (int i = 0; i < methodSpecials.size(); i++) {
                    specials[i] = methodSpecials.get(i);
                }
                consumer.setMethodSpecials(specials);
            }
        }
    }

    private void checkInited() {
        if (inited.compareAndSet(true, true))
            return;
        else
            throw new RuntimeException(LightConstant.NOT_INIT_BEAN);
    }

    /**
     * 如果target指定的IP不在可用地址列表里，则抛异常
     *
     * @param addresses
     * @param ip
     */
    private void validIp(List<String> addresses, String ip) {
        for (String address : addresses)
            if (address.startsWith(ip))
                return;
        if (addresses == null || addresses.size() == 0)
            throw new RuntimeException("There is no address for service you want to invoke.Maybe you invoke too fast or this service is not exist");
        throw new RuntimeException(LightConstant.INVALID_IP);
    }


    private Class<?> initConsumerClass() {
        try {
            return Class.forName("com.taobao.hsf.app.spring.util.HSFSpringConsumerBean", false, ServiceFactory.class.getClassLoader());
        } catch (ClassNotFoundException e) {
            throw new RuntimeException(LightConstant.INIT_C_FAIL, e);
        }
    }


    public class MockProxy implements InvocationHandler {

        private Object proxyInstance;
        private String proxyInterfaceName;

        private Method equalsMethod;
        private Method toStringMethod;
        private Method hashCodeMethod;

        public MockProxy(final Class<?>[] classes) {
            proxyInstance = Proxy.newProxyInstance(MockProxy.class.getClassLoader(), classes, this);
            proxyInterfaceName = classes[0].getName();
            try {
                Field hashCodeField = this.proxyInstance.getClass().getDeclaredField("m0");
                hashCodeField.setAccessible(true);
                hashCodeMethod = (Method) hashCodeField.get(this.proxyInstance);
                Field equalsField = this.proxyInstance.getClass().getDeclaredField("m1");
                equalsField.setAccessible(true);
                equalsMethod = (Method) equalsField.get(this.proxyInstance);
                Field toStringField = this.proxyInstance.getClass().getDeclaredField("m2");
                toStringField.setAccessible(true);
                toStringMethod = (Method) toStringField.get(this.proxyInstance);
            } catch (Exception e) {
                throw new RuntimeException(LightConstant.INIT_PROXY_FAIL, e);
            }
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            if (method == toStringMethod) {
                return proxyInterfaceName;
            }
            if (method == equalsMethod) {
                return proxy == args[0];
            }
            if (method == hashCodeMethod) {
                return System.identityHashCode(proxy);
            }
            Object returnValue = mockContext.get(method);
            if (returnValue != null)
                return returnValue;
            else
                return method.invoke(consumer.getObject(), args);
        }

        public Object getProxyInstance() {
            return proxyInstance;
        }
    }
}
