package com.taobao.hsf.lightapi;

import com.taobao.hsf.app.spring.util.HSFSpringProviderBean;
import com.taobao.hsf.lightapi.util.LightConstant;
import com.taobao.hsf.lightapi.util.Partner;
import com.taobao.hsf.model.ApplicationModel;
import com.taobao.hsf.model.metadata.MethodSpecial;
import com.taobao.hsf.standalone.HSFStarter;

import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * HSF服务端api类
 * @Author lanshan
 */
public class ProviderService {
    private HSFSpringProviderBean provider;
    private AtomicBoolean inited = new AtomicBoolean(false);
    private AtomicBoolean published = new AtomicBoolean(false);
    private List<MethodSpecial> methodSpecials = new LinkedList<MethodSpecial>();
    private String service;
    private String version = "1.0.0";

    public ProviderService newProvider() {
        if (inited.compareAndSet(false, true)) {
            Class<?> providerClazz = LightConstant.pandoraInited ? initProviderClass() : HSFStarter.getHSFSpringProviderBean();
            if (providerClazz != null) {
                try {
                    provider = (HSFSpringProviderBean) providerClazz.newInstance();
                } catch (Exception e) {
                    throw new RuntimeException(LightConstant.INIT_P_FAIL,e);
                }
            } else {
                throw new RuntimeException(LightConstant.INIT_P_FAIL+ "because there is no HSFSpringProviderBean Class found.");
            }
            return this;
        } else {
            throw new RuntimeException(LightConstant.MULTI_INIT);
        }
    }


    /**
     * 设置要提供的HSF服务名
     *
     * @param service HSF服务名
     * @return
     */
    public ProviderService service(String service) {
        checkInited();
        provider.setServiceInterface(service);
        this.service = service;
        return this;
    }

    /**
     * 设置要提供的HSF服务的版本号
     *
     * @param version
     * @return
     */
    public ProviderService version(String version) {
        checkInited();
        provider.setServiceVersion(version);
        this.version = version;
        return this;
    }

    /**
     * 设置要提供的HSF服务的组别
     *
     * @param group
     * @return
     */
    public ProviderService group(String group) {
        checkInited();
        provider.setServiceGroup(group);
        return this;
    }

    /**
     * 设置要提供的HSF服务的接口实现类，这里要传入一个对象
     *
     * @param impl HSF服务接口对应的实现类
     * @return
     */
    public ProviderService impl(Object impl) {
        checkInited();
        provider.setTarget(impl);
        return this;
    }

    /**
     * 统一设置所有服务的超时时间ͳ
     *
     * @param timeout 超时时间
     * @return
     */
    public ProviderService timeout(int timeout) {
        checkInited();
        provider.setClientTimeout(timeout);
        return this;
    }


    /**
     * 针对某个方法设置单独的超时时间
     *
     * @param methodName 方法名
     * @param timeout    对应方法的超时时间
     * @return
     */
    public ProviderService methodTimeout(String methodName, int timeout) {
        checkInited();
        MethodSpecial ms = new MethodSpecial();
        ms.setMethodName(methodName);
        ms.setClientTimeout(timeout);
        synchronized (methodSpecials) {
            methodSpecials.add(ms);
        }
        return this;
    }


    /**
     * 单元化服务writeMode和route的设置，writeMode为非unit的，不需要设置route
     *
     * @param writeMode 服务的writeMode
     * @param route     buyerId的位数，第一个参数为buyerId则route为0，非unit服务可以随便填一个。
     * @return
     */
    public ProviderService writeMode(String writeMode, int route) {
        checkInited();
        provider.setWriteMode(writeMode);
        if (writeMode.equals("unit")) {
            provider.setRoute(route);
        }
        return this;
    }

    /**
     * 标识此服务是否支持异步调用，默认值为false，也就是不支持异步调用
     *
     * @param support
     * @return
     */
    public ProviderService supportAsynCall(boolean support) {
        checkInited();
        provider.setSupportAsynCall(support ? "true" : "false");
        return this;
    }

    /**
     * 该配置对接口中的所有方法生效
     * 但是如果客户端通过MethodSpecial属性对某方法配置了超时时间，则该方法的超时时间以客户端配置为准，其他方法不受影响，还是以服务端配置为准
     *
     * @param clientTimeout
     * @return
     */
    public ProviderService clientTimeout(int clientTimeout) {
        checkInited();
        provider.setClientTimeout(clientTimeout);
        return this;
    }

    /**
     * 发布服务。在发布前一定要确保调用过service&version&impl方法
     */
    public void publish() {
        try {
            checkInited();
            putMethodSpecials();
            provider.init();
            published.compareAndSet(false, true);
        } catch (Exception e) {
            throw new RuntimeException(LightConstant.PUB_FAIL, e);
        }
    }

    public ProviderService serializeType(String serializeType){
        checkInited();
        if(serializeType.equals("java") || serializeType.equals("hessian"))
            provider.setSerializeType(serializeType);
        provider.setPreferSerializeType(serializeType);
        return this;
    }

    /**
     * 服务多注册
     * @param configServerNamespaces configserver的地域标识，详情见 http://gitlab.alibaba-inc.com/middleware/configserver/wikis/MultiCenter
     * @return
     */
    public ProviderService multiExport(List<String> configServerNamespaces){
        checkInited();
        provider.setConfigserverCenter(configServerNamespaces);
        return this;
    }


    /**
     * 是否已经published
     */
    public boolean isPublished(){
        String uniqueServiceName = this.service + ":" + this.version;
        return ApplicationModel.instance().getProvidedServiceModel(uniqueServiceName)!=null;
    }
    /**
     * 下线服务
     */
    public void offline() {
        checkInited();
        if (published.compareAndSet(true, false))
            provider.unregister();
    }

    /**
     * 上线服务
     */
    public void online() {
        checkInited();
        if (published.compareAndSet(false, true))
            provider.register();
    }


    /**
     * 判断是否发布服务成功
     *
     * @return
     */
    public boolean success() {
        return Partner.publishSuccess(this.provider);
    }

    private void putMethodSpecials() {
        synchronized (methodSpecials) {
            if (methodSpecials.size() > 0) {
                /**
                 * 坑爹啊，直接toArray回丢失类型
                 */
                MethodSpecial[] specials = new MethodSpecial[methodSpecials.size()];
                for (int i = 0; i < methodSpecials.size(); i++) {
                    specials[i] = methodSpecials.get(i);
                }
                provider.setMethodSpecials(specials);
            }
        }
    }

    private void checkInited() {
        if (inited.compareAndSet(true, true))
            return;
        else
            throw new RuntimeException(LightConstant.NOT_INIT_BEAN);
    }

    private Class<?> initProviderClass() {
        try {
            return Class.forName("com.taobao.hsf.app.spring.util.HSFSpringProviderBean", false, ServiceFactory.class.getClassLoader());
        } catch (ClassNotFoundException e) {
            throw new RuntimeException(LightConstant.INIT_P_FAIL, e);
        }
    }

}

