package com.taobao.hsf.standalone;

import com.taobao.hsf.standalone.sar.HSFSarUtil;
import com.taobao.hsf.standalone.util.Constant;
import com.taobao.hsf.standalone.util.IOUtil;

import java.io.File;
import java.util.Map;

/**
 * 脱离Web容器使用HSF的工具类。
 * <p>
 * 工作原理：通过自定义ClassLoader将HSF提供给用户使用的类挂到bootstrap loader和ext loader之间
 * </p>
 * <p>
 * 注意事项：一定要在使用任何HSF相关（包括通过HSF导出的diamond，notify，eagleeye等）类之前启动调用start方法启动，不然会有类加载冲突
 * </p>
 *
 * @author gaozhan@taobao.com
 * @author sixie.xyn
 * @author lanshan
 */
public class HSFEasyStarter {

    public static final String DEFAULT_LIGHTAPI_HOME = System.getProperty("user.home") + "/lightapi";

    public static final String DEFAULT_RELEASE_PATH = DEFAULT_LIGHTAPI_HOME + "/release";

    /**
     * 全部使用默认参数启动 hsf-standalone：
     * <ul>
     * <li>sar包路径：<b>${user.home}/lightapi/release/taobao-hsf.sar</b></li>
     * <li>sar包版本：<a href="http://ops.jm.taobao.org:9999/pandora-web/index.html"> Pandora运维系统</a> 上的推荐版本</li>
     * </ul>
     */
    @Deprecated
    public static void start() {
        startWithVersionAndIdentifier(Constant.version.get(), "default");
    }

    /**
     * sar包存放在 <b>[releasePath]/[version]/taobao-hsf.sar</b> 目录，当本地目录没有sar包时，会尝试下载。
     * <ul>
     * <li>若 releasePath 为""，则默认使用 <b>${user.home}/lightapi/release/</b></li>
     * <li>若 version 为""，则默认使用推荐版本的sar包，路径默认使用 <b>[releasePath]/taobao-hsf.sar</b></li>
     * </ul>
     *
     * @param releasePath 本地硬盘存放sar包的路径，<b>路径中不能包含空格</b>。
     * @param version 使用hsf的版本
     */
    public static void start(String releasePath, String version) {
        if (releasePath == null || version == null) {
            throw new IllegalArgumentException("releasePath or version can't be null");
        }

        // 准备sar包的base路径
        if (releasePath.trim().length() == 0) {
            releasePath = DEFAULT_RELEASE_PATH;
        }
        String sarBasePath = IOUtil.buildDirPath(releasePath, version);
        String sarPath = null;
        try {
            // 在base路径下准备sar包
            sarPath = HSFSarUtil.prepareHSFSar(sarBasePath, version);
            if (sarPath == null) {
                throw new RuntimeException("Prepare taobao-hsf.sar[" + version + "] under path[" + sarBasePath + "] failed");
            }
            // 启动pandora并hack classloader
            HSFMiniContainer.start(sarPath);
        } catch (Exception e) {
            throw new RuntimeException("HSFMiniContainer.start(sarPath) fail, please check your sar under path: " + sarPath, e);
        }
    }

    /**
     * 从"taobao-hsf.sar"的父目录启动主流版本hsf，若目录下没有taobao-hsf.sar，则尝试下载推荐版本的sar。
     *
     * @param releasePath "taobao-hsf.sar"的父目录
     */
    public static void startFromPath(String releasePath) {
        start(releasePath, "");
    }

    /**
     * 让用户指定版本启动HSF。</br>
     * 下载sar包到 <b>${user.home}/lightapi/release/[version]</b>
     *
     * @param version
     */
    public static void startWithVersion(String version) {
        startWithVersionAndIdentifier(version, "default");
    }

    /**
     * 指定用户ID启动。</br>
     * 下载推荐版本的sar到 <b>${user.home}/lightapi/release/taobao-hsf.sar</b>
     *
     * @param identifier 用户id，保证唯一性
     */
    @Deprecated
    public static void startWithIdentifier(String identifier) {
        startWithVersionAndIdentifier("", identifier);
    }

    /**
     * 指定版本和用户ID，适用于单机起多个standalone。
     * <ul>
     * <li>在 <b>${user.home}/lightapi/release/[version]/</b> 目录下寻找taobao-hsf.sar, 如果没有找到，则将尝试下载。</li>
     * <li>hsf.configuration 位于 <b>${user.home}/lightapi/osgidir/[identifier]/</b></li>
     * <li>hsf.log 位于 <b>${user.home}/lightapi/logs/[identifier]/</b></li>
     * <li>pandora临时目录 位于<b>${user.home}/lightapi/pandora/[identifier]/</b></li>
     * </ul>
     *
     * @param version
     * @param identifier 用户ID，注意要保证唯一性
     */
    public static void startWithVersionAndIdentifier(String version, String identifier) {
        String standaloneHome = DEFAULT_LIGHTAPI_HOME;
        String releasePath = DEFAULT_RELEASE_PATH;

        multiProcess(standaloneHome, identifier);
        start(releasePath, version);
    }

    /**
     * 指定版本、用户ID、ip和端口，适用于单机起多个standalone。
     * <ul>
     * <li>在 <b>${user.home}/lightapi/release/[version]/</b> 目录下寻找taobao-hsf.sar, 如果没有找到，则将尝试下载。</li>
     * <li>hsf.configuration 位于 <b>${user.home}/lightapi/osgidir/[identifier]/</b></li>
     * <li>hsf.log 位于 <b>${user.home}/lightapi/logs/[identifier]/</b></li>
     * <li>pandora临时目录 位于<b>${user.home}/lightapi/pandora/[identifier]/</b></li>
     * </ul>
     * 
     * @param version
     * @param identifier
     * @param port
     */
    public static void startWithVersionAndIdentifier(String version, String identifier, int port) {
        String standaloneHome = DEFAULT_LIGHTAPI_HOME;
        String releasePath = DEFAULT_RELEASE_PATH;

        multiProcess(standaloneHome, identifier);

        if (port != -1) {
            System.setProperty("hsf.server.port", Integer.toString(port));
        }
        start(releasePath, version);
    }

    /**
     * 设置单机启多个standalone的 hsf.configuration、hsf.log、pandora插件临时解压目录。<br />
     * 如果identifier可以转化为整型，则设置为pandora.qos.port。
     * 
     * @param standaloneHome
     * @param identifier
     */
    private static void multiProcess(String standaloneHome, String identifier) {
        String osgiDir = standaloneHome + "/osgidir/" + identifier;
        String logsDir = standaloneHome + "/logs/" + identifier;
        String pandoraDir = standaloneHome + "/pandora/" + identifier;

        System.setProperty("hsf.configuration.dir", osgiDir);
        System.setProperty("HSF.LOG.PATH", logsDir);
        System.setProperty("com.taobao.pandora.tmp_path", pandoraDir);
        System.setProperty("pandora.qos.port", "0");

        // add shutdown hook for cleaning tmp dirs.
        Runtime.getRuntime().addShutdownHook(new TmpDirCleaner(osgiDir, pandoraDir));
    }

    /**
     * 单机多standalone场景下，清除临时文件夹的shutdown hook。
     * 
     * @author xiaozi 2015年7月13日 下午3:16:24
     */
    private static class TmpDirCleaner extends Thread {

        private String osgiDirPath;

        private String pandoraDirPath;

        public TmpDirCleaner(String osgiDirPath, String pandoraDirPath) {
            this.osgiDirPath = osgiDirPath;
            this.pandoraDirPath = pandoraDirPath;
        }

        @Override
        public void run() {
            IOUtil.deleteDir(new File(osgiDirPath));
            IOUtil.deleteDir(new File(pandoraDirPath));
        }
    }

    /**
     * 让用户不依赖spring获取SpringConsumerBean生成的代理
     *
     * @param interfaceName
     * @param serviceVersion
     * @param group
     * @return
     */
    @Deprecated
    public static Object getRemotingServiceWithoutSpring(String interfaceName, String serviceVersion, String group) {
        return HSFStarter.getRemotingServiceWithoutSpring(interfaceName, serviceVersion, group);
    }

    /**
     * 让用户不依赖spring发布服务
     *
     * @param interfaceName
     * @param serviceVersion
     * @param serviceInstance
     * @param stringSetterParam
     * @return
     */
    @Deprecated
    public static Object createProviderWithoutSpring(String interfaceName, String serviceVersion,
                                                     Object serviceInstance, Map<String, String> stringSetterParam) {
        return HSFStarter.createProviderWithoutSpring(interfaceName, serviceVersion, serviceInstance,
                                                      stringSetterParam);
    }

    /**
     * 设置下载sar包的url，用于让用户自己搭建HSF sar包服务器
     *
     * @param downloadUrl
     */
    public static void setDownloadUrl(String downloadUrl) {
        if (downloadUrl == null || downloadUrl.trim().length() == 0) {
            throw new IllegalArgumentException("downloadUrl cannot be null or empty");
        }
        Constant.perferDownloadUrl.set(downloadUrl);
    }

    /**
     * 设置后，无参start()方法将以此version启动HSF
     *
     * @param version
     */
    public static void setVersion(String version) {
        if (version == null) {
            throw new IllegalArgumentException("version cannot be null");
        }
        Constant.version.set(version);
    }

}
