/*
 * Copyright 2008-2018 Ping Identity Corporation
 * All Rights Reserved.
 */
/*
 * Copyright (C) 2008-2018 Ping Identity Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License (GPLv2 only)
 * or the terms of the GNU Lesser General Public License (LGPLv2.1 only)
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses>.
 */
package com.unboundid.ldap.matchingrules;



import com.unboundid.asn1.ASN1OctetString;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.StaticUtils;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;

import static com.unboundid.ldap.matchingrules.MatchingRuleMessages.*;



/**
 * This class provides an implementation of a matching rule that performs
 * equality comparisons against Boolean values, which should be either "TRUE" or
 * "FALSE".  Substring and ordering matching are not supported.
 */
@ThreadSafety(level=ThreadSafetyLevel.COMPLETELY_THREADSAFE)
public final class BooleanMatchingRule
       extends MatchingRule
{
  /**
   * The singleton instance that will be returned from the {@code getInstance}
   * method.
   */
  private static final BooleanMatchingRule INSTANCE =
       new BooleanMatchingRule();



  /**
   * The pre-defined value that will be used as the normalized representation
   * of a "TRUE" value.
   */
  private static final ASN1OctetString TRUE_VALUE = new ASN1OctetString("TRUE");



  /**
   * The pre-defined value that will be used as the normalized representation
   * of a "FALSE" value.
   */
  private static final ASN1OctetString FALSE_VALUE =
       new ASN1OctetString("FALSE");



  /**
   * The name for the booleanMatch equality matching rule.
   */
  public static final String EQUALITY_RULE_NAME = "booleanMatch";



  /**
   * The name for the booleanMatch equality matching rule, formatted in all
   * lowercase characters.
   */
  static final String LOWER_EQUALITY_RULE_NAME =
       StaticUtils.toLowerCase(EQUALITY_RULE_NAME);



  /**
   * The OID for the booleanMatch equality matching rule.
   */
  public static final String EQUALITY_RULE_OID = "2.5.13.13";



  /**
   * The serial version UID for this serializable class.
   */
  private static final long serialVersionUID = 5137725892611277972L;



  /**
   * Creates a new instance of this Boolean matching rule.
   */
  public BooleanMatchingRule()
  {
    // No implementation is required.
  }



  /**
   * Retrieves a singleton instance of this matching rule.
   *
   * @return  A singleton instance of this matching rule.
   */
  public static BooleanMatchingRule getInstance()
  {
    return INSTANCE;
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public String getEqualityMatchingRuleName()
  {
    return EQUALITY_RULE_NAME;
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public String getEqualityMatchingRuleOID()
  {
    return EQUALITY_RULE_OID;
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public String getOrderingMatchingRuleName()
  {
    return null;
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public String getOrderingMatchingRuleOID()
  {
    return null;
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public String getSubstringMatchingRuleName()
  {
    return null;
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public String getSubstringMatchingRuleOID()
  {
    return null;
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public boolean valuesMatch(final ASN1OctetString value1,
                             final ASN1OctetString value2)
         throws LDAPException
  {
    return normalize(value1).equals(normalize(value2));
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public boolean matchesSubstring(final ASN1OctetString value,
                                  final ASN1OctetString subInitial,
                                  final ASN1OctetString[] subAny,
                                  final ASN1OctetString subFinal)
         throws LDAPException
  {
    throw new LDAPException(ResultCode.INAPPROPRIATE_MATCHING,
                            ERR_BOOLEAN_SUBSTRING_MATCHING_NOT_SUPPORTED.get());
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public int compareValues(final ASN1OctetString value1,
                           final ASN1OctetString value2)
         throws LDAPException
  {
    throw new LDAPException(ResultCode.INAPPROPRIATE_MATCHING,
                            ERR_BOOLEAN_ORDERING_MATCHING_NOT_SUPPORTED.get());
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public ASN1OctetString normalize(final ASN1OctetString value)
         throws LDAPException
  {
    final byte[] valueBytes = value.getValue();

    if ((valueBytes.length == 4) &&
        ((valueBytes[0] == 'T') || (valueBytes[0] == 't')) &&
        ((valueBytes[1] == 'R') || (valueBytes[1] == 'r')) &&
        ((valueBytes[2] == 'U') || (valueBytes[2] == 'u')) &&
        ((valueBytes[3] == 'E') || (valueBytes[3] == 'e')))
    {
      return TRUE_VALUE;
    }
    else if ((valueBytes.length == 5) &&
             ((valueBytes[0] == 'F') || (valueBytes[0] == 'f')) &&
             ((valueBytes[1] == 'A') || (valueBytes[1] == 'a')) &&
             ((valueBytes[2] == 'L') || (valueBytes[2] == 'l')) &&
             ((valueBytes[3] == 'S') || (valueBytes[3] == 's')) &&
             ((valueBytes[4] == 'E') || (valueBytes[4] == 'e')))
    {
      return FALSE_VALUE;
    }
    else
    {
      throw new LDAPException(ResultCode.INVALID_ATTRIBUTE_SYNTAX,
                              ERR_BOOLEAN_INVALID_VALUE.get());
    }
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public ASN1OctetString normalizeSubstring(final ASN1OctetString value,
                                            final byte substringType)
         throws LDAPException
  {
    throw new LDAPException(ResultCode.INAPPROPRIATE_MATCHING,
                            ERR_BOOLEAN_SUBSTRING_MATCHING_NOT_SUPPORTED.get());
  }
}
