/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.aliyun.openservices.shade.com.alibaba.rocketmq.client.retry;

import java.util.concurrent.ThreadLocalRandom;

import com.aliyun.openservices.shade.com.google.common.base.Preconditions;
import com.aliyun.openservices.shade.com.alibaba.rocketmq.remoting.protocol.RemotingSysResponseCode;

public class DefaultProducerRetryStrategy implements RetryStrategy {

    private static final int MAX_ATTEMPTS_NUM = 5;
    /**
     * specifies the timeout of all RPC request.
     * must have a duration value greater than 0.
     */
    private long timeoutMillis;

    /**
     * specifies the maximum number of RPC attempts, including the original request.
     * Value greater than 1. Values greater than 5 are treated as 5 without being considered a validation error.
     */
    private int maxAttempts;
    /**
     * must have a duration value greater than 0.
     */
    private long initialBackoffMillis;
    /**
     * must have a duration value greater than 0.
     */
    private long maxBackoffMillis;
    /**
     * must be specified and must greater than 0.
     */
    private double backoffMultiplier;

    private long nextBackoffIntervalMillis;
    private volatile int attempts = 0;
    private volatile long deadline = 0;

    public DefaultProducerRetryStrategy() {
        this(3000, 3, 200, 1000, 2);
    }

    public DefaultProducerRetryStrategy(long timeoutMillis, int maxAttempts) {
        this(timeoutMillis, maxAttempts, 200, 1000, 2);
    }

    public DefaultProducerRetryStrategy(
        long timeoutMillis,
        int maxAttempts,
        long initialBackoffMillis,
        long maxBackoffMillis,
        double backoffMultiplier) {
        Preconditions.checkArgument(timeoutMillis > 0);
        Preconditions.checkArgument(maxAttempts > 1);
        Preconditions.checkArgument(initialBackoffMillis > 0);
        Preconditions.checkArgument(maxBackoffMillis > 0);
        Preconditions.checkArgument(backoffMultiplier > 0);
        this.timeoutMillis = timeoutMillis;
        this.maxAttempts = Math.min(maxAttempts, MAX_ATTEMPTS_NUM);
        this.initialBackoffMillis = initialBackoffMillis;
        this.maxBackoffMillis = maxBackoffMillis;
        this.backoffMultiplier = backoffMultiplier;
        this.nextBackoffIntervalMillis = initialBackoffMillis;
        this.deadline = System.currentTimeMillis() + this.timeoutMillis;
    }

    @Override
    public RetryStrategy clone() {
        return new DefaultProducerRetryStrategy(timeoutMillis, maxAttempts, initialBackoffMillis,
            maxBackoffMillis, backoffMultiplier);
    }

    @Override
    public int getRetries() {
        return attempts;
    }

    @Override
    public RetryPlan makeRetryDecision(int requestCode, Throwable ex) {
        boolean shouldRetry = false;
        long backoffMillis = 0L;
        long now = System.currentTimeMillis();
        int expire = (int)(deadline - now);
        if (expire < 0) {
            return new RetryPlan(false, 0);
        }

        if (maxAttempts > attempts + 1) {
            shouldRetry = true;
            backoffMillis = Math.min((long)(nextBackoffIntervalMillis * ThreadLocalRandom.current().nextDouble()), expire);
            nextBackoffIntervalMillis = Math.min((long)(nextBackoffIntervalMillis * backoffMultiplier),
                maxBackoffMillis);
        }

        ++attempts;

        return new RetryPlan(shouldRetry, backoffMillis);
    }

    @Override
    public RetryPlan makeRetryDecision(Throwable ex) {
        return makeRetryDecision(RemotingSysResponseCode.SYSTEM_ERROR, ex);
    }
}
