/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.aliyun.openservices.shade.com.alibaba.rocketmq.remoting.vtoa;

import com.aliyun.openservices.shade.io.netty.channel.Channel;
import com.aliyun.openservices.shade.io.netty.channel.ChannelHandlerContext;
import com.aliyun.openservices.shade.io.netty.channel.socket.DefaultSocketChannelConfig;
import com.aliyun.openservices.shade.io.netty.channel.socket.nio.NioSocketChannel;
import java.lang.reflect.Field;
import java.net.Socket;
import com.aliyun.openservices.shade.com.alibaba.rocketmq.logging.InternalLogger;
import com.aliyun.openservices.shade.com.alibaba.rocketmq.logging.InternalLoggerFactory;
import com.aliyun.openservices.shade.com.alibaba.rocketmq.remoting.common.RemotingHelper;

public class VpcTunnelUtils {
    private static final InternalLogger log = InternalLoggerFactory.getLogger(RemotingHelper.ROCKETMQ_REMOTING);
    public static final String PROPERTY_VTOA_TUNNEL_ID = "VTOA_TUNNEL_ID";
    public static final String PROPERTY_VTOA_CLIENT_ADDR = "VTOA_CLIENT_ADDR";
    public static final String PROPERTY_VTOA_CLIENT_PORT = "VTOA_CLIENT_PORT";

    public native static int getvip(int fd, Vtoa v);

    private volatile static VpcTunnelUtils instance;

    private VpcTunnelUtils() {
        System.loadLibrary("getvip");
    }

    public static VpcTunnelUtils getInstance() {
        if (instance == null) {
            synchronized (VpcTunnelUtils.class) {
                if (instance == null) {
                    instance = new VpcTunnelUtils();
                }
            }
        }
        return instance;
    }

    public Vtoa getVtoa(ChannelHandlerContext ctx) {
        return getVtoa(ctx.channel());
    }

    public Vtoa getVtoa(Channel channel) {
        Vtoa vtoa = new Vtoa(-1, -1, -1);
        int result = getvip(getSocketFd(channel), vtoa);
        if (result == 0) {
            log.debug("Get tunnel_id from vtoa success: resultCode={}, vid={}, vaddr={}, vport={}",result, vtoa.getVid(), vtoa.getVaddr(), vtoa.getVport());
        } else {
            log.debug("Get tunnel_id from vtoa error: resultCode={}, vid={}, vaddr={}, vport={}", result, vtoa.getVid(), vtoa.getVaddr(), vtoa.getVport());
        }

        return vtoa;
    }

    /**
     * Get Vtoa by fd, used in other repository.
     * @param fd
     * @return Vtoa that include internal information
     */
    public Vtoa getVtoa(int fd) {
        Vtoa vtoa = new Vtoa(-1, -1, -1);
        int result = getvip(fd, vtoa);
        if (result == 0) {
            log.debug("Get tunnel_id from vtoa success: resultCode={}, vid={}, vaddr={}, vport={}",result, vtoa.getVid(), vtoa.getVaddr(), vtoa.getVport());
        } else {
            log.debug("Get tunnel_id from vtoa error: resultCode={}, vid={}, vaddr={}, vport={}", result, vtoa.getVid(), vtoa.getVaddr(), vtoa.getVport());
        }

        return vtoa;
    }

    /**
     * Fetch socket fd from Netty Channel
     *
     * @param channel
     * @return
     */
    private int getSocketFd(Channel channel) {
        try {
            if (!(channel instanceof NioSocketChannel)) {
                log.warn("Channel is not instance of NioSocketChannel");
                return 0;
            }
            NioSocketChannel nioChannel = (NioSocketChannel) channel;

            // NioSocketChannel config method return NioSocketChannelConfig whose class is private, so use
            // its super class type instead. And variable config is actual instance of NioSocketChannel.
            DefaultSocketChannelConfig config = (DefaultSocketChannelConfig) nioChannel.config();
            Field socketField = config.getClass().getSuperclass().getDeclaredField("javaSocket");
            socketField.setAccessible(true);
            Object socketValue = socketField.get(config);
            Socket socket = (Socket) socketValue;

            /* socket channel */
            java.nio.channels.SocketChannel socketChannel = socket.getChannel();
            Field fdValField = socketChannel.getClass().getDeclaredField("fdVal");
            fdValField.setAccessible(true);
            return fdValField.getInt(socketChannel);
        } catch (NoSuchFieldException e) {
            log.error("Get socket field failed. ", e);
        } catch (IllegalAccessException e) {
            log.error("Get socket field failed. ", e);
        } catch (Exception e) {
            log.error("Get socket field failed. ", e);
        }

        return 0;
    }
}