package com.jcloud.jcq.sdk;

import com.jcloud.jcq.sdk.auth.UserCredential;
import com.jcloud.jcq.client.Exception.ClientException;
import com.jcloud.jcq.client.consumer.ConsumerFactory;
import com.jcloud.jcq.client.consumer.SubscribeConsumer;
import com.jcloud.jcq.client.producer.ProducerFactory;
import com.jcloud.jcq.common.message.MessageType;
import com.jcloud.jcq.sdk.consumer.Consumer;
import com.jcloud.jcq.sdk.consumer.ConsumerConfig;
import com.jcloud.jcq.sdk.consumer.PullConsumer;
import com.jcloud.jcq.sdk.consumer.PullConsumerConfig;
import com.jcloud.jcq.sdk.consumer.impl.DefaultConsumerImpl;
import com.jcloud.jcq.sdk.consumer.impl.DefaultPullConsumerImpl;
import com.jcloud.jcq.sdk.producer.GlobalOrderProducer;
import com.jcloud.jcq.sdk.producer.Producer;
import com.jcloud.jcq.sdk.producer.ProducerConfig;
import com.jcloud.jcq.sdk.producer.impl.DefaultGlobalOrderProducerImpl;
import com.jcloud.jcq.sdk.producer.impl.DefaultProducerImpl;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * 客户端角色工厂类.
 *
 * @author wanglitao
 */
public class JCQClientFactory {
    private static final JCQClientFactory instance = new JCQClientFactory();
    private ProducerFactory producerFactory;
    private ConsumerFactory consumerFactory;
    private ConcurrentMap<com.jcloud.jcq.client.producer.Producer, Producer> producers = new ConcurrentHashMap<>();
    private ConcurrentMap<com.jcloud.jcq.client.producer.Producer, GlobalOrderProducer> orderProducers = new ConcurrentHashMap<>();
    private ConcurrentMap<com.jcloud.jcq.client.consumer.Consumer, Consumer> consumers = new ConcurrentHashMap<>();
    private ConcurrentMap<com.jcloud.jcq.client.consumer.Consumer, PullConsumer> pullConsumers = new ConcurrentHashMap<>();

    private JCQClientFactory() {
        producerFactory = ProducerFactory.getInstance();
        consumerFactory = ConsumerFactory.getInstance();
    }

    public static JCQClientFactory getInstance() {
        return instance;
    }

    /**
     * 创建普通生产者实例
     *
     * @param userCredential userCredential
     * @param producerConfig producerConfig
     * @return Producer
     * @throws ClientException ClientException
     */
    public synchronized Producer createProducer(final UserCredential userCredential, final ProducerConfig producerConfig) throws ClientException {
        com.jcloud.jcq.client.producer.ProducerConfig configOfNativeClient = toNativeClientConfig(producerConfig);
        configOfNativeClient.setToken(userCredential.getToken());
        com.jcloud.jcq.client.producer.Producer producer = producerFactory.createProducer(
                userCredential.getAccessKey(),
                userCredential.getSecretKey(),
                configOfNativeClient);
        Producer sdkProducer = producers.get(producer);
        if (sdkProducer == null) {
            sdkProducer = new DefaultProducerImpl(producer);
            producers.put(producer, sdkProducer);
        }
        return sdkProducer;
    }

    /**
     * 创建全局顺序消息生产者实例
     *
     * @param userCredential userCredential
     * @param producerConfig producerConfig
     * @return GlobalOrderProducer
     * @throws ClientException ClientException
     */
    public synchronized GlobalOrderProducer createGlobalOrderProducer(final UserCredential userCredential, final ProducerConfig producerConfig) throws ClientException {
        com.jcloud.jcq.client.producer.ProducerConfig configOfNativeClient = toNativeClientConfig(producerConfig);
        configOfNativeClient.setMessageType(MessageType.ORDER);
        configOfNativeClient.setToken(userCredential.getToken());
        com.jcloud.jcq.client.producer.Producer producer = producerFactory.createProducer(
                userCredential.getAccessKey(),
                userCredential.getSecretKey(),
                configOfNativeClient);
        GlobalOrderProducer globalOrderProducer = orderProducers.get(producer);
        if (globalOrderProducer == null) {
            globalOrderProducer = new DefaultGlobalOrderProducerImpl(producer);
            orderProducers.put(producer, globalOrderProducer);
        }
        return globalOrderProducer;
    }

    /**
     * 创建消费者
     *
     * @param userCredential userCredential
     * @param consumerConfig consumerConfig
     * @return Consumer
     * @throws ClientException ClientException
     */
    public synchronized Consumer createConsumer(final UserCredential userCredential, final ConsumerConfig consumerConfig) throws ClientException {
        com.jcloud.jcq.client.consumer.ConsumerConfig configOfNativeClient = toNativeClientConfig(consumerConfig);
        configOfNativeClient.setToken(userCredential.getToken());
        SubscribeConsumer consumer = consumerFactory.createSubscribeConsumer(
                userCredential.getAccessKey(),
                userCredential.getSecretKey(),
                configOfNativeClient);
        Consumer sdkConsumer = consumers.get(consumer);
        if (sdkConsumer == null) {
            sdkConsumer = new DefaultConsumerImpl(consumer);
            consumers.put(consumer, sdkConsumer);
        }
        return sdkConsumer;
    }

    /**
     * 创建GlobalFIFO消费者
     *
     * @param userCredential
     * @param consumerConfig
     * @return
     */
    public synchronized Consumer createGlobalFIFOConsumer(final UserCredential userCredential, final ConsumerConfig consumerConfig) throws ClientException {
        com.jcloud.jcq.client.consumer.ConsumerConfig configOfNativeClient = toNativeClientConfig(consumerConfig);
        configOfNativeClient.setToken(userCredential.getToken());
        // GlobalFIFO消费者batch size固定设成1, 无论用户设成多少
        configOfNativeClient.setRecommendedBatchSizePerPush(1);
        SubscribeConsumer consumer = consumerFactory.createSubscribeConsumer(
                userCredential.getAccessKey(),
                userCredential.getSecretKey(),
                configOfNativeClient);
        Consumer sdkConsumer = consumers.get(consumer);
        if (sdkConsumer == null) {
            sdkConsumer = new DefaultConsumerImpl(consumer);
            consumers.put(consumer, sdkConsumer);
        }
        return sdkConsumer;
    }

    /**
     * 创建消费者
     *
     * @param userCredential userCredential
     * @param pullConsumerConfig pullConsumerConfig
     * @return PullConsumer
     * @throws ClientException ClientException
     */
    public synchronized PullConsumer createPullConsumer(final UserCredential userCredential, final PullConsumerConfig pullConsumerConfig) throws ClientException {
        com.jcloud.jcq.client.consumer.ConsumerConfig configOfNativeClient = toNativeClientConfig(pullConsumerConfig);
        configOfNativeClient.setToken(userCredential.getToken());
        com.jcloud.jcq.client.consumer.PullConsumer pullConsumer = consumerFactory.createPullConsumer(
                userCredential.getAccessKey(),
                userCredential.getSecretKey(),
                configOfNativeClient);
        PullConsumer sdkPullConsumer = pullConsumers.get(pullConsumer);
        if (sdkPullConsumer == null) {
            sdkPullConsumer = new DefaultPullConsumerImpl(pullConsumer);
            pullConsumers.put(pullConsumer, sdkPullConsumer);
        }
        return sdkPullConsumer;
    }

    /**
     * 创建GlobalFIFO Pull消费者
     *
     * @param userCredential
     * @param pullConsumerConfig
     * @return
     */
    public synchronized PullConsumer createGlobalFIFOPullConsumer(final UserCredential userCredential, final PullConsumerConfig pullConsumerConfig) throws ClientException {
        com.jcloud.jcq.client.consumer.ConsumerConfig configOfNativeClient = toNativeClientConfig(pullConsumerConfig);
        configOfNativeClient.setToken(userCredential.getToken());
        // GlobalFIFO消费者batch size固定设成1, 无论用户设成多少
        configOfNativeClient.setRecommendedBatchSizePerPush(1);
        com.jcloud.jcq.client.consumer.PullConsumer pullConsumer = consumerFactory.createPullConsumer(
                userCredential.getAccessKey(),
                userCredential.getSecretKey(),
                configOfNativeClient);
        PullConsumer sdkPullConsumer = pullConsumers.get(pullConsumer);
        if (sdkPullConsumer == null) {
            sdkPullConsumer = new DefaultPullConsumerImpl(pullConsumer);
            pullConsumers.put(pullConsumer, sdkPullConsumer);
        }
        return sdkPullConsumer;
    }

    /**
     * sdk ConsumerConfig到原生client的ConsumerConfig转换
     *
     * @param consumerConfig ConsumerConfig
     * @return ConsumerConfig
     */
    private com.jcloud.jcq.client.consumer.ConsumerConfig toNativeClientConfig(ConsumerConfig consumerConfig) {
        com.jcloud.jcq.client.consumer.ConsumerConfig configOfNativeClient = new com.jcloud.jcq.client.consumer.ConsumerConfig();
        configOfNativeClient.setConsumerGroupId(consumerConfig.getConsumerGroupId());
        configOfNativeClient.setMetaServerAddress(consumerConfig.getMetaServerAddress());
        configOfNativeClient.setMaxRetryTimes(consumerConfig.getMaxRetryTimes());
        configOfNativeClient.setDefaultConsumePosition(consumerConfig.getDefaultConsumePosition());
        configOfNativeClient.setRecommendedBatchSizePerPush(consumerConfig.getMaxBatchSizePerPush());
        configOfNativeClient.setConsumePoolCoreSize(consumerConfig.getConsumePoolCoreSize());
        configOfNativeClient.setMessageBufferSize(consumerConfig.getMessageBufferSize());
        configOfNativeClient.setAckPoolCoreSize(consumerConfig.getAckPoolCoreSize());
        configOfNativeClient.setAckBufferSize(consumerConfig.getAckBufferSize());
        configOfNativeClient.setConsumeMaxRetryTimes(consumerConfig.getConsumeMaxRetryTimes());
        configOfNativeClient.setMessageTraceOn(consumerConfig.isEnableMessageTrace());
        configOfNativeClient.setRemoteCallRetryWaitTime(consumerConfig.getRemoteCallRetryWaitTime());
        configOfNativeClient.setRefreshRouteRetryWaitTime(consumerConfig.getRefreshRouteRetryWaitTime());
        return configOfNativeClient;
    }

    /**
     * sdk PullConsumerConfig到原生client的ConsumerConfig转换
     *
     * @param pullConsumerConfig pullConsumerConfig
     * @return ConsumerConfig
     */
    private com.jcloud.jcq.client.consumer.ConsumerConfig toNativeClientConfig(PullConsumerConfig pullConsumerConfig) {
        com.jcloud.jcq.client.consumer.ConsumerConfig configOfNativeClient = new com.jcloud.jcq.client.consumer.ConsumerConfig();
        configOfNativeClient.setConsumerGroupId(pullConsumerConfig.getConsumerGroupId());
        configOfNativeClient.setMetaServerAddress(pullConsumerConfig.getMetaServerAddress());
        configOfNativeClient.setMaxRetryTimes(pullConsumerConfig.getMaxRetryTimes());
        configOfNativeClient.setDefaultConsumePosition(pullConsumerConfig.getDefaultConsumePosition());
        configOfNativeClient.setRecommendedBatchSizePerPush(pullConsumerConfig.getMaxBatchSizePerPull());
        configOfNativeClient.setMessageTraceOn(pullConsumerConfig.isEnableMessageTrace());
        configOfNativeClient.setRemoteCallRetryWaitTime(pullConsumerConfig.getRemoteCallRetryWaitTime());
        configOfNativeClient.setRefreshRouteRetryWaitTime(pullConsumerConfig.getRefreshRouteRetryWaitTime());
        return configOfNativeClient;
    }

    /**
     * sdk ProducerConfig到原生client的ProducerConfig转换
     *
     * @param producerConfig ProducerConfig
     * @return ProducerConfig
     */
    private com.jcloud.jcq.client.producer.ProducerConfig toNativeClientConfig(ProducerConfig producerConfig) {
        com.jcloud.jcq.client.producer.ProducerConfig configOfNativeClient = new com.jcloud.jcq.client.producer.ProducerConfig();
        configOfNativeClient.setMetaServerAddress(producerConfig.getMetaServerAddress());
        configOfNativeClient.setMaxRetryTimes(producerConfig.getMaxRetryTimes());
        configOfNativeClient.setEnableCompress(producerConfig.isEnableCompress());
        configOfNativeClient.setMessageTraceOn(producerConfig.isEnableMessageTrace());
        configOfNativeClient.setShowInfoLevelSendMsgLog(producerConfig.isShowInfoLevelSendMsgLog());
        configOfNativeClient.setRemoteCallRetryWaitTime(producerConfig.getRemoteCallRetryWaitTime());
        configOfNativeClient.setRefreshRouteRetryWaitTime(producerConfig.getRefreshRouteRetryWaitTime());
        configOfNativeClient.setMaxMsgSize(producerConfig.getMaxMsgSize());
        configOfNativeClient.setSendTimeout(producerConfig.getSendTimeout());
        return configOfNativeClient;
    }
}
