/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.checkpoint;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.flink.api.common.JobStatus;
import org.apache.flink.runtime.checkpoint.CheckpointStoreUtil;
import org.apache.flink.runtime.checkpoint.CheckpointSubsumeHelper;
import org.apache.flink.runtime.checkpoint.CheckpointsCleaner;
import org.apache.flink.runtime.checkpoint.CompletedCheckpoint;
import org.apache.flink.runtime.checkpoint.CompletedCheckpointStore;
import org.apache.flink.runtime.persistence.ResourceVersion;
import org.apache.flink.runtime.persistence.StateHandleStore;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultCompletedCheckpointStore<R extends ResourceVersion<R>>
implements CompletedCheckpointStore {
    private static final Logger LOG = LoggerFactory.getLogger(DefaultCompletedCheckpointStore.class);
    private final StateHandleStore<CompletedCheckpoint, R> checkpointStateHandleStore;
    private final int maxNumberOfCheckpointsToRetain;
    private final ArrayDeque<CompletedCheckpoint> completedCheckpoints;
    private final Executor ioExecutor;
    private final CheckpointStoreUtil completedCheckpointStoreUtil;
    private final AtomicBoolean running = new AtomicBoolean(true);

    public DefaultCompletedCheckpointStore(int maxNumberOfCheckpointsToRetain, StateHandleStore<CompletedCheckpoint, R> stateHandleStore, CheckpointStoreUtil completedCheckpointStoreUtil, Collection<CompletedCheckpoint> completedCheckpoints, Executor executor) {
        Preconditions.checkArgument((maxNumberOfCheckpointsToRetain >= 1 ? 1 : 0) != 0, (Object)"Must retain at least one checkpoint.");
        this.maxNumberOfCheckpointsToRetain = maxNumberOfCheckpointsToRetain;
        this.checkpointStateHandleStore = (StateHandleStore)Preconditions.checkNotNull(stateHandleStore);
        this.completedCheckpoints = new ArrayDeque(maxNumberOfCheckpointsToRetain + 1);
        this.completedCheckpoints.addAll(completedCheckpoints);
        this.ioExecutor = (Executor)Preconditions.checkNotNull((Object)executor);
        this.completedCheckpointStoreUtil = (CheckpointStoreUtil)Preconditions.checkNotNull((Object)completedCheckpointStoreUtil);
    }

    @Override
    public boolean requiresExternalizedCheckpoints() {
        return true;
    }

    @Override
    public void addCheckpoint(CompletedCheckpoint checkpoint, CheckpointsCleaner checkpointsCleaner, Runnable postCleanup) throws Exception {
        Preconditions.checkState((boolean)this.running.get(), (Object)"Checkpoint store has already been shutdown.");
        Preconditions.checkNotNull((Object)checkpoint, (String)"Checkpoint");
        String path = this.completedCheckpointStoreUtil.checkpointIDToName(checkpoint.getCheckpointID());
        this.checkpointStateHandleStore.addAndLock(path, checkpoint);
        this.completedCheckpoints.addLast(checkpoint);
        CheckpointSubsumeHelper.subsume(this.completedCheckpoints, this.maxNumberOfCheckpointsToRetain, completedCheckpoint -> this.tryRemoveCompletedCheckpoint(completedCheckpoint, completedCheckpoint.shouldBeDiscardedOnSubsume(), checkpointsCleaner, postCleanup));
        LOG.debug("Added {} to {}.", (Object)checkpoint, (Object)path);
    }

    @Override
    public List<CompletedCheckpoint> getAllCheckpoints() {
        return new ArrayList<CompletedCheckpoint>(this.completedCheckpoints);
    }

    @Override
    public int getNumberOfRetainedCheckpoints() {
        return this.completedCheckpoints.size();
    }

    @Override
    public int getMaxNumberOfRetainedCheckpoints() {
        return this.maxNumberOfCheckpointsToRetain;
    }

    @Override
    public void shutdown(JobStatus jobStatus, CheckpointsCleaner checkpointsCleaner) throws Exception {
        if (this.running.compareAndSet(true, false)) {
            if (jobStatus.isGloballyTerminalState()) {
                LOG.info("Shutting down");
                for (CompletedCheckpoint checkpoint : this.completedCheckpoints) {
                    try {
                        this.tryRemoveCompletedCheckpoint(checkpoint, checkpoint.shouldBeDiscardedOnShutdown(jobStatus), checkpointsCleaner, () -> {});
                    }
                    catch (Exception e) {
                        LOG.warn("Fail to remove checkpoint during shutdown.", (Throwable)e);
                    }
                }
                this.completedCheckpoints.clear();
                this.checkpointStateHandleStore.clearEntries();
            } else {
                LOG.info("Suspending");
                this.completedCheckpoints.clear();
                this.checkpointStateHandleStore.releaseAll();
            }
        }
    }

    private void tryRemoveCompletedCheckpoint(CompletedCheckpoint completedCheckpoint, boolean shouldDiscard, CheckpointsCleaner checkpointsCleaner, Runnable postCleanup) throws Exception {
        if (this.tryRemove(completedCheckpoint.getCheckpointID())) {
            checkpointsCleaner.cleanCheckpoint(completedCheckpoint, shouldDiscard, postCleanup, this.ioExecutor);
        }
    }

    private boolean tryRemove(long checkpointId) throws Exception {
        return this.checkpointStateHandleStore.releaseAndTryRemove(this.completedCheckpointStoreUtil.checkpointIDToName(checkpointId));
    }
}

