/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.checkpoint;

import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.ConcurrentModificationException;
import java.util.List;
import java.util.concurrent.Executor;
import javax.annotation.Nullable;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.runtime.checkpoint.CompletedCheckpoint;
import org.apache.flink.runtime.checkpoint.CompletedCheckpointStore;
import org.apache.flink.runtime.jobgraph.JobStatus;
import org.apache.flink.runtime.state.RetrievableStateHandle;
import org.apache.flink.runtime.zookeeper.RetrievableStateStorageHelper;
import org.apache.flink.runtime.zookeeper.ZooKeeperStateHandleStore;
import org.apache.flink.shaded.org.apache.curator.framework.CuratorFramework;
import org.apache.flink.shaded.org.apache.curator.utils.ZKPaths;
import org.apache.flink.util.FlinkException;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZooKeeperCompletedCheckpointStore
implements CompletedCheckpointStore {
    private static final Logger LOG = LoggerFactory.getLogger(ZooKeeperCompletedCheckpointStore.class);
    private final CuratorFramework client;
    private final ZooKeeperStateHandleStore<CompletedCheckpoint> checkpointsInZooKeeper;
    private final int maxNumberOfCheckpointsToRetain;
    private final ArrayDeque<CompletedCheckpoint> completedCheckpoints;

    public ZooKeeperCompletedCheckpointStore(int maxNumberOfCheckpointsToRetain, CuratorFramework client, String checkpointsPath, RetrievableStateStorageHelper<CompletedCheckpoint> stateStorage, Executor executor) throws Exception {
        Preconditions.checkArgument((maxNumberOfCheckpointsToRetain >= 1 ? 1 : 0) != 0, (Object)"Must retain at least one checkpoint.");
        Preconditions.checkNotNull(stateStorage, (String)"State storage");
        this.maxNumberOfCheckpointsToRetain = maxNumberOfCheckpointsToRetain;
        Preconditions.checkNotNull((Object)client, (String)"Curator client");
        Preconditions.checkNotNull((Object)checkpointsPath, (String)"Checkpoints path");
        client.newNamespaceAwareEnsurePath(checkpointsPath).ensure(client.getZookeeperClient());
        this.client = client.usingNamespace(client.getNamespace() + checkpointsPath);
        this.checkpointsInZooKeeper = new ZooKeeperStateHandleStore<CompletedCheckpoint>(this.client, stateStorage, executor);
        this.completedCheckpoints = new ArrayDeque(maxNumberOfCheckpointsToRetain + 1);
        LOG.info("Initialized in '{}'.", (Object)checkpointsPath);
    }

    @Override
    public boolean requiresExternalizedCheckpoints() {
        return true;
    }

    @Override
    public void recover() throws Exception {
        List<Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String>> initialCheckpoints;
        LOG.info("Recovering checkpoints from ZooKeeper.");
        this.completedCheckpoints.clear();
        while (true) {
            try {
                initialCheckpoints = this.checkpointsInZooKeeper.getAllSortedByNameAndLock();
            }
            catch (ConcurrentModificationException e) {
                LOG.warn("Concurrent modification while reading from ZooKeeper. Retrying.");
                continue;
            }
            break;
        }
        int numberOfInitialCheckpoints = initialCheckpoints.size();
        LOG.info("Found {} checkpoints in ZooKeeper.", (Object)numberOfInitialCheckpoints);
        for (Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String> checkpointStateHandle : initialCheckpoints) {
            CompletedCheckpoint completedCheckpoint = null;
            try {
                completedCheckpoint = ZooKeeperCompletedCheckpointStore.retrieveCompletedCheckpoint(checkpointStateHandle);
                if (completedCheckpoint == null) continue;
                this.completedCheckpoints.add(completedCheckpoint);
            }
            catch (Exception e) {
                LOG.warn("Could not retrieve checkpoint. Removing it from the completed checkpoint store.", (Throwable)e);
                this.removeBrokenStateHandle((String)checkpointStateHandle.f1, (RetrievableStateHandle)checkpointStateHandle.f0);
            }
        }
    }

    @Override
    public void addCheckpoint(CompletedCheckpoint checkpoint) throws Exception {
        Preconditions.checkNotNull((Object)checkpoint, (String)"Checkpoint");
        String path = ZooKeeperCompletedCheckpointStore.checkpointIdToPath(checkpoint.getCheckpointID());
        this.checkpointsInZooKeeper.addAndLock(path, checkpoint);
        this.completedCheckpoints.addLast(checkpoint);
        while (this.completedCheckpoints.size() > this.maxNumberOfCheckpointsToRetain) {
            try {
                this.removeSubsumed(this.completedCheckpoints.removeFirst());
            }
            catch (Exception e) {
                LOG.warn("Failed to subsume the old checkpoint", (Throwable)e);
            }
        }
        LOG.debug("Added {} to {}.", (Object)checkpoint, (Object)path);
    }

    @Override
    public CompletedCheckpoint getLatestCheckpoint() {
        if (this.completedCheckpoints.isEmpty()) {
            return null;
        }
        return this.completedCheckpoints.peekLast();
    }

    @Override
    public List<CompletedCheckpoint> getAllCheckpoints() throws Exception {
        ArrayList<CompletedCheckpoint> checkpoints = new ArrayList<CompletedCheckpoint>(this.completedCheckpoints);
        return checkpoints;
    }

    @Override
    public int getNumberOfRetainedCheckpoints() {
        return this.completedCheckpoints.size();
    }

    @Override
    public int getMaxNumberOfRetainedCheckpoints() {
        return this.maxNumberOfCheckpointsToRetain;
    }

    @Override
    public void shutdown(JobStatus jobStatus) throws Exception {
        if (jobStatus.isGloballyTerminalState()) {
            LOG.info("Shutting down");
            for (CompletedCheckpoint checkpoint : this.completedCheckpoints) {
                try {
                    this.removeShutdown(checkpoint, jobStatus);
                }
                catch (Exception e) {
                    LOG.error("Failed to discard checkpoint.", (Throwable)e);
                }
            }
            this.completedCheckpoints.clear();
            String path = "/" + this.client.getNamespace();
            LOG.info("Removing {} from ZooKeeper", (Object)path);
            ZKPaths.deleteChildren(this.client.getZookeeperClient().getZooKeeper(), path, true);
        } else {
            LOG.info("Suspending");
            this.completedCheckpoints.clear();
            this.checkpointsInZooKeeper.releaseAll();
        }
    }

    private void removeSubsumed(final CompletedCheckpoint completedCheckpoint) throws Exception {
        if (completedCheckpoint == null) {
            return;
        }
        ZooKeeperStateHandleStore.RemoveCallback<CompletedCheckpoint> action = new ZooKeeperStateHandleStore.RemoveCallback<CompletedCheckpoint>(){

            @Override
            public void apply(@Nullable RetrievableStateHandle<CompletedCheckpoint> value) throws FlinkException {
                if (value != null) {
                    try {
                        completedCheckpoint.discardOnSubsume();
                    }
                    catch (Exception e) {
                        throw new FlinkException("Could not discard the completed checkpoint on subsume.", (Throwable)e);
                    }
                }
            }
        };
        this.checkpointsInZooKeeper.releaseAndTryRemove(ZooKeeperCompletedCheckpointStore.checkpointIdToPath(completedCheckpoint.getCheckpointID()), action);
    }

    private void removeShutdown(final CompletedCheckpoint completedCheckpoint, final JobStatus jobStatus) throws Exception {
        if (completedCheckpoint == null) {
            return;
        }
        ZooKeeperStateHandleStore.RemoveCallback<CompletedCheckpoint> removeAction = new ZooKeeperStateHandleStore.RemoveCallback<CompletedCheckpoint>(){

            @Override
            public void apply(@Nullable RetrievableStateHandle<CompletedCheckpoint> value) throws FlinkException {
                try {
                    completedCheckpoint.discardOnShutdown(jobStatus);
                }
                catch (Exception e) {
                    throw new FlinkException("Could not discard the completed checkpoint on subsume.", (Throwable)e);
                }
            }
        };
        this.checkpointsInZooKeeper.releaseAndTryRemove(ZooKeeperCompletedCheckpointStore.checkpointIdToPath(completedCheckpoint.getCheckpointID()), removeAction);
    }

    private void removeBrokenStateHandle(String pathInZooKeeper, final RetrievableStateHandle<CompletedCheckpoint> retrievableStateHandle) throws Exception {
        this.checkpointsInZooKeeper.releaseAndTryRemove(pathInZooKeeper, new ZooKeeperStateHandleStore.RemoveCallback<CompletedCheckpoint>(){

            @Override
            public void apply(@Nullable RetrievableStateHandle<CompletedCheckpoint> value) throws FlinkException {
                try {
                    retrievableStateHandle.discardState();
                }
                catch (Exception e) {
                    throw new FlinkException("Could not discard state handle.", (Throwable)e);
                }
            }
        });
    }

    public static String checkpointIdToPath(long checkpointId) {
        return String.format("/%019d", checkpointId);
    }

    public static long pathToCheckpointId(String path) {
        try {
            String numberString = '/' == path.charAt(0) ? path.substring(1) : path;
            return Long.parseLong(numberString);
        }
        catch (NumberFormatException e) {
            LOG.warn("Could not parse checkpoint id from {}. This indicates that the checkpoint id to path conversion has changed.", (Object)path);
            return -1L;
        }
    }

    private static CompletedCheckpoint retrieveCompletedCheckpoint(Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String> stateHandlePath) throws FlinkException {
        long checkpointId = ZooKeeperCompletedCheckpointStore.pathToCheckpointId((String)stateHandlePath.f1);
        LOG.info("Trying to retrieve checkpoint {}.", (Object)checkpointId);
        try {
            return (CompletedCheckpoint)((RetrievableStateHandle)stateHandlePath.f0).retrieveState();
        }
        catch (ClassNotFoundException cnfe) {
            throw new FlinkException("Could not retrieve checkpoint " + checkpointId + " from state handle under " + (String)stateHandlePath.f1 + ". This indicates that you are trying to recover from state written by an older Flink version which is not compatible. Try cleaning the state handle store.", (Throwable)cnfe);
        }
        catch (IOException ioe) {
            throw new FlinkException("Could not retrieve checkpoint " + checkpointId + " from state handle under " + (String)stateHandlePath.f1 + ". This indicates that the retrieved state handle is broken. Try cleaning the state handle store.", (Throwable)ioe);
        }
    }
}

