/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.client.cli;

import java.io.ByteArrayOutputStream;
import java.io.IOError;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Supplier;
import javax.annotation.Nullable;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.core.execution.JobClient;
import org.apache.flink.table.api.TableResult;
import org.apache.flink.table.api.config.TableConfigOptions;
import org.apache.flink.table.api.internal.TableResultImpl;
import org.apache.flink.table.catalog.ResolvedSchema;
import org.apache.flink.table.client.SqlClientException;
import org.apache.flink.table.client.cli.CliChangelogResultView;
import org.apache.flink.table.client.cli.CliResultView;
import org.apache.flink.table.client.cli.CliStatementSplitter;
import org.apache.flink.table.client.cli.CliStrings;
import org.apache.flink.table.client.cli.CliTableResultView;
import org.apache.flink.table.client.cli.CliTableauResultView;
import org.apache.flink.table.client.cli.CliUtils;
import org.apache.flink.table.client.cli.SqlCompleter;
import org.apache.flink.table.client.cli.SqlMultiLineParser;
import org.apache.flink.table.client.cli.TerminalUtils;
import org.apache.flink.table.client.config.ResultMode;
import org.apache.flink.table.client.config.SqlClientOptions;
import org.apache.flink.table.client.config.YamlConfigUtils;
import org.apache.flink.table.client.gateway.Executor;
import org.apache.flink.table.client.gateway.ResultDescriptor;
import org.apache.flink.table.client.gateway.SqlExecutionException;
import org.apache.flink.table.operations.BeginStatementSetOperation;
import org.apache.flink.table.operations.CatalogSinkModifyOperation;
import org.apache.flink.table.operations.EndStatementSetOperation;
import org.apache.flink.table.operations.ExplainOperation;
import org.apache.flink.table.operations.LoadModuleOperation;
import org.apache.flink.table.operations.ModifyOperation;
import org.apache.flink.table.operations.Operation;
import org.apache.flink.table.operations.QueryOperation;
import org.apache.flink.table.operations.UnloadModuleOperation;
import org.apache.flink.table.operations.UseOperation;
import org.apache.flink.table.operations.command.ClearOperation;
import org.apache.flink.table.operations.command.HelpOperation;
import org.apache.flink.table.operations.command.QuitOperation;
import org.apache.flink.table.operations.command.ResetOperation;
import org.apache.flink.table.operations.command.SetOperation;
import org.apache.flink.table.operations.ddl.AlterOperation;
import org.apache.flink.table.operations.ddl.CreateOperation;
import org.apache.flink.table.operations.ddl.DropOperation;
import org.apache.flink.table.utils.PrintUtils;
import org.apache.flink.types.Row;
import org.apache.flink.util.Preconditions;
import org.jline.reader.EndOfFileException;
import org.jline.reader.LineReader;
import org.jline.reader.LineReaderBuilder;
import org.jline.reader.MaskingCallback;
import org.jline.reader.UserInterruptException;
import org.jline.terminal.Terminal;
import org.jline.utils.AttributedString;
import org.jline.utils.AttributedStringBuilder;
import org.jline.utils.AttributedStyle;
import org.jline.utils.InfoCmp;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CliClient
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(CliClient.class);
    public static final Supplier<Terminal> DEFAULT_TERMINAL_FACTORY = TerminalUtils::createDefaultTerminal;
    private final Executor executor;
    private final String sessionId;
    private final Path historyFilePath;
    private final String prompt;
    @Nullable
    private final MaskingCallback inputTransformer;
    private final Supplier<Terminal> terminalFactory;
    private Terminal terminal;
    private boolean isRunning;
    private boolean isStatementSetMode;
    private List<ModifyOperation> statementSetOperations;
    private static final int PLAIN_TERMINAL_WIDTH = 80;
    private static final int PLAIN_TERMINAL_HEIGHT = 30;

    @VisibleForTesting
    public CliClient(Supplier<Terminal> terminalFactory, String sessionId, Executor executor, Path historyFilePath, @Nullable MaskingCallback inputTransformer) {
        this.terminalFactory = terminalFactory;
        this.sessionId = sessionId;
        this.executor = executor;
        this.inputTransformer = inputTransformer;
        this.historyFilePath = historyFilePath;
        this.prompt = new AttributedStringBuilder().style(AttributedStyle.DEFAULT.foreground(2)).append("Flink SQL").style(AttributedStyle.DEFAULT).append("> ").toAnsi();
    }

    public CliClient(Supplier<Terminal> terminalFactory, String sessionId, Executor executor, Path historyFilePath) {
        this(terminalFactory, sessionId, executor, historyFilePath, null);
    }

    public Terminal getTerminal() {
        return this.terminal;
    }

    public String getSessionId() {
        return this.sessionId;
    }

    public void clearTerminal() {
        if (this.isPlainTerminal()) {
            for (int i = 0; i < 200; ++i) {
                this.terminal.writer().println();
            }
        } else {
            this.terminal.puts(InfoCmp.Capability.clear_screen, new Object[0]);
        }
    }

    public boolean isPlainTerminal() {
        return this.terminal.getWidth() == 0 && this.terminal.getHeight() == 0;
    }

    public int getWidth() {
        if (this.isPlainTerminal()) {
            return 80;
        }
        return this.terminal.getWidth();
    }

    public int getHeight() {
        if (this.isPlainTerminal()) {
            return 30;
        }
        return this.terminal.getHeight();
    }

    public Executor getExecutor() {
        return this.executor;
    }

    @Override
    public void close() {
        if (this.terminal != null) {
            this.closeTerminal();
        }
    }

    public void executeInInteractiveMode() {
        try {
            this.terminal = this.terminalFactory.get();
            this.executeInteractive();
        }
        finally {
            this.closeTerminal();
        }
    }

    public void executeInNonInteractiveMode(String content) {
        try {
            this.terminal = this.terminalFactory.get();
            this.executeFile(content, ExecutionMode.NON_INTERACTIVE_EXECUTION);
        }
        finally {
            this.closeTerminal();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean executeInitialization(String content) {
        try {
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream(256);
            this.terminal = TerminalUtils.createDumbTerminal(outputStream);
            boolean success = this.executeFile(content, ExecutionMode.INITIALIZATION);
            LOG.info(((Object)outputStream).toString());
            boolean bl = success;
            return bl;
        }
        finally {
            this.closeTerminal();
        }
    }

    private void executeInteractive() {
        this.isRunning = true;
        LineReader lineReader = this.createLineReader(this.terminal);
        this.terminal.writer().println();
        this.terminal.writer().flush();
        this.terminal.writer().append(CliStrings.MESSAGE_WELCOME);
        while (this.isRunning) {
            String line;
            this.terminal.writer().append("\n");
            this.terminal.flush();
            try {
                line = lineReader.readLine(this.prompt, null, this.inputTransformer, null);
            }
            catch (UserInterruptException e) {
                continue;
            }
            catch (IOError | EndOfFileException e) {
                break;
            }
            catch (Throwable t) {
                throw new SqlClientException("Could not read from command line.", t);
            }
            if (line == null) continue;
            this.executeStatement(line, ExecutionMode.INTERACTIVE_EXECUTION);
        }
    }

    private boolean executeFile(String content, ExecutionMode mode) {
        this.terminal.writer().println(CliStrings.messageInfo("Executing SQL from file.").toAnsi());
        for (String statement : CliStatementSplitter.splitContent(content)) {
            this.terminal.writer().println(new AttributedString(String.format("%s%s", this.prompt, statement)).toString());
            this.terminal.flush();
            if (this.executeStatement(statement, mode)) continue;
            return false;
        }
        return true;
    }

    private boolean executeStatement(String statement, ExecutionMode executionMode) {
        try {
            Optional<Operation> operation = this.parseCommand(statement);
            operation.ifPresent(op -> this.callOperation((Operation)op, executionMode));
        }
        catch (SqlExecutionException e) {
            this.printExecutionException(e);
            return false;
        }
        return true;
    }

    private void validate(Operation operation, ExecutionMode executionMode) {
        if (executionMode.equals((Object)ExecutionMode.INITIALIZATION)) {
            if (!(operation instanceof SetOperation || operation instanceof ResetOperation || operation instanceof CreateOperation || operation instanceof DropOperation || operation instanceof UseOperation || operation instanceof AlterOperation || operation instanceof LoadModuleOperation || operation instanceof UnloadModuleOperation)) {
                throw new SqlExecutionException("Unsupported operation in sql init file: " + operation.asSummaryString());
            }
        } else if (executionMode.equals((Object)ExecutionMode.NON_INTERACTIVE_EXECUTION)) {
            ResultMode mode = (ResultMode)((Object)this.executor.getSessionConfig(this.sessionId).get(SqlClientOptions.EXECUTION_RESULT_MODE));
            if (operation instanceof QueryOperation && !mode.equals((Object)ResultMode.TABLEAU)) {
                throw new SqlExecutionException(String.format("In non-interactive mode, it only supports to use %s as value of %s when execute query. Please add 'SET %s=%s;' in the sql file.", new Object[]{ResultMode.TABLEAU, SqlClientOptions.EXECUTION_RESULT_MODE.key(), SqlClientOptions.EXECUTION_RESULT_MODE.key(), ResultMode.TABLEAU}));
            }
        }
        if (this.isStatementSetMode && !(operation instanceof CatalogSinkModifyOperation) && !(operation instanceof EndStatementSetOperation)) {
            throw new SqlExecutionException("Only INSERT statement is allowed in Statement Set.");
        }
    }

    private Optional<Operation> parseCommand(String stmt) {
        if ((stmt = stmt.trim()).endsWith(";")) {
            stmt = stmt.substring(0, stmt.length() - 1).trim();
        }
        if (stmt.trim().isEmpty()) {
            return Optional.empty();
        }
        Operation operation = this.executor.parseStatement(this.sessionId, stmt);
        return Optional.of(operation);
    }

    private void callOperation(Operation operation, ExecutionMode mode) {
        this.validate(operation, mode);
        if (operation instanceof QuitOperation) {
            this.callQuit();
        } else if (operation instanceof ClearOperation) {
            this.callClear();
        } else if (operation instanceof HelpOperation) {
            this.callHelp();
        } else if (operation instanceof SetOperation) {
            this.callSet((SetOperation)operation);
        } else if (operation instanceof ResetOperation) {
            this.callReset((ResetOperation)operation);
        } else if (operation instanceof CatalogSinkModifyOperation) {
            this.callInsert((CatalogSinkModifyOperation)operation);
        } else if (operation instanceof QueryOperation) {
            this.callSelect((QueryOperation)operation);
        } else if (operation instanceof ExplainOperation) {
            this.callExplain((ExplainOperation)operation);
        } else if (operation instanceof BeginStatementSetOperation) {
            this.callBeginStatementSet();
        } else if (operation instanceof EndStatementSetOperation) {
            this.callEndStatementSet();
        } else {
            this.executeOperation(operation);
        }
    }

    private void callQuit() {
        this.printInfo("Exiting Flink SQL CLI Client...");
        this.isRunning = false;
    }

    private void callClear() {
        this.clearTerminal();
    }

    private void callReset(ResetOperation resetOperation) {
        if (!resetOperation.getKey().isPresent()) {
            this.executor.resetSessionProperties(this.sessionId);
            this.printInfo("All session properties have been set to their default values.");
        } else {
            String key = (String)resetOperation.getKey().get();
            this.executor.resetSessionProperty(this.sessionId, key);
            this.printSetResetConfigKeyMessage(key, "Session property has been reset.");
        }
    }

    private void callSet(SetOperation setOperation) {
        if (setOperation.getKey().isPresent() && setOperation.getValue().isPresent()) {
            String key = ((String)setOperation.getKey().get()).trim();
            String value = ((String)setOperation.getValue().get()).trim();
            this.executor.setSessionProperty(this.sessionId, key, value);
            this.printSetResetConfigKeyMessage(key, "Session property has been set.");
        } else {
            Map<String, String> properties = this.executor.getSessionConfigMap(this.sessionId);
            if (properties.isEmpty()) {
                this.terminal.writer().println(CliStrings.messageInfo("Result was empty.").toAnsi());
            } else {
                List<String> prettyEntries = YamlConfigUtils.getPropertiesInPretty(properties);
                prettyEntries.forEach(entry -> this.terminal.writer().println((String)entry));
            }
            this.terminal.flush();
        }
    }

    private void callHelp() {
        this.terminal.writer().println(CliStrings.MESSAGE_HELP);
        this.terminal.flush();
    }

    private void callSelect(QueryOperation operation) {
        ResultDescriptor resultDesc = this.executor.executeQuery(this.sessionId, operation);
        if (resultDesc.isTableauMode()) {
            try (CliTableauResultView tableauResultView = new CliTableauResultView(this.terminal, this.executor, this.sessionId, resultDesc);){
                tableauResultView.displayResults();
            }
        } else {
            CliResultView view = resultDesc.isMaterialized() ? new CliTableResultView(this, resultDesc) : new CliChangelogResultView(this, resultDesc);
            view.open();
            this.printInfo("Result retrieval cancelled.");
        }
    }

    private void callInsert(CatalogSinkModifyOperation operation) {
        if (this.isStatementSetMode) {
            this.statementSetOperations.add((ModifyOperation)operation);
            this.printInfo("Add SQL update statement to the statement set.");
        } else {
            this.callInserts(Collections.singletonList(operation));
        }
    }

    private void callInserts(List<ModifyOperation> operations) {
        this.printInfo("Submitting SQL update statement to the cluster...");
        boolean sync = (Boolean)this.executor.getSessionConfig(this.sessionId).get(TableConfigOptions.TABLE_DML_SYNC);
        if (sync) {
            this.printInfo("Execute statement in sync mode. Please wait for the execution finish...");
        }
        TableResult tableResult = this.executor.executeModifyOperations(this.sessionId, operations);
        Preconditions.checkState((boolean)tableResult.getJobClient().isPresent());
        if (sync) {
            this.terminal.writer().println(CliStrings.messageInfo("Complete execution of the SQL update statement.").toAnsi());
        } else {
            this.terminal.writer().println(CliStrings.messageInfo("SQL update statement has been successfully submitted to the cluster:").toAnsi());
            this.terminal.writer().println(String.format("Job ID: %s\n", ((JobClient)tableResult.getJobClient().get()).getJobID().toString()));
        }
        this.terminal.flush();
    }

    public void callExplain(ExplainOperation operation) {
        TableResult tableResult = this.executor.executeOperation(this.sessionId, (Operation)operation);
        String explanation = Objects.requireNonNull(((Row)tableResult.collect().next()).getField(0)).toString();
        this.terminal.writer().println(explanation);
        this.terminal.flush();
    }

    private void callBeginStatementSet() {
        this.isStatementSetMode = true;
        this.statementSetOperations = new ArrayList<ModifyOperation>();
        this.printInfo("Begin a statement set.");
    }

    private void callEndStatementSet() {
        if (this.isStatementSetMode) {
            this.isStatementSetMode = false;
            if (!this.statementSetOperations.isEmpty()) {
                this.callInserts(this.statementSetOperations);
            } else {
                this.printInfo("No statement in the statement set, skip submit.");
            }
        } else {
            throw new SqlExecutionException("No Statement Set to submit, \"END;\" command should be used after \"BEGIN STATEMENT SET;\".");
        }
        this.statementSetOperations = null;
    }

    private void executeOperation(Operation operation) {
        TableResult result = this.executor.executeOperation(this.sessionId, operation);
        if (TableResultImpl.TABLE_RESULT_OK == result) {
            this.printInfo("Execute statement succeed.");
        } else {
            PrintUtils.printAsTableauForm((ResolvedSchema)result.getResolvedSchema(), (Iterator)result.collect(), (PrintWriter)this.terminal.writer(), (int)Integer.MAX_VALUE, (String)"", (boolean)false, (boolean)false, (ZoneId)CliUtils.getSessionTimeZone(this.executor.getSessionConfig(this.sessionId)));
            this.terminal.flush();
        }
    }

    private void printExecutionException(Throwable t) {
        String errorMessage = "Could not execute SQL statement.";
        LOG.warn("Could not execute SQL statement.", t);
        boolean isVerbose = (Boolean)this.executor.getSessionConfig(this.sessionId).get(SqlClientOptions.VERBOSE);
        this.terminal.writer().println(CliStrings.messageError("Could not execute SQL statement.", t, isVerbose).toAnsi());
        this.terminal.flush();
    }

    private void printInfo(String message) {
        this.terminal.writer().println(CliStrings.messageInfo(message).toAnsi());
        this.terminal.flush();
    }

    private void printWarning(String message) {
        this.terminal.writer().println(CliStrings.messageWarning(message).toAnsi());
        this.terminal.flush();
    }

    private void printSetResetConfigKeyMessage(String key, String message) {
        boolean isRemovedKey = YamlConfigUtils.isRemovedKey(key);
        boolean isDeprecatedKey = YamlConfigUtils.isDeprecatedKey(key);
        if (isRemovedKey || isDeprecatedKey) {
            String warningMsg = isRemovedKey ? "The specified key is not supported anymore." : String.format("The specified key '%s' is deprecated. Please use '%s' instead.", key, YamlConfigUtils.getOptionNameWithDeprecatedKey(key));
            this.printWarning(warningMsg);
        }
        if (!isRemovedKey) {
            this.terminal.writer().println(CliStrings.messageInfo(message).toAnsi());
            this.terminal.flush();
        }
    }

    private void closeTerminal() {
        try {
            this.terminal.close();
            this.terminal = null;
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private LineReader createLineReader(Terminal terminal) {
        LineReader lineReader = LineReaderBuilder.builder().terminal(terminal).appName("Flink SQL CLI Client").parser(new SqlMultiLineParser()).completer(new SqlCompleter(this.sessionId, this.executor)).build();
        lineReader.option(LineReader.Option.DISABLE_EVENT_EXPANSION, true);
        lineReader.setVariable("errors", 1);
        lineReader.option(LineReader.Option.CASE_INSENSITIVE, true);
        if (Files.exists(this.historyFilePath, new LinkOption[0]) || CliUtils.createFile(this.historyFilePath)) {
            String msg = "Command history file path: " + this.historyFilePath;
            terminal.writer().println(msg);
            LOG.info(msg);
            lineReader.setVariable("history-file", this.historyFilePath);
        } else {
            String msg = "Unable to create history file: " + this.historyFilePath;
            terminal.writer().println(msg);
            LOG.warn(msg);
        }
        return lineReader;
    }

    static enum ExecutionMode {
        INTERACTIVE_EXECUTION,
        NON_INTERACTIVE_EXECUTION,
        INITIALIZATION;

    }
}

