/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.security;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.security.auth.login.AppConfigurationEntry;
import javax.security.auth.login.Configuration;
import org.apache.kafka.common.config.types.Password;
import org.apache.kafka.common.network.ListenerName;
import org.apache.kafka.common.security.JaasConfig;
import org.apache.kafka.common.security.JaasContext;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class JaasContextTest {
    private File jaasConfigFile;

    @Before
    public void setUp() throws IOException {
        this.jaasConfigFile = File.createTempFile("jaas", ".conf");
        this.jaasConfigFile.deleteOnExit();
        System.setProperty("java.security.auth.login.config", this.jaasConfigFile.toString());
        Configuration.setConfiguration(null);
    }

    @After
    public void tearDown() throws Exception {
        Files.delete(this.jaasConfigFile.toPath());
    }

    @Test
    public void testConfigNoOptions() throws Exception {
        this.checkConfiguration("test.testConfigNoOptions", AppConfigurationEntry.LoginModuleControlFlag.REQUIRED, new HashMap<String, Object>());
    }

    @Test
    public void testControlFlag() throws Exception {
        AppConfigurationEntry.LoginModuleControlFlag[] controlFlags = new AppConfigurationEntry.LoginModuleControlFlag[]{AppConfigurationEntry.LoginModuleControlFlag.REQUIRED, AppConfigurationEntry.LoginModuleControlFlag.REQUISITE, AppConfigurationEntry.LoginModuleControlFlag.SUFFICIENT, AppConfigurationEntry.LoginModuleControlFlag.OPTIONAL};
        HashMap<String, Object> options = new HashMap<String, Object>();
        options.put("propName", "propValue");
        for (AppConfigurationEntry.LoginModuleControlFlag controlFlag : controlFlags) {
            this.checkConfiguration("test.testControlFlag", controlFlag, options);
        }
    }

    @Test
    public void testSingleOption() throws Exception {
        HashMap<String, Object> options = new HashMap<String, Object>();
        options.put("propName", "propValue");
        this.checkConfiguration("test.testSingleOption", AppConfigurationEntry.LoginModuleControlFlag.REQUISITE, options);
    }

    @Test
    public void testMultipleOptions() throws Exception {
        HashMap<String, Object> options = new HashMap<String, Object>();
        for (int i = 0; i < 10; ++i) {
            options.put("propName" + i, "propValue" + i);
        }
        this.checkConfiguration("test.testMultipleOptions", AppConfigurationEntry.LoginModuleControlFlag.SUFFICIENT, options);
    }

    @Test
    public void testQuotedOptionValue() throws Exception {
        HashMap<String, Object> options = new HashMap<String, Object>();
        options.put("propName", "prop value");
        options.put("propName2", "value1 = 1, value2 = 2");
        String config = String.format("test.testQuotedOptionValue required propName=\"%s\" propName2=\"%s\";", options.get("propName"), options.get("propName2"));
        this.checkConfiguration(config, "test.testQuotedOptionValue", AppConfigurationEntry.LoginModuleControlFlag.REQUIRED, options);
    }

    @Test
    public void testQuotedOptionName() throws Exception {
        HashMap<String, Object> options = new HashMap<String, Object>();
        options.put("prop name", "propValue");
        String config = "test.testQuotedOptionName required \"prop name\"=propValue;";
        this.checkConfiguration(config, "test.testQuotedOptionName", AppConfigurationEntry.LoginModuleControlFlag.REQUIRED, options);
    }

    @Test
    public void testMultipleLoginModules() throws Exception {
        StringBuilder builder = new StringBuilder();
        int moduleCount = 3;
        HashMap<Integer, HashMap<String, Object>> moduleOptions = new HashMap<Integer, HashMap<String, Object>>();
        for (int i = 0; i < moduleCount; ++i) {
            HashMap<String, Object> options = new HashMap<String, Object>();
            options.put("index", "Index" + i);
            options.put("module", "Module" + i);
            moduleOptions.put(i, options);
            String module = this.jaasConfigProp("test.Module" + i, AppConfigurationEntry.LoginModuleControlFlag.REQUIRED, options);
            builder.append(' ');
            builder.append(module);
        }
        String jaasConfigProp = builder.toString();
        String clientContextName = "CLIENT";
        JaasConfig configuration = new JaasConfig(clientContextName, jaasConfigProp);
        AppConfigurationEntry[] dynamicEntries = configuration.getAppConfigurationEntry(clientContextName);
        Assert.assertEquals((long)moduleCount, (long)dynamicEntries.length);
        for (int i = 0; i < moduleCount; ++i) {
            AppConfigurationEntry entry = dynamicEntries[i];
            this.checkEntry(entry, "test.Module" + i, AppConfigurationEntry.LoginModuleControlFlag.REQUIRED, (Map)moduleOptions.get(i));
        }
        String serverContextName = "SERVER";
        this.writeConfiguration(serverContextName, jaasConfigProp);
        AppConfigurationEntry[] staticEntries = Configuration.getConfiguration().getAppConfigurationEntry(serverContextName);
        for (int i = 0; i < moduleCount; ++i) {
            AppConfigurationEntry staticEntry = staticEntries[i];
            this.checkEntry(staticEntry, dynamicEntries[i].getLoginModuleName(), AppConfigurationEntry.LoginModuleControlFlag.REQUIRED, dynamicEntries[i].getOptions());
        }
    }

    @Test
    public void testMissingLoginModule() throws Exception {
        this.checkInvalidConfiguration("  required option1=value1;");
    }

    @Test
    public void testMissingControlFlag() throws Exception {
        this.checkInvalidConfiguration("test.loginModule option1=value1;");
    }

    @Test
    public void testMissingOptionValue() throws Exception {
        this.checkInvalidConfiguration("loginModule required option1;");
    }

    @Test
    public void testMissingSemicolon() throws Exception {
        this.checkInvalidConfiguration("test.testMissingSemicolon required option1=value1");
    }

    @Test
    public void testNumericOptionWithoutQuotes() throws Exception {
        this.checkInvalidConfiguration("test.testNumericOptionWithoutQuotes required option1=3;");
    }

    @Test
    public void testNumericOptionWithQuotes() throws Exception {
        HashMap<String, Object> options = new HashMap<String, Object>();
        options.put("option1", "3");
        String config = "test.testNumericOptionWithQuotes required option1=\"3\";";
        this.checkConfiguration(config, "test.testNumericOptionWithQuotes", AppConfigurationEntry.LoginModuleControlFlag.REQUIRED, options);
    }

    @Test
    public void testLoadForServerWithListenerNameOverride() throws IOException {
        this.writeConfiguration(Arrays.asList("KafkaServer { test.LoginModuleDefault required; };", "plaintext.KafkaServer { test.LoginModuleOverride requisite; };"));
        JaasContext context = JaasContext.loadServerContext((ListenerName)new ListenerName("plaintext"), (String)"SOME-MECHANISM", Collections.emptyMap());
        Assert.assertEquals((Object)"plaintext.KafkaServer", (Object)context.name());
        Assert.assertEquals((Object)JaasContext.Type.SERVER, (Object)context.type());
        Assert.assertEquals((long)1L, (long)context.configurationEntries().size());
        this.checkEntry((AppConfigurationEntry)context.configurationEntries().get(0), "test.LoginModuleOverride", AppConfigurationEntry.LoginModuleControlFlag.REQUISITE, Collections.emptyMap());
    }

    @Test
    public void testLoadForServerWithListenerNameAndFallback() throws IOException {
        this.writeConfiguration(Arrays.asList("KafkaServer { test.LoginModule required; };", "other.KafkaServer { test.LoginModuleOther requisite; };"));
        JaasContext context = JaasContext.loadServerContext((ListenerName)new ListenerName("plaintext"), (String)"SOME-MECHANISM", Collections.emptyMap());
        Assert.assertEquals((Object)"KafkaServer", (Object)context.name());
        Assert.assertEquals((Object)JaasContext.Type.SERVER, (Object)context.type());
        Assert.assertEquals((long)1L, (long)context.configurationEntries().size());
        this.checkEntry((AppConfigurationEntry)context.configurationEntries().get(0), "test.LoginModule", AppConfigurationEntry.LoginModuleControlFlag.REQUIRED, Collections.emptyMap());
    }

    @Test(expected=IllegalArgumentException.class)
    public void testLoadForServerWithWrongListenerName() throws IOException {
        this.writeConfiguration("Server", "test.LoginModule required;");
        JaasContext.loadServerContext((ListenerName)new ListenerName("plaintext"), (String)"SOME-MECHANISM", Collections.emptyMap());
    }

    private AppConfigurationEntry configurationEntry(JaasContext.Type contextType, String jaasConfigProp) {
        Password saslJaasConfig = jaasConfigProp == null ? null : new Password(jaasConfigProp);
        JaasContext context = JaasContext.load((JaasContext.Type)contextType, null, (String)contextType.name(), (Password)saslJaasConfig);
        List entries = context.configurationEntries();
        Assert.assertEquals((long)1L, (long)entries.size());
        return (AppConfigurationEntry)entries.get(0);
    }

    private String controlFlag(AppConfigurationEntry.LoginModuleControlFlag loginModuleControlFlag) {
        String[] tokens = loginModuleControlFlag.toString().split(" ");
        return tokens[tokens.length - 1];
    }

    private String jaasConfigProp(String loginModule, AppConfigurationEntry.LoginModuleControlFlag controlFlag, Map<String, Object> options) {
        StringBuilder builder = new StringBuilder();
        builder.append(loginModule);
        builder.append(' ');
        builder.append(this.controlFlag(controlFlag));
        for (Map.Entry<String, Object> entry : options.entrySet()) {
            builder.append(' ');
            builder.append(entry.getKey());
            builder.append('=');
            builder.append(entry.getValue());
        }
        builder.append(';');
        return builder.toString();
    }

    private void writeConfiguration(String contextName, String jaasConfigProp) throws IOException {
        List<String> lines = Arrays.asList(contextName + " { ", jaasConfigProp, "};");
        this.writeConfiguration(lines);
    }

    private void writeConfiguration(List<String> lines) throws IOException {
        Files.write(this.jaasConfigFile.toPath(), lines, StandardCharsets.UTF_8, new OpenOption[0]);
        Configuration.setConfiguration(null);
    }

    private void checkConfiguration(String loginModule, AppConfigurationEntry.LoginModuleControlFlag controlFlag, Map<String, Object> options) throws Exception {
        String jaasConfigProp = this.jaasConfigProp(loginModule, controlFlag, options);
        this.checkConfiguration(jaasConfigProp, loginModule, controlFlag, options);
    }

    private void checkEntry(AppConfigurationEntry entry, String loginModule, AppConfigurationEntry.LoginModuleControlFlag controlFlag, Map<String, ?> options) {
        Assert.assertEquals((Object)loginModule, (Object)entry.getLoginModuleName());
        Assert.assertEquals((Object)controlFlag, (Object)entry.getControlFlag());
        Assert.assertEquals(options, entry.getOptions());
    }

    private void checkConfiguration(String jaasConfigProp, String loginModule, AppConfigurationEntry.LoginModuleControlFlag controlFlag, Map<String, Object> options) throws Exception {
        AppConfigurationEntry dynamicEntry = this.configurationEntry(JaasContext.Type.CLIENT, jaasConfigProp);
        this.checkEntry(dynamicEntry, loginModule, controlFlag, options);
        Assert.assertNull((String)"Static configuration updated", (Object)Configuration.getConfiguration().getAppConfigurationEntry(JaasContext.Type.CLIENT.name()));
        this.writeConfiguration(JaasContext.Type.SERVER.name(), jaasConfigProp);
        AppConfigurationEntry staticEntry = this.configurationEntry(JaasContext.Type.SERVER, null);
        this.checkEntry(staticEntry, loginModule, controlFlag, options);
    }

    private void checkInvalidConfiguration(String jaasConfigProp) throws IOException {
        AppConfigurationEntry entry;
        try {
            this.writeConfiguration(JaasContext.Type.SERVER.name(), jaasConfigProp);
            entry = this.configurationEntry(JaasContext.Type.SERVER, null);
            Assert.fail((String)("Invalid JAAS configuration file didn't throw exception, entry=" + entry));
        }
        catch (SecurityException entry2) {
            // empty catch block
        }
        try {
            entry = this.configurationEntry(JaasContext.Type.CLIENT, jaasConfigProp);
            Assert.fail((String)("Invalid JAAS configuration property didn't throw exception, entry=" + entry));
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }
}

