/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.cloud.context.cs;

import com.alibaba.cloud.context.AliCloudConfiguration;
import com.alibaba.cloud.context.AliCloudServerMode;
import com.alibaba.cloud.context.edas.*;
import com.alibaba.cloud.context.utils.StringUtils;
import com.aliyuncs.edas.model.v20170801.GetSecureTokenResponse;
import com.aliyuncs.edas.model.v20170801.InsertApplicationResponse;
import com.taobao.config.client.LocalAttribute;
import com.taobao.config.client.PublisherRegistration;
import com.taobao.config.client.SubscriberRegistration;
import com.taobao.config.client.bean.InstanceMetaData;

/**
 * configserver初始化工具类
 *
 * @author xiaolongzuo
 */
public class AliCloudCsInitializer {

    private AliCloudCsInitializer() {
    }

    /**
     * 初始化发布者的权限
     *
     * @param csHandledConfiguration configserver已处理的配置
     * @param registration           发布者
     */
    public static void initialize(CsHandledConfiguration csHandledConfiguration,
                                  PublisherRegistration<String> registration) {
        initialize(csHandledConfiguration, registration, null);
    }

    /**
     * 初始化订阅者权限
     *
     * @param csHandledConfiguration configserver已处理的配置
     * @param registration           订阅者
     */
    public static void initialize(CsHandledConfiguration csHandledConfiguration,
                                  SubscriberRegistration registration) {
        initialize(csHandledConfiguration, null, registration);
    }

    private static void initialize(CsHandledConfiguration csHandledConfiguration,
                                   PublisherRegistration<String> publisherRegistration,
                                   SubscriberRegistration subscriberRegistration) {
        if (csHandledConfiguration.getServerMode() == AliCloudServerMode.EDAS) {
            InstanceMetaData instanceMetaData = new InstanceMetaData(
                csHandledConfiguration.getProjectName(),
                csHandledConfiguration.getDauthAccessKey(),
                csHandledConfiguration.getDauthSecretKey());
            if (publisherRegistration != null) {
                publisherRegistration.setTenant(csHandledConfiguration.getTenantId());
                publisherRegistration.setInstanceMetaData(instanceMetaData);
            }
            if (subscriberRegistration != null) {
                subscriberRegistration.setTenant(csHandledConfiguration.getTenantId());
                subscriberRegistration.setInstanceMetaData(instanceMetaData);
            }
        } else {

            InstanceMetaData instanceMetaData = new InstanceMetaData(
                csHandledConfiguration.getProjectName(),
                csHandledConfiguration.getDauthAccessKey(),
                csHandledConfiguration.getDauthSecretKey());

            if (publisherRegistration != null) {
                publisherRegistration.setInstanceMetaData(instanceMetaData);
                if (!StringUtils.isEmpty(csHandledConfiguration.getServerList())) {
                    publisherRegistration.setLocalAttribute(LocalAttribute.ATTRIBUTE_SERVER,
                        csHandledConfiguration.getServerList() + "?key=default");
                }
            }

            if (subscriberRegistration != null) {
                subscriberRegistration.setInstanceMetaData(instanceMetaData);
                if (!StringUtils.isEmpty(csHandledConfiguration.getServerList())) {
                    subscriberRegistration.setLocalAttribute(LocalAttribute.ATTRIBUTE_SERVER,
                        csHandledConfiguration.getServerList() + "?key=default");
                }
            }
        }
    }

    /**
     * 处理cs配置
     *
     * @param aliCloudConfiguration 阿里云通用配置
     * @param edasConfiguration     EDAS通用配置
     * @param csConfiguration       configserver配置
     * @return configserver已处理的配置
     */
    public static CsHandledConfiguration handleCsConfiguration(
        AliCloudConfiguration aliCloudConfiguration,
        EdasConfiguration edasConfiguration, CsConfiguration csConfiguration) {
        return handleCsConfiguration(aliCloudConfiguration, edasConfiguration,
            csConfiguration, null);
    }

    /**
     * 处理cs配置
     *
     * @param aliCloudConfiguration 阿里云通用配置
     * @param edasConfiguration     EDAS通用配置
     * @param csConfiguration       configserver配置
     * @param aliCloudEdasSdk       EDAS的POP-API-SDK
     * @return configserver已处理的配置
     */
    public static CsHandledConfiguration handleCsConfiguration(
        AliCloudConfiguration aliCloudConfiguration,
        EdasConfiguration edasConfiguration, CsConfiguration csConfiguration,
        AliCloudEdasSdk aliCloudEdasSdk) {
        SimpleCsHandledConfiguration csHandledConfiguration = new SimpleCsHandledConfiguration();
        EdasChangeOrderConfiguration edasChangeOrderConfiguration = EdasChangeOrderConfigurationFactory
            .getEdasChangeOrderConfiguration();
        if (edasChangeOrderConfiguration.isEdasManaged()) {
            csHandledConfiguration.setDauthAccessKey(edasChangeOrderConfiguration.getDauthAccessKey());
            csHandledConfiguration.setDauthSecretKey(edasChangeOrderConfiguration.getDauthSecretKey());
            csHandledConfiguration.setTenantId(edasChangeOrderConfiguration.getTenantId());
            csHandledConfiguration.setProjectName(edasChangeOrderConfiguration.getProjectName());
            csHandledConfiguration.setServerMode(AliCloudServerMode.EDAS);
        } else if (edasConfiguration.isEnabled() || csConfiguration.getServerMode() == AliCloudServerMode.EDAS) {
            if (aliCloudEdasSdk == null) {
                aliCloudEdasSdk = AliCloudEdasSdkFactory.getDefaultAliCloudEdasSdk(
                    aliCloudConfiguration, edasConfiguration.getRegionId());
            }
            GetSecureTokenResponse.SecureToken secureToken = aliCloudEdasSdk.getSecureToken(
                edasConfiguration.getNamespace());
            InsertApplicationResponse.ApplicationInfo applicationInfo = aliCloudEdasSdk.getApplicationInfo(
                edasConfiguration.getApplicationName(), edasConfiguration.getNamespace());
            csHandledConfiguration.setDauthAccessKey(secureToken.getAccessKey());
            csHandledConfiguration.setDauthSecretKey(secureToken.getSecretKey());
            csHandledConfiguration.setTenantId(secureToken.getTenantId());
            csHandledConfiguration.setProjectName(applicationInfo.getAppId());
            csHandledConfiguration.setServerMode(AliCloudServerMode.EDAS);
            System.setProperty("configserver.client.port", "8000");
            System.setProperty("address.server.domain", secureToken.getAddressServerHost());
        } else {
            csHandledConfiguration.setServerList(csConfiguration.getServerList());
            csHandledConfiguration.setServerPort(csConfiguration.getServerPort());
            csHandledConfiguration.setServerMode(AliCloudServerMode.LOCAL);
        }
        if (null != csHandledConfiguration.getServerPort()) {
            System.setProperty("configserver.client.port", csHandledConfiguration.getServerPort());
        }
        return csHandledConfiguration;
    }

}
