package com.taobao.hsf.invocation.filter;

import com.taobao.hsf.annotation.Scope;
import com.taobao.hsf.invocation.Invocation;
import com.taobao.hsf.invocation.InvocationHandler;
import com.taobao.hsf.invocation.RPCResult;
import com.taobao.hsf.util.concurrent.ListenableFuture;

/**
 * <pre>
 * RPC调用的过滤器，客户端可以通过实现{@link RPCFilter}来将自己的逻辑植入到HSF的调用流程中
 *
 *  ┌--------------------------------------------------------------------------------┐
 *  |                com.taobao.hsf.invocation.filter.RPCFilterBuilder$HeadNode      |
 *  └--------------------------------------------------------------------------------┘
 *            | nextHandler.invoke(invocation);                            ^
 *            v                                                            | onResponse(invocation, rpcResult)
 *  ┌--------------------------------------------------------------------------------┐
 *  |                       com.taobao.hsf.invocation.filter.TestFilter1             |
 *  └--------------------------------------------------------------------------------┘
 *            | nextHandler.invoke(invocation);                            ^
 *            v                                                            | onResponse(invocation, rpcResult)
 *  ┌--------------------------------------------------------------------------------┐
 *  |                    com.taobao.hsf.invocation.filter.TestFilter2222222          |
 *  └--------------------------------------------------------------------------------┘
 *            | nextHandler.invoke(invocation);                            ^
 *            v                                                            | onResponse(invocation, rpcResult)
 *  ┌--------------------------------------------------------------------------------┐
 *  |                com.taobao.hsf.invocation.filter.RPCFilterBuilder$TailNode      |
 *  └--------------------------------------------------------------------------------┘
 *            |                                                            ^
 *            v                                                            |
 *     nextHandler.invoke(invocation); ------同步或者异步-------> future.set(rpcResult);
 *
 * 用户可以如下的方式扩展：
 *
 * <code>@Order(2)</code>
 * <code>public class TestFilter2222222 implements ServerFilter {</code>
 *
 *   <code>@Override</code>
 *   <code>public ListenableFuture<RPCResult> invoke(InvocationHandler invocationHandler, Invocation invocation)</code>
 *   <code>throws Throwable {</code>
 *   <code>   System.out.println("Before TestFilter2 invoke");</code>
 *   <code>   System.out.println("TestFilter2 invoke...");</code>
 *   <code>   invocation.put("B", 2);</code>
 *   <code>   ListenableFuture<RPCResult> result = invocationHandler.invoke(invocation);</code>
 *   <code>   System.out.println("After TestFilter2 invoke");</code>
 *   <code>   return result;</code>
 *   <code>}</code>
 *
 *   <code>@Override</code>
 *   <code>public void onResponse(Invocation invocation, RPCResult rpcResult) {</code>
 *   <code>   System.out.println("TestFilter2 on " + Thread.currentThread());</code>
 *   <code>   System.out.println("TestFilter2 onResponse. got " + rpcResult.getAppResponse());</code>
 *   <code>}</code>
 *
 * <code>}</code>
 *
 * 如果需要在{@link RPCFilter#invoke(InvocationHandler, Invocation)}过程中提前返回（中断当前的调用链），通过使用
 * { com.taobao.hsf.util.concurrent.Futures#createSettableFuture()}来创建一个{ SettableFuture}并完成设置，同时返回
 *
 * 如果扩展点在后端，而前端提前结束，需要知道结束的情况，可以实现{ BlockAwareFilter}
 *
 *
 * </pre>
 *
 *  ClientFilter 消费端扩展
 *  ServerFilter 服务端扩展
 *  BlockAwareFilter 知晓阻塞的扩展
 *  Order 调用排序
 * <p>
 * Created by sixie.xyn on 2015/11/13.
 */
@Scope(Scope.Option.PROTOTYPE)
public interface RPCFilter {

    /**
     * <pre>
     * 当调用当前的{@link RPCFilter}时，会将下一步执行的{@link InvocationHandler}传递进来，如果不执行
     * <code>invocationHandler.invoke(invocation);</code>
     * 则当前调用链就此终结
     *
//     * 可以通过{ com.taobao.hsf.util.concurrent.Futures#createSettableFuture()}
     * 提供的方法构建{@link ListenableFuture}
     *
     * </pre>
     *
     * @param nextHandler 下一环调用者
     * @param invocation  当前调用
     * @return 调用结果
     * @throws Throwable 调用异常，一般调用中不建议抛出异常
     */
    ListenableFuture<RPCResult> invoke(InvocationHandler nextHandler, Invocation invocation) throws Throwable;

    /**
     * <pre>
     * 响应写回
     * 当数据抵达，结果将会通过该方法写回，回调该接口
     *
     * 一般是结果抵达，调用{ SettableFuture}的set方法，将会触发以往调用链的{@link #onResponse(Invocation, RPCResult)}方法
     * </pre>
     *
     * @param invocation 当前调用
     * @param rpcResult  调用的结果
     */
    void onResponse(Invocation invocation, RPCResult rpcResult);

}
