package com.alibaba.dts.client;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextRefreshedEvent;

import com.alibaba.dts.client.executor.grid.flowcontrol.FlowControlStrategy;
import com.alibaba.dts.client.executor.job.context.ClientContext;
import com.alibaba.dts.client.executor.job.context.ClientContextImpl;
import com.alibaba.dts.client.executor.job.processor.FailureJobProcessor;
import com.alibaba.dts.client.executor.job.processor.StopJobProcessor;
import com.alibaba.dts.client.route.RouteRule;
import com.alibaba.dts.client.store.StoreStrategy;
import com.alibaba.dts.common.constants.Constants;
import com.alibaba.dts.common.domain.result.Result;
import com.alibaba.dts.common.domain.result.ResultCode;
import com.alibaba.dts.common.exception.InitException;
import com.alibaba.dts.common.logger.LoggerCleaner;
import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;
import com.alibaba.dts.common.util.GroupIdUtil;
import com.alibaba.dts.common.util.PathUtil;
import com.alibaba.dts.common.util.StringUtil;
import com.alibaba.edas.schedulerx.SchedulerXLogLevel;
import com.taobao.common.fulllinkstresstesting.SplitEnvUtil;

/**
 * SchedulerX client, entrance of the client
 * <p>
 * Created by yif on 16/8/30.
 *
 * @author <a href="mailto:zhihong.zyf@alibaba-inc.com">zhihong</a>
 */
public class SchedulerxClient implements ApplicationContextAware, ApplicationListener, Constants, ClientService {

    private static final Logger logger = SchedulerXLoggerFactory.getLogger(SchedulerxClient.class);

    private final ClientContextImpl clientContext = new ClientContextImpl();

    private static final ConcurrentHashMap<String, Boolean> clientTable = new ConcurrentHashMap<String, Boolean>();

    private static Object lock = new Object();

    private SchedulerXLogLevel defaultLogLevel = SchedulerXLogLevel.WARN;

    private SchedulerXLogLevel executeLogLevel = SchedulerXLogLevel.WARN;


    //日志清理
    private LoggerCleaner loggerCleaner = new LoggerCleaner(PathUtil.getLoggerPath());


    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        //设置Spring上下文
        clientContext.getJobProcessorFactory().setApplicationContext(applicationContext);

        //设置Spring
        clientContext.getClientConfig().setSpring(true);
        logger.warn("[SchedulerxClient]: setApplicationContext over, applicationContext:" + applicationContext);
    }

    @Override
    public void onApplicationEvent(ApplicationEvent applicationEvent) {
        if (!(applicationEvent instanceof ContextRefreshedEvent)) {
            return;
        }

        if (!clientContext.getClientConfig().isAutoInit()) {
            logger.info("[SchedulerxClient]: do not AutoInit, clientConfig:" + clientContext.getClientConfig());
            return;
        }

        try {
            init();
        } catch (Throwable e) {
            String info = "[SchedulerxClient]: init error, clientConfig:" + clientContext.getClientConfig();
            logger.error(info, e);
            throw new RuntimeException(info, e);
        }
    }


    public void init() throws InitException {
        SchedulerXLoggerFactory.initLogByLevel(defaultLogLevel.getInformation(), executeLogLevel.getInformation());

        if (!Constants.ENVIRONMENT_SCX.equals(clientContext.getClientConfig().getEnvironment())) {
            if (!Constants.ENVIRONMENT_PRIVATE_CLOUD.equals(clientContext.getClientConfig().getEnvironment())) {
                if (!SplitEnvUtil.needStartDtsOnlyByEnv()) {
//                    logger.error("[SchedulerxClient]: init error, because of isolation environment");
//                    return;
                    logger.info("SchedulerxClient, yace env!");
                }
            }
        }
        /** 客户端各项参数配置初始化 */
        clientContext.getClientConfig().init();

        clientContext.getNodeServerServiceLocal().init();

        if (Constants.ENVIRONMENT_SCX.equals(clientContext.getClientConfig().getEnvironment())) {

            //初使化httpHead
            if (clientContext.getClientConfig().getInnerDebug() == 1) {
                clientContext.getHttpService().setUseHttps(false);
            } else {
                clientContext.getHttpService().setUseHttps(true);
            }

            //初使化groupID
            if (StringUtil.isEmpty(clientContext.getClientConfig().getGroupId())) {
                initGroupId(clientContext.getClientConfig().getServiceGroup(), clientContext.getClientConfig().getRegionName(), clientContext.getClientConfig().getServiceGroupId());

            }
        }

        /** 检查分组ID */
        GroupIdUtil.checkGroupId(clientContext.getClientConfig().getGroupId());

        synchronized (lock) {

            logger.warn("[SchedulerxClient]: tell initialized"
                    + ", thread:" + Thread.currentThread().getName()
                    + ", groupId:" + clientContext.getClientConfig().getGroupId());

            Boolean init = clientTable.get(clientContext.getClientConfig().getGroupId());

            if (null == init) {
                clientTable.put(clientContext.getClientConfig().getGroupId(), new Boolean(true));
            } else {
                logger.warn("SchedulerxClient has already initialized"
                        + ", thread:" + Thread.currentThread().getName()
                        + ", groupId:" + clientContext.getClientConfig().getGroupId());
                return;
            }

        }

        /**
         * 初始化数据库相关资源
         */
        clientContext.getStore().init();

        /**
         * 初始化执行计数器
         */
//        clientContext.getExecutionCounterUpdateTimer().init();
//        clientContext.getTaskDeleteTimer().init();
//        clientContext.getExecutionCounterDeleteTimer().init();
        clientContext.getCompensationTimer().init();

        //初始化日志清理定时器
        clientContext.getLogCleaner().init();

        /** Zookeeper初始化 */
        clientContext.getZookeeper().init();

        if (Constants.ENVIRONMENT_SCX.equals(clientContext.getClientConfig().getEnvironment())) {
            clientContext.getSecurityCheck().init();
//			clientContext.getSecurityCheck().doSecurityCheck();
        }

        clientContext.getFlowControlChain().init();
        clientContext.getHealthChecker().init();


        /** 客户端远程通信初始化 */
        clientContext.getClientRemoting().init();

        clientContext.getNodeRemoting().init();

//        initDataSource();

        /** 初始化Job处理器工厂 */
        if (!clientContext.getClientConfig().isAgent()) {
            clientContext.getJobProcessorFactory().init();
        }

        if (clientContext.getClientConfig().isAgent()) {
            //初始化日志清理
            loggerCleaner.init(); //是SchedulerX-Agent就启动日志清理
        }


        clientContext.getGridTaskSender().init();


        logger.warn("[SchedulerxClient]: init over");
        logger.warn("[SchedulerxClient]: ClientConfig:" + clientContext.getClientConfig().toString());
        logger.warn("[SchedulerxClient]: NodeConfig:" + clientContext.getNodeConfig());
    }


    public void initGroupId(String serviceGroup, String regionName, String serviceGroupId) throws InitException {

        if (StringUtil.isBlank(serviceGroupId)) {
            throw new InitException("[InitGroupId]: serviceGroupId  is null!");
        }

        if (StringUtil.isBlank(serviceGroup) || StringUtil.isBlank(regionName)) {
            throw new InitException("[InitGroupId]: serviceGroup or regionName is null!");
        }

        if (StringUtil.isBlank(serviceGroup) || StringUtil.isBlank(regionName)) {
            throw new InitException("[InitGroupId]: serviceGroup or regionName is null!");
        }
        Result<String> result = clientContext.getHttpService().acquireGroupID(this.clientContext.getClientConfig().getDomainName(), serviceGroupId, regionName);
        if (result.getResultCode() == null) {
            throw new InitException("[InitGroupId]: get groupid error, return null!");
        }

        if (result.getResultCode() != ResultCode.SUCCESS) {
            throw new InitException("[InitGroupId]: get groupid error," + result.getResultCode().getInformation());
        }
        if (StringUtil.isEmpty(result.getData())) {
            throw new InitException("[InitGroupId]: get groupid error," + result.getResultCode().getInformation());
        } else {
            clientContext.getClientConfig().setGroupId(result.getData());
        }
    }

    public ClientContext getClientContext() {
        return clientContext;
    }

    public void setNewInstance(boolean newInstance) {
        clientContext.getClientConfig().setNewInstance(newInstance);
    }

    public void setSpring(boolean isSpring, ApplicationContext applicationContext) {
        clientContext.getClientConfig().setSpring(isSpring);
        if (isSpring) {
            clientContext.getJobProcessorFactory().setApplicationContext(applicationContext);
        }
    }

    /**
     * 远程通信服务线程数量
     *
     *  remotingThreads
     */
    public void setRemotingThreads(int remotingThreads) {
        clientContext.getClientConfig().setRemotingThreads(remotingThreads);
        clientContext.getNodeConfig().setRemotingThreads(remotingThreads);
    }

    /**
     * 心跳间隔时间
     *
     *  heartBeatIntervalTime
     */
    public void setHeartBeatIntervalTime(long heartBeatIntervalTime) {
        clientContext.getClientConfig().setHeartBeatIntervalTime(heartBeatIntervalTime);
    }

    /**
     * 连接超时时间
     *
     *  connectionTimeout
     */
    public void setConnectionTimeout(long connectionTimeout) {
        clientContext.getClientConfig().setConnectionTimeout(connectionTimeout);
    }

    /**
     * 分组ID
     *
     *  groupId
     */
    public void setGroupId(String groupId) {
        clientContext.getClientConfig().setGroupId(groupId);
        clientContext.getNodeConfig().setGroupId(groupId);
    }

    /**
     * ZK地址列表
     *
     *  zkHosts
     */
    public void setZkHosts(String zkHosts) {
        clientContext.getClientConfig().setZkHosts(zkHosts);
    }

    /**
     * ZK根目录
     *
     *  namespace
     */
    public void setNamespace(String namespace) {
        clientContext.getClientConfig().setNamespace(namespace);
    }

    /**
     * ZK会话超时时间
     *
     *  zkSessionTimeout
     */
    public void setZkSessionTimeout(int zkSessionTimeout) {
        clientContext.getClientConfig().setZkSessionTimeout(zkSessionTimeout);
    }

    /**
     * ZK连接超时时间
     *
     *  zkConnectionTimeout
     */
    public void setZkConnectionTimeout(int zkConnectionTimeout) {
        clientContext.getClientConfig().setZkConnectionTimeout(zkConnectionTimeout);
    }

    /**
     * 队列大小
     *
     *  queueSize
     */
    public void setQueueSize(int queueSize) {
        clientContext.getClientConfig().setQueueSize(queueSize);
    }

    /**
     * 消费线程数量
     *
     *  consumerThreads
     */
    public void setConsumerThreads(int consumerThreads) {
        clientContext.getClientConfig().setConsumerThreads(consumerThreads);
    }

    /**
     * 消费线程数量Map
     *
     *  consumerThreadsMap
     */
    public void setConsumerThreadsMap(Map<String, Integer> consumerThreadsMap) {
        clientContext.getClientConfig().setConsumerThreadsMap(consumerThreadsMap);
    }

    /**
     * 一次从服务端拉取的任务数量
     *
     *  pageSize
     */
    public void setPageSize(int pageSize) {
        if (pageSize <= 0)
            pageSize = 1000;
        clientContext.getClientConfig().setPageSize(pageSize);
        clientContext.getNodeConfig().setTaskInsertBatchSize(pageSize);
    }

    /**
     * 一次从服务端拉取的任务数量Map
     *
     *  pageSizeMap
     */
    public void setPageSizeMap(Map<String, Integer> pageSizeMap) {
        clientContext.getClientConfig().setPageSizeMap(pageSizeMap);
    }

    /**
     * 访问键
     *
     *  accessKey
     */
    public void setAccessKey(String accessKey) {
        clientContext.getClientConfig().setAccessKey(accessKey);
    }

    public void setSecretKey(String secretKey) {
        clientContext.getClientConfig().setSecretKey(secretKey);
    }

    /**
     * 宕机重试
     *
     *  crashRetry
     */
    public void setCrashRetry(boolean crashRetry) {
        clientContext.getClientConfig().setCrashRetry(crashRetry);
    }

    public void setZkHostsAutoChange(boolean zkHostsAutoChange) {
        clientContext.getClientConfig().setZkHostsAutoChange(zkHostsAutoChange);
    }

    /**
     * 空队列暂停拉取间隔时间
     *
     *  pullTaskListOverSleepTime
     */
    public void setPullTaskListOverSleepTime(long pullTaskListOverSleepTime) {
        clientContext.getClientConfig().setPullTaskListOverSleepTime(pullTaskListOverSleepTime);
    }

    public void setFailureJobProcessorMap(Map<String, FailureJobProcessor> failureJobProcessorMap) {
        clientContext.getClientConfig().setFailureJobProcessorMap(failureJobProcessorMap);
    }

    public void setFinishLog(boolean finishLog) {
        clientContext.getClientConfig().setFinishLog(finishLog);
    }

    public void setStopJobProcessor(StopJobProcessor stopJobProcessor) {
        clientContext.getClientConfig().setStopJobProcessor(stopJobProcessor);
    }

    public void setEveryTimeNew(boolean isEveryTimeNew) {
        clientContext.getClientConfig().setEveryTimeNew(isEveryTimeNew);
    }

    public void setEnvironment(String environment) {
        clientContext.getClientConfig().setEnvironment(environment);
        clientContext.getNodeConfig().setEnvironment(environment);
    }

    public void setServiceGroup(String serviceGroup) {
        clientContext.getClientConfig().setServiceGroup(serviceGroup);
    }

    public void setServiceGroupId(String serviceGroupId) {
        clientContext.getClientConfig().setServiceGroupId(serviceGroupId);
    }

    public void setRegionName(String regionName) {
        clientContext.getClientConfig().setRegionName(regionName);
    }

    public void setInnerDebug(int innerDebug) {
        clientContext.getClientConfig().setInnerDebug(innerDebug);
    }

    public void setNewVersion(boolean newVersion) {
        clientContext.getClientConfig().setNewVersion(newVersion);
    }

    public void setDomainName(String domainName) {
        if (StringUtil.isBlank(domainName) || "default".equals(domainName)) {
            return;
        }
        clientContext.getClientConfig().setDomainName(domainName);
    }

    public void setAutoInit(boolean autoInit) {
        clientContext.getClientConfig().setAutoInit(autoInit);
    }

    public ClassLoader getCustomerClassLoader() {
        return clientContext.getProxyService().getCustomerClassLoader();
    }

    public void setCustomerClassLoader(ClassLoader customerClassLoader) {
        clientContext.getProxyService().setCustomerClassLoader(customerClassLoader);
    }

    public SchedulerXLogLevel getDefaultLogLevel() {
        return defaultLogLevel;
    }

    public void setDefaultLogLevel(SchedulerXLogLevel defaultLogLevel) {
        this.defaultLogLevel = defaultLogLevel;
    }

    public SchedulerXLogLevel getExecuteLogLevel() {
        return executeLogLevel;
    }

    public void setExecuteLogLevel(SchedulerXLogLevel executeLogLevel) {
        this.executeLogLevel = executeLogLevel;
    }

    public void setMaxBodySize(long maxBodySize) {
        clientContext.getClientConfig().setMaxBodySize(maxBodySize);
    }

    //node config start

    public void setListenPort(int listenerPort) {
        clientContext.getNodeConfig().setListenPort(listenerPort);
    }

    public void setSystemListenPort(int systemListenPort) {
        clientContext.getNodeConfig().setSystemListenPort(systemListenPort);
    }


    public void setLocalAddress(String localAddress) {
        clientContext.getNodeConfig().setLocalAddress(localAddress);
    }

    public void setConnectTimeout(long connectTimeout) {
        clientContext.getNodeConfig().setConnectTimeout(connectTimeout);
    }

    public void setHeartbeatInterval(long heartbeatInterval) {
        clientContext.getNodeConfig().setHeartbeatInterval(heartbeatInterval);
    }

    public void setDbPath(String dbPath) {
        clientContext.getNodeConfig().setDbPath(dbPath);
    }

    public void setReceiveBufferSize(int bufferSize) {
        if (bufferSize > 0)
            clientContext.getNodeConfig().setReceiveBufferSize(bufferSize);
    }

    public void setRouteMap(Map<String, RouteRule> routeMap) {
        clientContext.getNodeConfig().setRouteMap(routeMap);
    }


    public boolean isDispatchOnly() {
        return clientContext.getNodeConfig().isDispatchOnly();
    }

    /**
     * 是否执行自己分发的任务, true不执行, false执行, 默认false
     *
     *  dispatchOnly
     */
    public void setDispatchOnly(boolean dispatchOnly) {
        clientContext.getNodeConfig().setDispatchOnly(dispatchOnly);
    }


    public void setFlowControlStrategy(FlowControlStrategy flowControlStrategy) {
        clientContext.getNodeConfig().setFlowControlStrategy(flowControlStrategy);
    }

    public void setStoreStrategy(StoreStrategy storeStrategy) {
        clientContext.getNodeConfig().setStoreStrategy(storeStrategy);
    }

    public boolean isEnableRedispatch() {
        return clientContext.getNodeConfig().isEnableRedispatch();
    }

    public void setEnableRedispatch(boolean enableRedispatch) {
        clientContext.getNodeConfig().setEnableRedispatch(enableRedispatch);
    }

    public void setFlowControlStrategyMap(Map<String, FlowControlStrategy> consumerThreadsMap) {
        clientContext.getNodeConfig().setFlowControlStrategyMap(consumerThreadsMap);
    }

    public void setEnableIsolateEnvSupport(boolean enableIsolateEnvSupport) {
        clientContext.getClientConfig().setEnableIsolateEnvSupport(enableIsolateEnvSupport);
    }

    public void setRouteQueueSize(int routeQueueSize) {
        if (routeQueueSize <= 0) {
            return;
        }
        clientContext.getGridTaskSender().getSendManager().setRouteQueueSize(routeQueueSize);
    }

    public void setMergeQueueSize(int mergeQueueSize) {
        if (mergeQueueSize <= 0) {
            return;
        }
        clientContext.getGridTaskSender().getSendManager().setMergeQueueSize(mergeQueueSize);
    }

    public void setSendQueueSize(int sendQueueSize) {
        if (sendQueueSize <= 0) {
            return;
        }
        clientContext.getGridTaskSender().getSendManager().setSendQueueSize(sendQueueSize);
    }

    //node config end
}
