package com.alibaba.dts.client.executor.job.factory;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;


import com.alibaba.dts.client.executor.job.processor.*;
import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;
import org.springframework.aop.TargetClassAware;
import org.springframework.aop.framework.Advised;
import org.springframework.aop.support.AopUtils;
import org.springframework.context.ApplicationContext;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import com.alibaba.dts.client.executor.job.context.ClientContextImpl;
import com.alibaba.dts.common.constants.Constants;
import com.alibaba.dts.common.context.InvocationContext;
import com.alibaba.dts.common.domain.remoting.RemoteMachine;
import com.alibaba.dts.common.domain.result.Result;
import com.alibaba.dts.common.domain.store.Job;
import com.alibaba.dts.common.exception.InitException;
import com.alibaba.dts.common.service.ServerService;
import com.alibaba.dts.common.util.CommonUtil;
import com.alibaba.dts.common.util.RandomUtil;
import com.alibaba.dts.common.util.StringUtil;
import com.alibaba.edas.schedulerx.ScxSimpleJobProcessor;


public class JobProcessorFactory implements Constants {

    private static final Logger logger = SchedulerXLoggerFactory.getLogger(JobProcessorFactory.class);

    /**
     * spring bean上下文
     */
    private ApplicationContext applicationContext = null;

    /**
     * SimpleJobProcessor缓存
     */
    private ConcurrentHashMap<String, ConcurrentHashMap<String, SimpleJobProcessor>> simpleJobProcessorCache =
            new ConcurrentHashMap<String, ConcurrentHashMap<String, SimpleJobProcessor>>();

    /**
     * ScxSimpleJobProcessor缓存
     */
    private ConcurrentHashMap<String, ConcurrentHashMap<String, ScxSimpleJobProcessor>> scxSimpleJobProcessorCache =
            new ConcurrentHashMap<String, ConcurrentHashMap<String, ScxSimpleJobProcessor>>();

    /**
     * ParallelJobProcessor缓存
     */
    private ConcurrentHashMap<String, ConcurrentHashMap<String, ParallelJobProcessor>> parallelJobProcessorCache =
            new ConcurrentHashMap<String, ConcurrentHashMap<String, ParallelJobProcessor>>();


    /**
     * GridJobProcessor缓存
     */
    private ConcurrentHashMap<String, ConcurrentHashMap<String, GridJobProcessor>> gridJobProcessorCache =
            new ConcurrentHashMap<String, ConcurrentHashMap<String, GridJobProcessor>>();


    /**
     * LongTimeJobProcessor缓存
     */
    private ConcurrentHashMap<String, ConcurrentHashMap<String, LongTimeJobProcessor>> longTimeJobProcessorCache =
            new ConcurrentHashMap<String, ConcurrentHashMap<String, LongTimeJobProcessor>>();

    private final ServerService serverService;

    private final ClientContextImpl clientContext;

    public JobProcessorFactory(final ClientContextImpl clientContext) {
        this.clientContext = clientContext;

        this.serverService = this.clientContext.getClientRemoting().proxyInterface(ServerService.class);
    }

    public void init() throws InitException {

		if(this.clientContext.getClientConfig().isNewInstance()) {
			/** 初始化检查Job处理器 */
			initCheckJobProcessors();
		}

        /** Spring环境初始化 */
        if (this.clientContext.getClientConfig().isSpring()) {

            /** 初始化SimpleJobProcessor缓存 */
            initSimpleJobProcessorCache4Spring();

            /** 初始化ParallelJobProcessor缓存 */
            initParallelJobProcessorCache4Spring();

            initGridJobProcessorCache4Spring();

        }

        if (Constants.ENVIRONMENT_JST.equals(this.clientContext.getClientConfig().getEnvironment())) {

            //初始化注册Job
            initRegisterJobs();
        }
    }


    public void initRegisterJobs() throws InitException {

        Map<String, String> jobMap = this.clientContext.getClientConfig().getJobMap();

        if (CollectionUtils.isEmpty(jobMap)) {
            logger.warn("[JobProcessorFactory]: jobMap is empty after check, clientConfig:" + this.clientContext.getClientConfig());
            return;
        }

        List<String> serverList = this.clientContext.getClientRemoting().getServerList();
        if (CollectionUtils.isEmpty(serverList)) {
            logger.warn("[JobProcessorFactory]: initRegisterJobs error, serverList is empty");
            return;
        }
        for (String server : serverList) {

            //向服务器注册JobMap
            initRegisterJobMap(server, jobMap);
        }
    }

    private void initRegisterJobMap(String server, Map<String, String> jobMap) {

        //注册到服务器
        InvocationContext.setRemoteMachine(new RemoteMachine(server, 10 * DEFAULT_INVOKE_TIMEOUT));
        Result<Boolean> registerResult = serverService.registerJobs(this.clientContext.getClientConfig().getMachine(), jobMap);

        if (null == registerResult) {
            logger.error("[JobProcessorFactory]: initRegisterJobMap timeout error"
                    + ", jobMap:" + jobMap + ", machine:" + this.clientContext.getClientConfig().getMachine());
            return;
        }

        logger.warn("[JobProcessorFactory]: initRegisterJobMap"
                + ", registerResult:" + registerResult + ", machine:" + this.clientContext.getClientConfig().getMachine());
    }


    private String getTargetClassName(Object processObject){

        String className = "";

        try{

            while (processObject instanceof Advised)
            {
                processObject =  ((Advised)processObject).getTargetSource().getTarget();
            }

            className = processObject.getClass().getName();

        }catch (Exception e){
            logger.error("[JobProcessorFactory]: getTargetClassName  error ",e);
        }

        return className;

    }

    @SuppressWarnings("rawtypes")
    private void initSimpleJobProcessorCache4Spring() {
        Map<String, SimpleJobProcessor> beanMap = applicationContext.getBeansOfType(SimpleJobProcessor.class);
        if (null == beanMap || beanMap.isEmpty()) {
            logger.warn("[JobProcessorFactory]: initSimpleJobProcessorCache beanMap is empty");
            return;
        }
        Iterator iterator = beanMap.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry entry = (Map.Entry) iterator.next();
            String beanId = (String) entry.getKey();
            SimpleJobProcessor simpleJobProcessor = (SimpleJobProcessor) entry.getValue();
            String jobProcessor = simpleJobProcessor.getClass().getName();
            if (AopUtils.isAopProxy(simpleJobProcessor)
                    || AopUtils.isCglibProxy(simpleJobProcessor)
                    || AopUtils.isJdkDynamicProxy(simpleJobProcessor)) {
                jobProcessor = getTargetClassName(simpleJobProcessor);
            }
            ConcurrentHashMap<String, SimpleJobProcessor> simpleJobProcessorMap = this.simpleJobProcessorCache.get(jobProcessor);
            if (null == simpleJobProcessorMap) {
                simpleJobProcessorMap = new ConcurrentHashMap<String, SimpleJobProcessor>();
                this.simpleJobProcessorCache.put(jobProcessor, simpleJobProcessorMap);
            }
            simpleJobProcessorMap.put(beanId, simpleJobProcessor);
            logger.warn("[JobProcessorFactory]: initSimpleJobProcessorCache jobProcessor:" + jobProcessor + ", beanId:" + beanId);
        }
    }

    @SuppressWarnings("rawtypes")
    private void initParallelJobProcessorCache4Spring() {
        Map<String, ParallelJobProcessor> beanMap = applicationContext.getBeansOfType(ParallelJobProcessor.class);
        if (null == beanMap || beanMap.isEmpty()) {
            logger.warn("[JobProcessorFactory]: initParallelJobProcessorCache beanMap is empty");
            return;
        }
        Iterator iterator = beanMap.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry entry = (Map.Entry) iterator.next();
            String beanId = (String) entry.getKey();
            ParallelJobProcessor parallelJobProcessor = (ParallelJobProcessor) entry.getValue();
            String jobProcessor = parallelJobProcessor.getClass().getName();
            if (AopUtils.isAopProxy(parallelJobProcessor)
                    || AopUtils.isCglibProxy(parallelJobProcessor)
                    || AopUtils.isJdkDynamicProxy(parallelJobProcessor)) {
                jobProcessor = AopUtils.getTargetClass(parallelJobProcessor).getName();
            }
            ConcurrentHashMap<String, ParallelJobProcessor> parallelJobProcessorMap = this.parallelJobProcessorCache.get(jobProcessor);
            if (null == parallelJobProcessorMap) {
                parallelJobProcessorMap = new ConcurrentHashMap<String, ParallelJobProcessor>();
                this.parallelJobProcessorCache.put(jobProcessor, parallelJobProcessorMap);
            }
            parallelJobProcessorMap.put(beanId, parallelJobProcessor);
            logger.warn("[JobProcessorFactory]: initParallelJobProcessorCache jobProcessor:" + jobProcessor + ", beanId:" + beanId);
        }
    }

    private void initGridJobProcessorCache4Spring() {
        Map<String, GridJobProcessor> beanMap = applicationContext.getBeansOfType(GridJobProcessor.class);
        if (null == beanMap || beanMap.isEmpty()) {
            logger.warn("[JobProcessorFactory]: initParallelJobProcessorCache beanMap is empty");
            return;
        }
        Iterator iterator = beanMap.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry entry = (Map.Entry) iterator.next();
            String beanId = (String) entry.getKey();
            GridJobProcessor gridJobProcessor = (GridJobProcessor) entry.getValue();
            String jobProcessor = gridJobProcessor.getClass().getName();
            if (AopUtils.isAopProxy(gridJobProcessor)
                    || AopUtils.isCglibProxy(gridJobProcessor)
                    || AopUtils.isJdkDynamicProxy(gridJobProcessor)) {
                jobProcessor = AopUtils.getTargetClass(gridJobProcessor).getName();
            }
            ConcurrentHashMap<String, GridJobProcessor> gridJobProcessorMap = this.gridJobProcessorCache.get(jobProcessor);
            if (null == gridJobProcessorMap) {
                gridJobProcessorMap = new ConcurrentHashMap<String, GridJobProcessor>();
                this.gridJobProcessorCache.put(jobProcessor, gridJobProcessorMap);
            }
            gridJobProcessorMap.put(beanId, gridJobProcessor);
            logger.warn("[JobProcessorFactory]: initGridlJobProcessorCache jobProcessor:" + jobProcessor + ", beanId:" + beanId);
        }

    }


    @SuppressWarnings("rawtypes")
    private void initCheckJobProcessors() throws InitException {

        if (Constants.ENVIRONMENT_JST.equals(this.clientContext.getClientConfig().getEnvironment())) {

            Map<String, String> jobMap = this.clientContext.getClientConfig().getJobMap();

            if (CollectionUtils.isEmpty(jobMap)) {
                logger.warn("[JobProcessorFactory]: initCheckJobProcessors jobMap is isEmpty error"
                        + ", clientConfig:" + this.clientContext.getClientConfig().toString());
                return;
            }

            Iterator iterator = jobMap.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry entry = (Map.Entry) iterator.next();

                String taskName = (String) entry.getKey();
                String jobProcessor = (String) entry.getValue();

                Job job = new Job();
                job.setTaskName(taskName);
                job.setJobProcessor(jobProcessor);

                createAndGetSimpleJobProcessor(job, true);

                createAndGetParallelJobProcessor(job, true);
                createAndGetGridJobProcessor(job, true);
            }

            return;
        }

        List<String> serverList = this.clientContext.getClientRemoting().getServerList();
        if (CollectionUtils.isEmpty(serverList)) {
            logger.error("[JobProcessorFactory]: initCheckJobProcessors serverList is isEmpty error");
            return;
        }

        InvocationContext.setRemoteMachine(new RemoteMachine(RandomUtil.getRandomObj(serverList)));
        List<Job> jobList = serverService.acquireJobList(this.clientContext.getClientConfig().getGroupId());
        if (CollectionUtils.isEmpty(jobList)) {
            logger.warn("[JobProcessorFactory]: acquireJobList jobList is isEmpty error");
            return;
        }

        for (Job job : jobList) {
            try {
                if (CommonUtil.isSimpleJob(job.getType())) {

                    if (Constants.ENVIRONMENT_SCX.equals(this.clientContext.getClientConfig().getEnvironment()) && !this.clientContext.getClientConfig().isNewVersion()) {
                        createAndGetScxSimpleJobProcessor(job, true);
                    } else {
                        createAndGetSimpleJobProcessor(job, true);
                    }

                } else if (CommonUtil.isGridJob(job.getType())) {
                    createAndGetGridJobProcessor(job, true);
                } else {
                    if (CommonUtil.isLongTimeJob(job.getType())) {
                        createAndGetLongTimeJobProcessor(job, true);
                    } else {
                        if (CommonUtil.isParallelJob(job.getType()))
                          createAndGetParallelJobProcessor(job, true);
                    }
                }
            } catch (Throwable e) {
                logger.warn("[JobProcessorFactory]: initCheckJobProcessors error", e);
            }
        }

    }

    public SimpleJobProcessor createAndGetSimpleJobProcessor(Job job, boolean isCheck) {
        String[] jobProcessorProperties = job.getJobProcessor().split(COLON);
        String jobProcessor = jobProcessorProperties[POSITION_PROCESSOR].trim();

        ConcurrentHashMap<String, SimpleJobProcessor> simpleJobProcessorMap = this.simpleJobProcessorCache.get(jobProcessor);
        if (null == simpleJobProcessorMap && !this.clientContext.getClientConfig().isEveryTimeNew()) {
            simpleJobProcessorMap = new ConcurrentHashMap<String, SimpleJobProcessor>();
            this.simpleJobProcessorCache.put(jobProcessor, simpleJobProcessorMap);
        }

        SimpleJobProcessor simpleJobProcessor = null;

        if (!CollectionUtils.isEmpty(simpleJobProcessorMap) && !this.clientContext.getClientConfig().isEveryTimeNew()) {

            if (1 == simpleJobProcessorMap.size()) {
                simpleJobProcessor = (SimpleJobProcessor) simpleJobProcessorMap.values().toArray()[0];
                return simpleJobProcessor;
            }

            if (jobProcessorProperties.length < 3) {
                simpleJobProcessor = simpleJobProcessorMap.get(jobProcessor);
                if (simpleJobProcessor != null) {
                    return simpleJobProcessor;
                }
                throw new RuntimeException("[JobProcessorFactory]: you have more than one jobProcessor instance for " + job.getJobProcessor()
                        + ", but you do not fill beanId on console. please check job config, jobId:" + job.getId()
                        + "! you should fill the config item like this 'com.xxx.app.JobProcessor::beanId'.");
            } else {
                String beanId = jobProcessorProperties[POSITION_BEAN_ID].trim();
                if (StringUtil.isNotBlank(beanId)) {
                    simpleJobProcessor = simpleJobProcessorMap.get(beanId);
                    if (simpleJobProcessor != null) {
                        return simpleJobProcessor;
                    }
                }
                throw new RuntimeException("[JobProcessorFactory]: you have more than one jobProcessor instance for " + job.getJobProcessor()
                        + ". you maybe fill wrong beanId on console. please check job config, jobId:" + job.getId()
                        + "! DtsClient can not find bean:" + beanId);
            }
        }


		if(! this.clientContext.getClientConfig().isNewInstance()) {
			return simpleJobProcessor;
		}

        Object object = this.clientContext.getProxyService().newInstance(jobProcessor);
        if (null == object) {
            throw new RuntimeException("[JobProcessorFactory]: can not create a new simple job processor, please check:" + job.getJobProcessor());
        }

        Type[] types = this.clientContext.getProxyService().aquireInterface(object);
        if (0 == types.length) {
            throw new RuntimeException("[JobProcessorFactory]: your choice is simple job"
                    + ", but not implements " + SimpleJobProcessor.class.getName()
                    + ", please check:" + job.getJobProcessor());
        }

        int result = checkInterface(types, job);

        if (result == 1) simpleJobProcessor = (SimpleJobProcessor) object;


        if (this.clientContext.getClientConfig().isSpring()) {
            initSpringJobProcessor(job, simpleJobProcessor);
        }

        if (!isCheck && !this.clientContext.getClientConfig().isEveryTimeNew()) {
            simpleJobProcessorMap.put(jobProcessor, simpleJobProcessor);
        }
        return simpleJobProcessor;
    }

    public ScxSimpleJobProcessor createAndGetScxSimpleJobProcessor(Job job, boolean isCheck) {
        String[] jobProcessorProperties = job.getJobProcessor().split(COLON);
        String jobProcessor = jobProcessorProperties[POSITION_PROCESSOR].trim();

        ConcurrentHashMap<String, ScxSimpleJobProcessor> scxSimpleJobProcessorMap = this.scxSimpleJobProcessorCache.get(jobProcessor);
        if (null == scxSimpleJobProcessorMap && !this.clientContext.getClientConfig().isEveryTimeNew()) {
            scxSimpleJobProcessorMap = new ConcurrentHashMap<String, ScxSimpleJobProcessor>();
            this.scxSimpleJobProcessorCache.put(jobProcessor, scxSimpleJobProcessorMap);
        }

        ScxSimpleJobProcessor scxSimpleJobProcessor = null;

        if (!CollectionUtils.isEmpty(scxSimpleJobProcessorMap) && !this.clientContext.getClientConfig().isEveryTimeNew()) {

            if (1 == scxSimpleJobProcessorMap.size()) {
                scxSimpleJobProcessor = (ScxSimpleJobProcessor) scxSimpleJobProcessorMap.values().toArray()[0];
                return scxSimpleJobProcessor;
            }

			if(jobProcessorProperties.length < 3) {
				scxSimpleJobProcessor = scxSimpleJobProcessorMap.get(jobProcessor);
				if(scxSimpleJobProcessor != null) {
					return scxSimpleJobProcessor;
				}
				throw new RuntimeException("[JobProcessorFactory]: you have more than one jobProcessor instance for " + job.getJobProcessor()
						+ ", but you do not fill beanId on console. please check job config, jobId:" + job.getId()
						+ "! you should fill the config item like this 'com.xxx.app.JobProcessor::beanId'.");
			} else {
				String beanId = jobProcessorProperties[POSITION_BEAN_ID].trim();
				if(StringUtil.isNotBlank(beanId)) {
					scxSimpleJobProcessor = scxSimpleJobProcessorMap.get(beanId);
					if(scxSimpleJobProcessor != null) {
						return scxSimpleJobProcessor;
					}
				}
				throw new RuntimeException("[JobProcessorFactory]: you have more than one jobProcessor instance for " + job.getJobProcessor()
						+ ". you maybe fill wrong beanId on console. please check job config, jobId:" + job.getId()
						+ "! DtsClient can not find bean:" + beanId);
			}
		}

		if(! this.clientContext.getClientConfig().isNewInstance()) {
			return scxSimpleJobProcessor;
		}

        Object object = this.clientContext.getProxyService().newInstance(jobProcessor);
        if (null == object) {
            throw new RuntimeException("[JobProcessorFactory]: can not create a new simple job processor, please check:" + job.getJobProcessor());
        }

        Type[] types = this.clientContext.getProxyService().aquireInterface(object);
        if (0 == types.length) {
            throw new RuntimeException("[JobProcessorFactory]: your choice is simple job"
                    + ", but not implements " + SimpleJobProcessor.class.getName()
                    + ", please check:" + job.getJobProcessor());
        }

        int result = checkInterface(types, job);

        if (result == 2) scxSimpleJobProcessor = (ScxSimpleJobProcessor) object;

        if (this.clientContext.getClientConfig().isSpring()) {
            initSpringJobProcessor(job, scxSimpleJobProcessor);
        }

        if (!isCheck && !this.clientContext.getClientConfig().isEveryTimeNew()) {
            scxSimpleJobProcessorMap.put(jobProcessor, scxSimpleJobProcessor);
        }
        return scxSimpleJobProcessor;
    }


    public ParallelJobProcessor createAndGetParallelJobProcessor(Job job, boolean isCheck) {
        String[] jobProcessorProperties = job.getJobProcessor().split(COLON);
        String jobProcessor = jobProcessorProperties[POSITION_PROCESSOR].trim();

        ConcurrentHashMap<String, ParallelJobProcessor> parallelJobProcessorMap = this.parallelJobProcessorCache.get(jobProcessor);
        if (null == parallelJobProcessorMap && !this.clientContext.getClientConfig().isEveryTimeNew()) {
            parallelJobProcessorMap = new ConcurrentHashMap<String, ParallelJobProcessor>();
            this.parallelJobProcessorCache.put(jobProcessor, parallelJobProcessorMap);
        }

        ParallelJobProcessor parallelJobProcessor = null;

        if (!CollectionUtils.isEmpty(parallelJobProcessorMap) && !this.clientContext.getClientConfig().isEveryTimeNew()) {
            if (1 == parallelJobProcessorMap.size()) {
                parallelJobProcessor = (ParallelJobProcessor) parallelJobProcessorMap.values().toArray()[0];
                return parallelJobProcessor;
            }

            if (jobProcessorProperties.length < 3) {
                parallelJobProcessor = parallelJobProcessorMap.get(jobProcessor);
                if (parallelJobProcessor != null) {
                    return parallelJobProcessor;
                }
                throw new RuntimeException("[JobProcessorFactory]: you have more than one jobProcessor instance for " + job.getJobProcessor()
                        + ", but you do not fill beanId on console. please check job config, jobId:" + job.getId()
                        + "! you should fill the config item like this 'com.xxx.app.JobProcessor::beanId'.");
            } else {
                String beanId = jobProcessorProperties[POSITION_BEAN_ID].trim();
                if (StringUtil.isNotBlank(beanId)) {
                    parallelJobProcessor = parallelJobProcessorMap.get(beanId);
                    if (parallelJobProcessor != null) {
                        return parallelJobProcessor;
                    }
                }
                throw new RuntimeException("[JobProcessorFactory]: you have more than one jobProcessor instance for " + job.getJobProcessor()
                        + ". you maybe fill wrong beanId on console. please check job config, jobId:" + job.getId()
                        + "! DtsClient can not find bean:" + beanId);
            }
        }

		if(! this.clientContext.getClientConfig().isNewInstance()) {
			return parallelJobProcessor;
		}

		Object object = this.clientContext.getProxyService().newInstance(jobProcessor);
		if(null == object) {
			throw new RuntimeException("[JobProcessorFactory]: can not create a new parallel job processor, please check:" + job.getJobProcessor());
		}

        Type[] types = this.clientContext.getProxyService().aquireInterface(object);
        if (0 == types.length) {
            throw new RuntimeException("[JobProcessorFactory]: your choice is parallel job"
                    + ", but not implements " + ParallelJobProcessor.class.getName()
                    + ", please check:" + job.getJobProcessor());
        }

        checkInterface(types, job);

        parallelJobProcessor = (ParallelJobProcessor) object;

        if (this.clientContext.getClientConfig().isSpring()) {
            initSpringJobProcessor(job, parallelJobProcessor);
        }

        if (!isCheck && !this.clientContext.getClientConfig().isEveryTimeNew()) {
            parallelJobProcessorMap.put(jobProcessor, parallelJobProcessor);
        }
        return parallelJobProcessor;
    }


    public LongTimeJobProcessor createAndGetLongTimeJobProcessor(Job job, boolean isCheck) {
        String[] jobProcessorProperties = job.getJobProcessor().split(COLON);
        String jobProcessor = jobProcessorProperties[POSITION_PROCESSOR].trim();

        ConcurrentHashMap<String, LongTimeJobProcessor> longTimeJobProcessorMap = this.longTimeJobProcessorCache.get(jobProcessor);
        if (null == longTimeJobProcessorMap && !this.clientContext.getClientConfig().isEveryTimeNew()) {
            longTimeJobProcessorMap = new ConcurrentHashMap<String, LongTimeJobProcessor>();
            this.longTimeJobProcessorCache.put(jobProcessor, longTimeJobProcessorMap);
        }

        LongTimeJobProcessor longTimeJobProcessor = null;

        if (!CollectionUtils.isEmpty(longTimeJobProcessorMap) && !this.clientContext.getClientConfig().isEveryTimeNew()) {
            if (1 == longTimeJobProcessorMap.size()) {
                longTimeJobProcessor = (LongTimeJobProcessor) longTimeJobProcessorMap.values().toArray()[0];
                return longTimeJobProcessor;
            }

			if(jobProcessorProperties.length < 3) {
				longTimeJobProcessor = longTimeJobProcessorMap.get(jobProcessor);
				if(longTimeJobProcessor != null) {
					return longTimeJobProcessor;
				}
				throw new RuntimeException("[JobProcessorFactory]: you have more than one jobProcessor instance for " + job.getJobProcessor()
						+ ", but you do not fill beanId on console. please check job config, jobId:" + job.getId()
						+ "! you should fill the config item like this 'com.xxx.app.JobProcessor::beanId'.");
			} else {
				String beanId = jobProcessorProperties[POSITION_BEAN_ID].trim();
				if(StringUtil.isNotBlank(beanId)) {
					longTimeJobProcessor = longTimeJobProcessorMap.get(beanId);
					if(longTimeJobProcessor != null) {
						return longTimeJobProcessor;
					}
				}
				throw new RuntimeException("[JobProcessorFactory]: you have more than one jobProcessor instance for " + job.getJobProcessor()
						+ ". you maybe fill wrong beanId on console. please check job config, jobId:" + job.getId()
						+ "! DtsClient can not find bean:" + beanId);
			}
		}

		if(! this.clientContext.getClientConfig().isNewInstance()) {
			return longTimeJobProcessor;
		}

        Object object = this.clientContext.getProxyService().newInstance(jobProcessor);
        if (null == object) {
            throw new RuntimeException("[JobProcessorFactory]: can not create a new longtime job processor, please check:" + job.getJobProcessor());
        }

        Type[] types = this.clientContext.getProxyService().aquireInterface(object);
        if (0 == types.length) {
            throw new RuntimeException("[JobProcessorFactory]: your choice is longtime job"
                    + ", but not implements " + ParallelJobProcessor.class.getName()
                    + ", please check:" + job.getJobProcessor());
        }

        checkLongTimeInterface(types, job);

        longTimeJobProcessor = (LongTimeJobProcessor) object;

        if (this.clientContext.getClientConfig().isSpring()) {
            initSpringJobProcessor(job, longTimeJobProcessor);
        }

        if (!isCheck && !this.clientContext.getClientConfig().isEveryTimeNew()) {
            longTimeJobProcessorMap.put(jobProcessor, longTimeJobProcessor);
        }
        return longTimeJobProcessor;
    }


    private void checkLongTimeInterface(Type[] types, Job job) {
        boolean longTime = false;
        for (Type type : types) {
            if (type.equals(LongTimeJobProcessor.class)) {
                longTime = true;
            }
        }
        if (!longTime) {
            throw new RuntimeException("[JobProcessorFactory]: your choice is longTime job"
                    + ", but not implements " + LongTimeJobProcessor.class.getName()
                    + ", please check:" + job.getJobProcessor());
        }
    }

    private int checkInterface(Type[] types, Job job) {
        boolean simple = false;
        boolean parallel = false;
        boolean grid = false;
        int result = 0;
        for (Type type : types) {
            if (type.equals(SimpleJobProcessor.class)) {
                simple = true;
                result = 1;
            }

            if (type.equals(ScxSimpleJobProcessor.class)) {
                simple = true;
                result = 2;
            }

            if (type.equals(ParallelJobProcessor.class)) {
                parallel = true;
                result = 3;
            }

            if (type.equals(GridJobProcessor.class)) {
                grid = true;
                result = 4;
            }
        }
        if (CommonUtil.isSimpleJob(job.getType())) {
            if (simple && parallel) {
                throw new RuntimeException("[JobProcessorFactory]: your choice is simple job"
                        + ", can not implements both " + SimpleJobProcessor.class.getName()
                        + " and " + ParallelJobProcessor.class.getName()
                        + ", please check:" + job.getJobProcessor());
            } else if (simple && !parallel) {

            } else if (!simple && parallel) {
                throw new RuntimeException("[JobProcessorFactory]: your choice is simple job"
                        + ", but implements " + ParallelJobProcessor.class.getName()
                        + ", please check:" + job.getJobProcessor());
            } else {
                throw new RuntimeException("[JobProcessorFactory]: your choice is simple job"
                        + ", but not implements " + SimpleJobProcessor.class.getName()
                        + ", please check:" + job.getJobProcessor());
            }
        } else if (CommonUtil.isParallelJob(job.getType())) {
            if (simple && parallel) {
                throw new RuntimeException("[JobProcessorFactory]: your choice is parallel job"
                        + ", can not implements both " + SimpleJobProcessor.class.getName()
                        + " and " + ParallelJobProcessor.class.getName()
                        + ", please check:" + job.getJobProcessor());
            } else if (simple && !parallel) {
                throw new RuntimeException("[JobProcessorFactory]: your choice is parallel job"
                        + ", but implements " + SimpleJobProcessor.class.getName()
                        + ", please check:" + job.getJobProcessor());
            } else if (!simple && parallel) {

            } else {
                throw new RuntimeException("[JobProcessorFactory]: your choice is parallel job"
                        + ", but not implements " + ParallelJobProcessor.class.getName()
                        + ", please check:" + job.getJobProcessor());
            }
        } else if (CommonUtil.isGridJob(job.getType())) {
            if (simple && parallel && grid) {
                throw new RuntimeException("[JobProcessorFactory]: your choice is grid job"
                        + ", can not implements both " + GridJobProcessor.class.getName()
                        + " and " + SimpleJobProcessor.class.getName()
                        + " and " + ParallelJobProcessor.class.getName()
                        + ", please check:" + job.getJobProcessor());
            } else if (simple && parallel) {
                throw new RuntimeException("[JobProcessorFactory]: your choice is grid job"
                        + ", but implements " + SimpleJobProcessor.class.getName()
                        + " and " + ParallelJobProcessor.class.getName()
                        + ", please check:" + job.getJobProcessor());
            } else if (simple) {
                throw new RuntimeException("[JobProcessorFactory]: your choice is grid job"
                        + ", but implements " + SimpleJobProcessor.class.getName()
                        + ", please check:" + job.getJobProcessor());
            } else if (parallel) {
                throw new RuntimeException("[JobProcessorFactory]: your choice is grid job"
                        + ", but implements " + ParallelJobProcessor.class.getName()
                        + ", please check:" + job.getJobProcessor());
            } else if (grid) {
                //GridJob check succeeded
            } else {
                throw new RuntimeException("[JobProcessorFactory]: your choice is grid job"
                        + ", but not implements " + GridJobProcessor.class.getName()
                        + ", please check:" + job.getJobProcessor());
            }
        }
        return result;
    }

    private void initSpringJobProcessor(Job job, Object jobProcessor) {
        String[] jobProcessorProperties = job.getJobProcessor().split(COLON);

        /** 填充字段 */
        Field[] fields = jobProcessor.getClass().getDeclaredFields();
        for (int i = 0; i < fields.length; i++) {
            fields[i].setAccessible(true);
            String fieldName = fields[i].getName();
            Object object = null;
            try {
                object = applicationContext.getBean(fieldName);
            } catch (Throwable e) {
                logger.warn("[JobProcessorFactory]: initSpringJobProcessor field not found"
                        + ", jobProcessor:" + jobProcessorProperties[0].trim()
                        + ", fieldName:" + fieldName);
            }
            if (object != null) {
                try {
                    fields[i].set(jobProcessor, object);
                } catch (Throwable e) {
                    logger.error("[JobProcessorFactory]: initSpringJobProcessor field set error"
                            + ", jobProcessor:" + jobProcessorProperties[0].trim()
                            + ", fieldName:" + fieldName
                            + ", object:" + object, e);
                    continue;
                }
                logger.warn("[JobProcessorFactory]: initSpringJobProcessor set field"
                        + ", jobProcessor:" + jobProcessorProperties[0].trim()
                        + ", fieldName:" + fieldName);
            }
        }

        /** 调用初始化方法 */
        if (jobProcessorProperties.length >= 2) {
            String initMethod = jobProcessorProperties[1].trim();
            if (StringUtil.isNotBlank(initMethod)) {
                Method method = null;
                try {
                    method = jobProcessor.getClass().getDeclaredMethod(initMethod);
                } catch (Throwable e) {
                    logger.error("[JobProcessorFactory]: initSpringJobProcessor getDeclaredMethod error"
                            + ", jobProcessor:" + jobProcessorProperties[0].trim()
                            + ", initMethod:" + initMethod, e);
                }
                if (null == method) {
                    logger.error("[JobProcessorFactory]: initSpringJobProcessor getDeclaredMethod failed"
                            + ", jobProcessor:" + jobProcessorProperties[0].trim()
                            + ", initMethod:" + initMethod);
                } else {
                    try {
                        method.invoke(jobProcessor);
                    } catch (Throwable e) {
                        logger.error("[JobProcessorFactory]: initSpringJobProcessor invoke initMethod error"
                                + ", jobProcessor:" + jobProcessorProperties[0].trim()
                                + ", initMethod:" + initMethod, e);
                    }
                }
            } else {
                logger.error("[JobProcessorFactory]: initSpringJobProcessor initMethod is null"
                        + ", jobProcessor:" + jobProcessorProperties[0].trim()
                        + ", initMethod:" + initMethod);
            }
        }
    }

    public ApplicationContext getApplicationContext() {
        return applicationContext;
    }

    public void setApplicationContext(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }

    public GridJobProcessor createAndGetGridJobProcessor(Job job, boolean isCheck) {
        String[] jobProcessorProperties = job.getJobProcessor().split(COLON);
        String jobProcessor = jobProcessorProperties[POSITION_PROCESSOR].trim();

        ConcurrentHashMap<String, GridJobProcessor> gridJobProcessorMap = this.gridJobProcessorCache.get(jobProcessor);
        if (null == gridJobProcessorMap && !this.clientContext.getClientConfig().isEveryTimeNew()) {
            gridJobProcessorMap = new ConcurrentHashMap<String, GridJobProcessor>();
            this.gridJobProcessorCache.put(jobProcessor, gridJobProcessorMap);
        }

        GridJobProcessor gridJobProcessor = null;

        if (!CollectionUtils.isEmpty(gridJobProcessorMap) && !this.clientContext.getClientConfig().isEveryTimeNew()) {
            if (1 == gridJobProcessorMap.size()) {
                gridJobProcessor = (GridJobProcessor) gridJobProcessorMap.values().toArray()[0];
                return gridJobProcessor;
            }

            if (jobProcessorProperties.length < 3) {
                gridJobProcessor = gridJobProcessorMap.get(jobProcessor);
                if (gridJobProcessor != null) {
                    return gridJobProcessor;
                }
                throw new RuntimeException("[JobProcessorFactory]: you have more than one jobProcessor instance for " + job.getJobProcessor()
                        + ", but you do not fill beanId on console. please check job config, jobId:" + job.getId()
                        + "! you should fill the config item like this 'com.xxx.app.JobProcessor::beanId'.");
            } else {
                String beanId = jobProcessorProperties[POSITION_BEAN_ID].trim();
                if (StringUtil.isNotBlank(beanId)) {
                    gridJobProcessor = gridJobProcessorMap.get(beanId);
                    if (gridJobProcessor != null) {
                        return gridJobProcessor;
                    }
                }
                throw new RuntimeException("[JobProcessorFactory]: you have more than one jobProcessor instance for " + job.getJobProcessor()
                        + ". you maybe fill wrong beanId on console. please check job config, jobId:" + job.getId()
                        + "! DtsClient can not find bean:" + beanId);
            }
        }

        if(!this.clientContext.getClientConfig().isNewInstance()) {
            return gridJobProcessor;
        }

        Object object = this.clientContext.getProxyService().newInstance(jobProcessor);
        if (null == object) {
            throw new RuntimeException("[JobProcessorFactory]: can not create a new grid job processor, please check:" + job.getJobProcessor());
        }

        Type[] types = this.clientContext.getProxyService().aquireInterface(object);
        if (0 == types.length) {
            throw new RuntimeException("[JobProcessorFactory]: your choice is parallel job"
                    + ", but not implements " + ParallelJobProcessor.class.getName()
                    + ", please check:" + job.getJobProcessor());
        }

        checkInterface(types, job);

        gridJobProcessor = (GridJobProcessor) object;

        if (this.clientContext.getClientConfig().isSpring()) {
            initSpringJobProcessor(job, gridJobProcessor);
        }

        if (!isCheck && !this.clientContext.getClientConfig().isEveryTimeNew()) {
            gridJobProcessorMap.put(jobProcessor, gridJobProcessor);
        }
        return gridJobProcessor;
    }
}
