package com.alibaba.dts.client.executor.longtime;

import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import com.alibaba.dts.client.executor.job.context.ClientContextImpl;
import com.alibaba.dts.client.executor.job.processor.LongTimeJobProcessor;
import com.alibaba.dts.client.executor.longtime.unit.ExecutorUnit;
import com.alibaba.dts.common.constants.Constants;
import com.alibaba.dts.common.domain.ExecutableTask;
import com.alibaba.dts.common.domain.result.Result;
import com.alibaba.dts.common.domain.result.ResultCode;
import com.alibaba.dts.common.domain.store.TaskSnapshot;
import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;
import org.springframework.util.CollectionUtils;

/**
 * LongTime job执行池
 *
 */
public class LongTimePool implements Constants {

	private static final Logger logger = SchedulerXLoggerFactory.getLogger(LongTimePool.class);

	/** 任务执行单元映射表 */
	private ConcurrentHashMap<Long, ConcurrentHashMap<Long, ExecutorUnit>> executorUnitTable =
			new ConcurrentHashMap<Long, ConcurrentHashMap<Long, ExecutorUnit>>();

	private final ClientContextImpl clientContext;

	public LongTimePool(final ClientContextImpl clientContext){
		this.clientContext = clientContext;
	}

	public void stopService(){
		Iterator iterator = executorUnitTable.entrySet().iterator();

		long jobCount = 0;
		long processorCount = 0;

		while(iterator.hasNext()) {

			long jobid=0;
			jobCount++;
			try {

				Map.Entry entry = (Map.Entry)iterator.next();

				jobid = (Long)entry.getKey();

				ConcurrentHashMap<Long, ExecutorUnit> executorUnitMap = (ConcurrentHashMap<Long, ExecutorUnit>)entry.getValue();

				if(CollectionUtils.isEmpty(executorUnitMap)) {
					continue;
				}

				Iterator executorUnitIterator = executorUnitMap.entrySet().iterator();

				while(executorUnitIterator.hasNext()) {
					processorCount++;
					Map.Entry processorEntity = (Map.Entry)executorUnitIterator.next();
					ExecutorUnit processor = (ExecutorUnit)processorEntity.getValue();
					processor.stopTask();
				}

			} catch (Throwable e) {

				logger.error("[SimplePool]: stopService error, jobid:" + jobid, e);
			}

			logger.info("[SimplePool]: stopService, jobCount:"+jobCount +
					", processorCount:" + processorCount);
		}

		executorUnitTable.clear();
	}

	@SuppressWarnings("unchecked")
	public boolean executeTask(ExecutableTask executableTask) {
		ConcurrentHashMap<Long, ExecutorUnit> executorUnitMap = this.executorUnitTable.get(executableTask.getJob().getId());
		if(null == executorUnitMap) {
			if (!isImplProcessor(executableTask)) return false;
			executorUnitMap = new ConcurrentHashMap<Long, ExecutorUnit>();
			ConcurrentHashMap<Long, ExecutorUnit> existExecutorUnitMap =
			this.executorUnitTable.putIfAbsent(executableTask.getJob().getId(), executorUnitMap);
			if (existExecutorUnitMap!=null){
				executorUnitMap = existExecutorUnitMap;
			}
		}

		ExecutorUnit executorUnit = executorUnitMap.get(executableTask.getJobInstanceSnapshot().getId());
		if(null == executorUnit) {
			if (!isImplProcessor(executableTask)) return false;
			executorUnit = new ExecutorUnit(clientContext,this, executableTask);
			ExecutorUnit existExecutorUnit =
			executorUnitMap.putIfAbsent(executableTask.getJobInstanceSnapshot().getId(), executorUnit);
			if (existExecutorUnit!=null){
				executorUnit = null;
			}else{
				try {
					executorUnit.init();
					logger.info("[LongTimePool]: executeTask init success"
							+ ", instanceId:" + executableTask.getJobInstanceSnapshot().getId());
				} catch (Throwable e) {
					logger.error("[LongTimePool]: executeTask init error"
							+ ", instanceId:" + executableTask.getJobInstanceSnapshot().getId(), e);
					return false;
				}
			}
		}else{
			logger.info("[LongTimePool]: instanceId exists,restartPull"
					+ ", instanceId:" + executableTask.getJobInstanceSnapshot().getId());
			executorUnit.restartPull();
		}

		return true;
	}

	public ExecutorUnit getExecutorUnit(long jobid, long instanceid){
		ConcurrentHashMap<Long, ExecutorUnit> executorUnitMap = this.executorUnitTable.get(jobid);
		if(null == executorUnitMap) {
			return null;
		}
		ExecutorUnit executorUnit = executorUnitMap.get(instanceid);
		return executorUnit;
	}

	public boolean isImplProcessor(ExecutableTask executableTask){

		LongTimeJobProcessor longTimeJobProcessor = null;
		try {
			longTimeJobProcessor = this.clientContext.getJobProcessorFactory().createAndGetLongTimeJobProcessor(executableTask.getJob(), false);
		} catch (Throwable e) {
			logger.error("[LongTimePool]: LongTimeJobProcessor is not implement."
					+ ", jobProcessor:" + executableTask.getJob().getJobProcessor(), e);
			return false;
		}
		if (longTimeJobProcessor==null) return false;

		return true;
	}

	public boolean activeTask(ExecutableTask executableTask) {
		ConcurrentHashMap<Long, ExecutorUnit> executorUnitMap = this.executorUnitTable.get(executableTask.getJob().getId());
		if(null == executorUnitMap) {
			executorUnitMap = new ConcurrentHashMap<Long, ExecutorUnit>();
			this.executorUnitTable.put(executableTask.getJob().getId(), executorUnitMap);
		}

		ExecutorUnit executorUnit = executorUnitMap.get(executableTask.getJobInstanceSnapshot().getId());
		if(null == executorUnit) {
			executorUnit = new ExecutorUnit(clientContext,this, executableTask);
			try {
				executorUnit.activeInit();
				logger.info("[LongTimePool]: activeTask init success"
						+ ", instanceId:" + executableTask.getJobInstanceSnapshot().getId());
			} catch (Throwable e) {
				logger.error("[LongTimePool]: activeTask init error"
						+ ", instanceId:" + executableTask.getJobInstanceSnapshot().getId(), e);
				return false;
			}
			executorUnitMap.put(executableTask.getJobInstanceSnapshot().getId(), executorUnit);
		}

		return true;
	}


	public boolean releaseCompleteTask(ExecutableTask executableTask) {
		boolean result=true;
		try{

			ConcurrentHashMap<Long, ExecutorUnit> executorUnitMap = this.executorUnitTable.get(executableTask.getJob().getId());
			if(null == executorUnitMap) {
				return false;
			}
			ExecutorUnit executorUnit = executorUnitMap.get(executableTask.getJobInstanceSnapshot().getId());
			if(null == executorUnit) {
				return false;
			}else{
				executorUnit.releaseCompleteTask();
				logger.info("[LongTimePool]: activeTask releaseCompleteTask success"
						+ ", instanceId:" + executableTask.getJobInstanceSnapshot().getId());
			}

		}catch (Throwable e){
			result = false;
			logger.error("[LongTimePool]: releaseCompleteTask error"
					+ ", instanceId:" + executableTask.getJobInstanceSnapshot().getId(), e);
		}
		return result;
	}



	public boolean stopTask(long jobId, long jobInstanceId) {
		ConcurrentHashMap<Long, ExecutorUnit> executorUnitMap = this.executorUnitTable.get(jobId);
		if(null == executorUnitMap || executorUnitMap.isEmpty()) {
			return true;
		}
		ExecutorUnit executorUnit = executorUnitMap.get(jobInstanceId);
		if(null == executorUnit) {
			return true;
		}
		
		/** 停止任务执行单元 */
		executorUnit.stopTask();
		
		/** 删除任务执行单元 */
		try {
			executorUnitMap.remove(jobInstanceId);
			logger.info("[LongTimePool]: stopTask remove executorUnitMap success"
					+ ", instanceId:" + jobInstanceId);
		} catch (Throwable e) {
			logger.error("[LongTimePool]: stopTask remove executorUnitMap error"
					+ ", jobId:" + jobId 
					+ ", jobInstanceId:" + jobInstanceId, e);
			return false;
		}
		return true;
	}

	public boolean forceStopTask(long jobId, long jobInstanceId) {
		ConcurrentHashMap<Long, ExecutorUnit> executorUnitMap = this.executorUnitTable.get(jobId);
		if(null == executorUnitMap || executorUnitMap.isEmpty()) {
			return true;
		}
		ExecutorUnit executorUnit = executorUnitMap.get(jobInstanceId);
		if(null == executorUnit) {
			return true;
		}
		
		/** 强制停止任务执行单元 */
		executorUnit.forceStopTask();
		
		/** 删除任务执行单元 */
		try {
			executorUnitMap.remove(jobInstanceId);
			logger.info("[LongTimePool]: forceStopTask remove executorUnitMap success"
					+ ", instanceId:" + jobInstanceId);
		} catch (Throwable e) {
			logger.error("[LongTimePool]: forceStopTask remove executorUnitMap error"
					+ ", jobId:" + jobId 
					+ ", jobInstanceId:" + jobInstanceId, e);
			return false;
		}
		return true;
	}

	public Result<String> heartBeatCheckJobInstance(long jobId, long jobInstanceId) {
		Result<String> result = new Result<String>();
		ConcurrentHashMap<Long, ExecutorUnit> executorUnitMap = this.executorUnitTable.get(jobId);
		if(null == executorUnitMap || executorUnitMap.isEmpty()) {
			result.setResultCode(ResultCode.HEART_BEAT_CHECK_EXIT);
			return result;
		}
		ExecutorUnit executorUnit = executorUnitMap.get(jobInstanceId);
		if(null == executorUnit) {
			result.setResultCode(ResultCode.HEART_BEAT_CHECK_EXIT);
			return result;
		}
		result.setResultCode(ResultCode.HEART_BEAT_CHECK_SUCCESS);
		logger.info("[LongTimePool]: heartBeatCheckJobInstance"
				+ ", jobId:" + jobId
				+ ", jobInstanceId:" + jobInstanceId
				+ ", result:" + result.getResultCode().getInformation());
		return result;
	}
	

	public Result<Boolean> push(long jobId, long jobInstanceId, TaskSnapshot taskSnapshot) {
		
		ConcurrentHashMap<Long, ExecutorUnit> executorUnitMap = this.executorUnitTable.get(jobId);
		if(null == executorUnitMap || executorUnitMap.isEmpty()) {
			return new Result<Boolean>(false, ResultCode.PUSH_UNIT_MAP_IS_EMPTY_ERROR);
		}
		
		ExecutorUnit executorUnit = executorUnitMap.get(jobInstanceId);
		if(null == executorUnit) {
			return new Result<Boolean>(false, ResultCode.PUSH_UNIT_IS_NULL_ERROR);
		}
		boolean result = false;
		try{
			//将任务放入队列
			result = executorUnit.offer(taskSnapshot);
			logger.info("[LongTimePool]: push success"
					+ ", result:" + result
					+ ", jobId:" + jobId
					+ ", jobInstanceId:" + jobInstanceId
			);
		}catch (Throwable e){
			logger.error("[LongTimePool]: push error"
					+ ", jobId:" + jobId
					+ ", jobInstanceId:" + jobInstanceId, e);
		}

		
		return new Result<Boolean>(result, result ? ResultCode.SUCCESS : ResultCode.FAILURE);
	}
	
}
