package com.alibaba.dts.client.executor.longtime.processor;

import java.util.List;
import java.util.concurrent.BlockingQueue;

import com.alibaba.dts.client.executor.longtime.LongTimePool;
import org.springframework.util.CollectionUtils;

import com.alibaba.dts.client.executor.job.context.ClientContextImpl;
import com.alibaba.dts.client.executor.longtime.unit.ExecutorUnit;
import com.alibaba.dts.common.constants.Constants;
import com.alibaba.dts.common.domain.ExecutableTask;
import com.alibaba.dts.common.domain.result.Result;
import com.alibaba.dts.common.domain.store.TaskSnapshot;
import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;

/**
 * 拉任务快照线程
 * @author tianyao.myc
 *
 */
public class PullProcessor extends Thread implements Constants {

	private static final Logger logger = SchedulerXLoggerFactory.getLogger(PullProcessor.class);

	private int retryTimes = 0;

	private static final int MAX_RETRY_TIMES = 5;

	private static final long PULL_LOCK_SLEEP_TIME_INTERVAL= 150;

	private static final long PULL_EMPTY_SLEEP_TIME_INTERVAL= 2*1000;

	private static final long PULL_FAIL_SLEEP_TIME_INTERVAL= 2*1000;

	private static final long PULL_SUCCESS_SLEEP_TIME_INTERVAL= 150;

	private static final long PULL_SLEEP_TIME_INTERVAL= 1*10;
	
	/** 执行单元 */
	private ExecutorUnit executorUnit;
	
	/** 是否停止拉取线程 */
	private volatile boolean stop = false;

	private final ClientContextImpl clientContext;
	
	public PullProcessor(final ClientContextImpl clientContext,ExecutorUnit executorUnit) {
		this.clientContext = clientContext;
		this.executorUnit = executorUnit;
		super.setName(LT_PULL_TASK_THREAD_NAME + this.executorUnit.getExecutableTask().getJob().getId()
				+ HORIZONTAL_LINE +  executorUnit.getExecutableTask().getJob().getJobProcessor() 
				+ HORIZONTAL_LINE +  executorUnit.getExecutableTask().getJobInstanceSnapshot().getId() 
				+ HORIZONTAL_LINE +  executorUnit.getExecutableTask().getJobInstanceSnapshot().getFireTime() 
				+ HORIZONTAL_LINE +  executorUnit.getExecutableTask().getJobInstanceSnapshot().getRetryCount());
	}
	
	/**
	 * 刷新线程信息
	 *  executorUnit
	 */
	public void refresh(ExecutorUnit executorUnit) {
		this.executorUnit = executorUnit;
		super.setName(LT_PULL_TASK_THREAD_NAME + this.executorUnit.getExecutableTask().getJob().getId()
				+ HORIZONTAL_LINE +  executorUnit.getExecutableTask().getJob().getJobProcessor() 
				+ HORIZONTAL_LINE +  executorUnit.getExecutableTask().getJobInstanceSnapshot().getId() 
				+ HORIZONTAL_LINE +  executorUnit.getExecutableTask().getJobInstanceSnapshot().getFireTime() 
				+ HORIZONTAL_LINE +  executorUnit.getExecutableTask().getJobInstanceSnapshot().getRetryCount());
	}
	
	@Override
	public void run() {

		try {
			retryTimes=0;
			executorUnit.setPullTaskFlag(true);
			BlockingQueue<TaskSnapshot> queue = this.executorUnit.getQueue();
			while(! stop) {
				if (!this.executorUnit.isReleaseTaskFlag()){

					/** 拉取任务列表并放入队列 */
					try {
						pullAndPut(queue);
					} catch (Throwable e) {
						logger.error("[LPullProcessor]: pullAndPut error"
								+ ", instanceId:" + this.executorUnit.getExecutableTask().getJobInstanceSnapshot().getId(), e);
					}

				}else{

					try {
						Thread.sleep(PULL_SLEEP_TIME_INTERVAL);
					} catch (Throwable e) {
						logger.error("[LPullProcessor]: ReleaseTaskLock not release, executorUnit:" + executorUnit, e);
					}
				}

			}
		} catch (Throwable e) {
			logger.error("[LPullProcessor]: run error"
					+ ", instanceId:" + this.executorUnit.getExecutableTask().getJobInstanceSnapshot().getId(), e);
		} finally {

			executorUnit.setPullTaskFlag(false);

		}
	}
	
	/**
	 * 拉取任务列表并放入队列
	 *  queue
	 */
	private void pullAndPut(BlockingQueue<TaskSnapshot> queue) {
		Result<ExecutableTask> pullResult = null;

		try {
			pullResult = this.clientContext.getExecutor().pullLongTimeTask(this.executorUnit.getExecutableTask());
		} catch (Throwable e) {
			logger.error("[LPullProcessor]: pull error"
					+ ", instanceId:" + this.executorUnit.getExecutableTask().getJobInstanceSnapshot().getId(), e);
		}
		if(null == pullResult) {
			logger.error("[LPullProcessor]: pullResult is null"
					+ ", instanceId:" + this.executorUnit.getExecutableTask().getJobInstanceSnapshot().getId());
			
			try {
				Thread.sleep(PULL_EMPTY_SLEEP_TIME_INTERVAL);
			} catch (Throwable e) {
				logger.error("[LPullProcessor]: pullResult sleep error, executorUnit:" + executorUnit, e);
			}
			
			return ;
		}
		
		ExecutableTask executableTaskResult = pullResult.getData();
		if(null == executableTaskResult) {
			switch(pullResult.getResultCode()) {
			case PULL_TASK_LIST_OVER:
                retryTimes++;
				try {
					//Thread.sleep(clientConfig.getPullTaskListOverSleepTime());//拉到空数据就停歇一段时间 不然会给服务端造成很大压力
					Thread.sleep(PULL_FAIL_SLEEP_TIME_INTERVAL);
				} catch (Throwable e) {
					logger.error("[LPullProcessor]: PULL_TASK_LIST_OVER sleep error"
							+ ", instanceId:" + this.executorUnit.getExecutableTask().getJobInstanceSnapshot().getId(), e);
				}

				if (retryTimes > MAX_RETRY_TIMES)	this.setStop(true);
				break;
			case PULL_TASK_GET_LOCK_FAILURE:
				try {
					Thread.sleep(PULL_LOCK_SLEEP_TIME_INTERVAL);
				} catch (Throwable e) {
					logger.error("[LPullProcessor]: PULL_TASK_GET_LOCK_FAILURE sleep error"
							+ ", instanceId:" + this.executorUnit.getExecutableTask().getJobInstanceSnapshot().getId(), e);
				}
				break;
			case PULL_OVER:
				retryTimes++;

				try {
					Thread.sleep(PULL_FAIL_SLEEP_TIME_INTERVAL);
				} catch (Throwable e) {
					logger.error("[LPullProcessor]: PULL_OVER sleep before error, executorUnit:" + executorUnit, e);
				}

				try {
					if (retryTimes > 0)	this.setStop(true);

					//实例已经结束，中止运行
					LongTimePool longTimePool = executorUnit.getLongTimePool();
					longTimePool.stopTask(this.executorUnit.getExecutableTask().getJob().getId(),
							this.executorUnit.getExecutableTask().getJobInstanceSnapshot().getId());

					this.executorUnit.stopTask();//补偿停止

				} catch (Throwable e) {
					logger.error("[LPullProcessor]: PULL_OVER error"
							+ ", instanceId:" + this.executorUnit.getExecutableTask().getJobInstanceSnapshot().getId(), e);
				} finally {
					if(this.clientContext.getClientConfig().isFinishLog()) {
						logger.info("[LPullProcessor]: PULL_OVER EXIT"
								+ ", instanceId:" + this.executorUnit.getExecutableTask().getJobInstanceSnapshot().getId());
					}
				}
				break;
			default:
				logger.error("[LPullProcessor]: executableTask is null"
						+ ", instanceId:" + this.executorUnit.getExecutableTask().getJobInstanceSnapshot().getId()
						+ ", pullResult:" + pullResult.toString());
			}
			return ;
		}
		
		List<TaskSnapshot> taskSnapshotList = executableTaskResult.getTaskSnapshotList();
		if(CollectionUtils.isEmpty(taskSnapshotList)) {
			logger.warn("[LPullProcessor]: taskSnapshotList is empty error"
					+ ", instanceId:" + this.executorUnit.getExecutableTask().getJobInstanceSnapshot().getId());
			return ;
		}
		
		for(TaskSnapshot taskSnapshot : taskSnapshotList) {
			try {

				if (executorUnit.isExistsInTaskRunStatisticMap(taskSnapshot.getId())){
					executorUnit.updateTaskRunStatisticMap(taskSnapshot.getId(),0L,TASK_STATUS_QUEUE);
					logger.info("[LPullProcessor] pull duplicate task, instanceid:"+taskSnapshot.getJobInstanceId()+",taskid(db):"+taskSnapshot.getId());
				}else{
					queue.put(taskSnapshot);
					logger.info("[LPullProcessor] pull task, instanceid:"+taskSnapshot.getJobInstanceId()+",taskid(db):"+taskSnapshot.getId());
				}

			} catch (Throwable e) {
				logger.error("[LPullProcessor]: put error"
						+ ", instanceId:" + taskSnapshot.getJobInstanceId() 
						+ ", id:" + taskSnapshot.getId(), e);
			}
		}
		
		try {
			Thread.sleep(PULL_SUCCESS_SLEEP_TIME_INTERVAL);//成功拿到数据就停歇一段时间再抢 多给其他客户端一些机会 而且也不然会给服务端造成很大压力
		} catch (Throwable e) {
			logger.error("[LPullProcessor]: PULL_TASK_SUCCESS sleep error"
					+ ", instanceId:" + this.executorUnit.getExecutableTask().getJobInstanceSnapshot().getId(), e);
		}
	}
	
	public boolean isStop() {
		return stop;
	}

	public void setStop(boolean stop) {
		this.stop = stop;
	}

}
