package com.alibaba.dts.client.executor.simple.processor;

import com.alibaba.dts.client.executor.job.context.ClientContextImpl;
import com.alibaba.dts.client.service.JobRunningStateManager;
import com.alibaba.dts.common.constants.Constants;
import com.alibaba.dts.common.domain.ExecutableTask;
import com.alibaba.dts.common.domain.result.ProcessResult;
import com.alibaba.dts.common.domain.store.TaskSnapshot;
import com.alibaba.dts.common.logger.DtsLogger;
import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;
import com.alibaba.edas.schedulerx.ScxSimpleJobContext;
import com.alibaba.edas.schedulerx.ScxSimpleJobProcessor;

/**
 * 简单任务处理器
 * @author tianyao.myc
 *
 */
public class ScxSimpleTaskProcessor extends Thread implements Constants {

	private static final Logger logger = SchedulerXLoggerFactory.getLogger(ScxSimpleTaskProcessor.class);

	private static final Logger executeLogger = SchedulerXLoggerFactory.getExecuteLogger();
	
	/** 可执行任务 */
	private ExecutableTask executableTask;
	
	/** 状态 */
	private int status = TASK_PROCESSOR_STATUS_STOP;
	
	/** 简单Job执行上下文 */
	private ScxSimpleJobContext context;
	
	private final ClientContextImpl clientContext;

	public ScxSimpleTaskProcessor(final ClientContextImpl clientContext, ExecutableTask executableTask) {
		this.clientContext = clientContext;
		this.executableTask = executableTask;
		super.setName(SCX_TASK_THREAD_NAME + executableTask.getJob().getId()
				+ HORIZONTAL_LINE + executableTask.getJob().getJobProcessor() 
				+ HORIZONTAL_LINE + executableTask.getJobInstanceSnapshot().getId() 
				+ HORIZONTAL_LINE + executableTask.getJobInstanceSnapshot().getFireTime() 
				+ HORIZONTAL_LINE + executableTask.getJobInstanceSnapshot().getRetryCount());
		this.context = new ScxSimpleJobContext(executableTask.getJob(), executableTask.getJobInstanceSnapshot(),
				executableTask.getJobInstanceSnapshot().getRetryCount());
		
		this.context.setAvailableMachineAmount(executableTask.getAvailableMachineAmount());
		this.context.setCurrentMachineNumber(executableTask.getCurrentMachineNumber());
	}
	
	@Override
	public void run() {
		
		/** 任务开始 计数器加一 */
		this.status = TASK_PROCESSOR_STATUS_RUNNING;
		
		if(Constants.ENVIRONMENT_JST.equals(this.clientContext.getClientConfig().getEnvironment())) {
			DtsLogger.info(executableTask.getJob().getId(), 
					executableTask.getJobInstanceSnapshot().getId(), "task init start ...");
		}
		
		try {
			
			/** 处理器准备 */
			ScxSimpleJobProcessor scxSimpleJobProcessor = null;
			try {
				scxSimpleJobProcessor = this.clientContext.getJobProcessorFactory().createAndGetScxSimpleJobProcessor(executableTask.getJob(), false);
			} catch (Throwable e) {
				
				logger.error("[ScxSimpleTaskProcessor]: createAndGetSimpleJobProcessor error"
						+ ", jobProcessor:" + executableTask.getJob().getJobProcessor(), e);

				executeLogger.error("jobId:"
						+ executableTask.getJob().getId()
						+",jobInstanceId:"
						+executableTask.getJobInstanceSnapshot().getId()
						+",createAndGetSimpleJobProcessor error"
						+ ", jobProcessor:" + executableTask.getJob().getJobProcessor(), e);
				
				if(Constants.ENVIRONMENT_JST.equals(this.clientContext.getClientConfig().getEnvironment())) {
					DtsLogger.info(executableTask.getJob().getId(), 
							executableTask.getJobInstanceSnapshot().getId(), "createAndGetSimpleJobProcessor error"
							+ ", jobProcessor:" + executableTask.getJob().getJobProcessor(), e);
				}
			}
			
			if(Constants.ENVIRONMENT_JST.equals(this.clientContext.getClientConfig().getEnvironment())) {
				DtsLogger.info(executableTask.getJob().getId(), 
						executableTask.getJobInstanceSnapshot().getId(), "task execute start ...");
			}

			executeLogger.info("jobId:"
							+ executableTask.getJob().getId()
							+ ",jobInstanceId:"
							+ executableTask.getJobInstanceSnapshot().getId()
							+ ",task execute start......"
			);

			/** 执行任务 */
			executeTask(this.executableTask, scxSimpleJobProcessor);

			executeLogger.info("jobId:"
							+ executableTask.getJob().getId()
							+ ",jobInstanceId:"
							+ executableTask.getJobInstanceSnapshot().getId()
							+ ",task execute end......"
			);
		} catch (Throwable e) {
			logger.error("[ScxSimpleTaskProcessor]: executeTask error"
					+ ", instanceId:" + this.executableTask.getJobInstanceSnapshot().getId(), e);
		} finally {
			this.status = TASK_PROCESSOR_STATUS_STOP;
			
			/** 删除处理器 */
			this.clientContext.getExecutor().getScxSimplePool().removeTask(this.executableTask);
		}
		
	}

	/**
	 * 执行任务
	 *  executableTask
	 *  scxSimpleJobProcessor
	 */
	private void executeTask(ExecutableTask executableTask, ScxSimpleJobProcessor scxSimpleJobProcessor) {

		TaskSnapshot taskSnapshot = executableTask.getTaskSnapshot();
		
		if(null == scxSimpleJobProcessor) {
			
			logger.error("[ScxSimpleTaskProcessor]: jobProcessor is null"
					+ ", please check " + executableTask.getJob().getJobProcessor());

			executeLogger.error("jobId:"
							+ executableTask.getJob().getId()
							+",jobInstanceId:"
							+executableTask.getJobInstanceSnapshot().getId()
							+",jobProcessor is null,"
							+ executableTask.getJob().getJobProcessor()
			);

			if(Constants.ENVIRONMENT_JST.equals(this.clientContext.getClientConfig().getEnvironment())) {
				DtsLogger.info(executableTask.getJob().getId(), 
						executableTask.getJobInstanceSnapshot().getId(), "jobProcessor is null"
								+ ", please check " + executableTask.getJob().getJobProcessor());
			}
			
			//失败确认
			this.clientContext.getExecutor().acknowledge(taskSnapshot, TASK_STATUS_FAILURE, 0);
			return ;
		}
		
		/** 设置任务 */
		this.context.setTask(taskSnapshot);
		
		ProcessResult processResult = null;
		boolean happenException = false;
		try {
			processResult = scxSimpleJobProcessor.process(this.context);
		} catch (Throwable e) {

			happenException = true;
			logger.error("[ScxSimpleTaskProcessor]: process error"
					+ ", instanceId:" + executableTask.getJobInstanceSnapshot().getId(), e);

			executeLogger.error("jobId:"
							+ executableTask.getJob().getId()
							+",jobInstanceId:"
							+executableTask.getJobInstanceSnapshot().getId()
							+",happen exception:",e
			);

			JobRunningStateManager.getManageHandler().addJobRunningException(executableTask.getJobInstanceSnapshot().getId(),e);
		}
		if(null == processResult) {
			logger.error("[ScxSimpleTaskProcessor]: process error, processResult is null"
					+ ", instanceId:" + executableTask.getJobInstanceSnapshot().getId());
			processResult = new ProcessResult(false);

			executeLogger.error("jobId:"
							+ executableTask.getJob().getId()
							+",jobInstanceId:"
							+executableTask.getJobInstanceSnapshot().getId()
							+"process error, processResult is null!"
			);

			if (!happenException)
			JobRunningStateManager.getManageHandler().addJobRunningException(executableTask.getJobInstanceSnapshot().getId(),new Throwable("processResult is null!"));
		}


		/** ACK确认 执行结果 */
		handleRetryCount(taskSnapshot, processResult);
		this.clientContext.getExecutor().acknowledge(taskSnapshot, processResult.isSuccess() ? TASK_STATUS_SUCCESS : TASK_STATUS_FAILURE, processResult.getRetryCount());


		executeLogger.info("jobId:"
						+ executableTask.getJob().getId()
						+ ",jobInstanceId:"
						+ executableTask.getJobInstanceSnapshot().getId()
						+ ",task processResult:"
						+ processResult.toString()
		);

		if ((processResult!=null) && (!processResult.isSuccess()) && (!happenException)){
			JobRunningStateManager.getManageHandler().addJobRunningException(executableTask.getJobInstanceSnapshot().getId(),new Throwable("processor return failure!"));
		}


		if(Constants.ENVIRONMENT_JST.equals(this.clientContext.getClientConfig().getEnvironment())) {
			DtsLogger.info(executableTask.getJob().getId(), 
					executableTask.getJobInstanceSnapshot().getId(), "task execute end, processResult:" + processResult);
		}
	}

	/**
	 * 处理重试次数
	 *  taskSnapshot
	 *  processResult
	 */
	private void handleRetryCount(TaskSnapshot taskSnapshot, ProcessResult processResult) {
		
		/** 如果是执行成功就不设置重试 */
		if(processResult.isSuccess()) {
			processResult.setRetryCount(0);
			return ;
		}
		
		/** 如果是补偿任务就重试次数减一 */
		if(this.executableTask.isCompensation()) {
			if(taskSnapshot.getRetryCount() > 0) {
				processResult.setRetryCount(taskSnapshot.getRetryCount() - 1);
			} else {
				processResult.setRetryCount(0);
			}
			return ;
		}
		
		/** 不能超过最大重试次数 */
		if(processResult.getRetryCount() > MAX_RETRY_COUNT) {
			processResult.setRetryCount(MAX_RETRY_COUNT);
			return ;
		}
	}
	
	public int getStatus() {
		return status;
	}

	public void setStop(boolean stop){
		this.context.setStop(stop);
	}

}
