package com.alibaba.tmq.client.remoting.timer;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.alibaba.tmq.client.TMQFactory;
import com.alibaba.tmq.client.context.ClientContext;
import com.alibaba.tmq.client.system.consumer.executer.ConsumerExecuter;
import com.alibaba.tmq.client.system.consumer.listener.MessageListener;
import com.alibaba.tmq.client.system.producer.executer.NormalProducerExecuter;
import com.alibaba.tmq.client.system.producer.executer.TransactionProducerExecuter;
import com.alibaba.tmq.client.util.StringUtil;
import com.alibaba.tmq.common.constants.Constants;
import com.alibaba.tmq.common.domain.ConsumerKey;
import com.alibaba.tmq.common.util.ListUtil;

/**
 * 心跳定时器
 * @author tianyao.myc
 *
 */
public class ClientHeartBeatTimer extends TimerTask implements Constants {

	private static final Log logger = LogFactory.getLog(ClientHeartBeatTimer.class);
	
	@Override
	public void run() {
		try {
			List<String> serverList = ClientContext.zookeeper.getServerList();

			if(ListUtil.isEmpty(serverList)) {
				logger.warn("[ClientHeartBeatTimer]: serverList is empty, clientConfig:" + ClientContext.clientConfig);
			} else {

				/** 更新服务端地址列表缓存 */
				ClientContext.clientRemoting.setCachedServerList(serverList);
				
				checkHeartBeat(serverList);
			}
			
			if(StringUtil.isNotBlank(ClientContext.clientConfig.getBackupDomainName())) {
				
				List<String> backupServerList = ClientContext.zookeeper.getbackupServerList();
				
				if(ListUtil.isEmpty(backupServerList)) {
					logger.warn("[ClientHeartBeatTimer]: backupServerList is empty, clientConfig:" + ClientContext.clientConfig);
				} else {

					/** 更新服务端地址列表缓存 */
					ClientContext.clientRemoting.setCachedBackupServerList(backupServerList);
					
					checkHeartBeat(backupServerList);
				}
				
			}
			
		} catch (Throwable e) {
			logger.error("[ClientHeartBeatTimer]: run error, clientConfig:" + ClientContext.clientConfig, e);
		}
	}
	
	/**
	 * 检查心跳
	 *  serverList
	 */
	private void checkHeartBeat(List<String> serverList) {
		
		for(String server : serverList) {
			try {
				//检查生产者心跳
				checkProducerHeartBeat(server);
				
				//检查消费者心跳
				checkConsumerHeartBeat(server);
			} catch (Throwable e) {
				logger.error("[ClientHeartBeatTimer]: checkHeartBeat error"
						+ ", clientConfig:" + ClientContext.clientConfig 
						+ ", server:" + server, e);
			}
		}
		
	}
	
	/**
	 * 检查生产者心跳
	 *  server
	 */
	private void checkProducerHeartBeat(String server) {
		
		//拿到普通Producer映射表
		ConcurrentHashMap<String, ConcurrentHashMap<String, NormalProducerExecuter>> normalProducerTable = TMQFactory.getNormalproducertable();
		
		//检查每一个普通生产者心跳
		checkNormalProducerHeartBeat(server, normalProducerTable, "normalProducerTable");
		
		//拿到事务Producer映射表
		ConcurrentHashMap<String, ConcurrentHashMap<String, TransactionProducerExecuter>> transactionProducerTable = TMQFactory.getTransactionproducertable();
		
		//检查每一个事务生产者心跳
		checkTransactionProducerHeartBeat(server, transactionProducerTable, "transactionProducerTable");
		
	}
	
	/**
	 * 检查每一个普通生产者心跳
	 *  server
	 *  producerTable
	 *  source
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	private void checkNormalProducerHeartBeat(String server, 
			ConcurrentHashMap<String, ConcurrentHashMap<String, NormalProducerExecuter>> producerTable, String source) {
		
		Iterator iterator = producerTable.entrySet().iterator();
		while(iterator.hasNext()) {
			Map.Entry entry = (Map.Entry)iterator.next();
			String producerId = (String)entry.getKey();
			
			ConcurrentHashMap<String, NormalProducerExecuter> producerMap = (ConcurrentHashMap<String, NormalProducerExecuter>)entry.getValue();
			
			Iterator producerIterator = producerMap.entrySet().iterator();
			while(producerIterator.hasNext()) {
				Map.Entry producerEntry = (Map.Entry)producerIterator.next();
				String instanceName = (String)producerEntry.getKey();
			
				try {
					ClientContext.clientRemoting.connectServer(instanceName, server, ROLE_TYPE_PRODUCER, producerId, NULL, NULL);
				} catch (Throwable e) {
					logger.error("[ClientHeartBeatTimer]: checkNormalProducerHeartBeat connectServer error"
							+ ", server:" + server 
							+ ", clientConfig:" + ClientContext.clientConfig 
							+ ", producerId:" + producerId 
							+ ", instanceName:" + instanceName, e);
				}
			}
			
		}
		
	}
	
	/**
	 * 检查每一个事务生产者心跳
	 *  server
	 *  producerTable
	 *  source
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	private void checkTransactionProducerHeartBeat(String server, 
			ConcurrentHashMap<String, ConcurrentHashMap<String, TransactionProducerExecuter>> producerTable, String source) {
		
		Iterator iterator = producerTable.entrySet().iterator();
		while(iterator.hasNext()) {
			Map.Entry entry = (Map.Entry)iterator.next();
			String producerId = (String)entry.getKey();
			
			ConcurrentHashMap<String, TransactionProducerExecuter> producerMap = (ConcurrentHashMap<String, TransactionProducerExecuter>)entry.getValue();
			
			Iterator producerIterator = producerMap.entrySet().iterator();
			while(producerIterator.hasNext()) {
				Map.Entry producerEntry = (Map.Entry)producerIterator.next();
				String instanceName = (String)producerEntry.getKey();
				
				try {
					ClientContext.clientRemoting.connectServer(instanceName, server, ROLE_TYPE_PRODUCER, producerId, NULL, NULL);
				} catch (Throwable e) {
					logger.error("[ClientHeartBeatTimer]: checkTransactionProducerHeartBeat connectServer error"
							+ ", server:" + server 
							+ ", clientConfig:" + ClientContext.clientConfig 
							+ ", producerId:" + producerId 
							+ ", instanceName:" + instanceName, e);
				}
			}
			
		}
		
	}
	
	/**
	 * 检查消费者心跳
	 *  server
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	private void checkConsumerHeartBeat(String server) {
		
		ConcurrentHashMap<String, ConcurrentHashMap<String, ConsumerExecuter>> consumerTable = TMQFactory.getConsumertable();
		
		Iterator iterator = consumerTable.entrySet().iterator();
		while(iterator.hasNext()) {
			Map.Entry entry = (Map.Entry)iterator.next();
			String consumerId = (String)entry.getKey();
			ConcurrentHashMap<String, ConsumerExecuter> consumerMap = (ConcurrentHashMap<String, ConsumerExecuter>)entry.getValue();
			
			Iterator consumerIterator = consumerMap.entrySet().iterator();
			while(consumerIterator.hasNext()) {
				Map.Entry consumerEntry = (Map.Entry)consumerIterator.next();
				String instanceName = (String)consumerEntry.getKey();
				ConsumerExecuter consumerExecuter = (ConsumerExecuter)consumerEntry.getValue();
				
				ConcurrentHashMap<ConsumerKey, MessageListener> listenerTable = consumerExecuter.getListenerTable();
				
				Iterator listenerIterator = listenerTable.entrySet().iterator();
				while(listenerIterator.hasNext()) {
					Map.Entry listenerEntry = (Map.Entry)listenerIterator.next();
					ConsumerKey consumerKey = (ConsumerKey)listenerEntry.getKey();
				
					try {
						ClientContext.clientRemoting.connectServer(instanceName, server, ROLE_TYPE_CONSUMER, consumerId, consumerKey.getTopic(), consumerKey.getTag());
					} catch (Throwable e) {
						logger.error("[ClientHeartBeatTimer]: connectServer error"
								+ ", server:" + server 
								+ ", clientConfig:" + ClientContext.clientConfig 
								+ ", consumerId:" + consumerId 
								+ ", instanceName:" + instanceName 
								+ ", consumerKey:" + consumerKey, e);
					}
					
				}
			}
			
		}
		
	}
	
}
