package com.alibaba.tmq.common.proxy;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.HashMap;
import java.util.Map;

import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;
import com.alibaba.tmq.common.constants.Constants;
import com.alibaba.tmq.common.domain.remoting.ClassKey;
import com.alibaba.tmq.common.domain.result.ResultCode;

/**
 * 代理服务
 * @author tianyao.myc
 *
 */
public class ProxyService implements Constants {

	private static final Logger logger = SchedulerXLoggerFactory.getLogger(ProxyService.class);

	/** 基本类型 */
	public static final Map<String, Class<?>> BASE_CLASS = new HashMap<String, Class<?>>();
	static {
		BASE_CLASS.put("double", 	double.class);
		BASE_CLASS.put("long", 		long.class);
		BASE_CLASS.put("float", 	float.class);
		BASE_CLASS.put("int", 		int.class);
		BASE_CLASS.put("short", 	short.class);
		BASE_CLASS.put("char", 		char.class);
		BASE_CLASS.put("byte", 		byte.class);
		BASE_CLASS.put("boolean", 	boolean.class);
		BASE_CLASS.put("void", 		void.class);
	}

	/** 方法缓存 */
	private final Map<ClassKey, Method> methodCache = new HashMap<ClassKey, Method>();

	/**
	 * 获取类型
	 *  parameterTypeString
	 *
	 *  Throwable
	 */
	public static Class<?> getClass(String parameterTypeString) {
		Class<?> parameterTypeClass = ProxyService.BASE_CLASS.get(parameterTypeString);
		if(parameterTypeClass != null) {
			return parameterTypeClass;
		} else {
			try {
				return Class.forName(parameterTypeString);
			} catch (Throwable e) {
				logger.error("[ProxyService]: getClass error, parameterTypeString:" + parameterTypeString, e);
				return null;
			}
		}
	}

	/**
	 * 代理接口
	 *  interfaceClass
	 *  invocationHandler
	 *
	 */
	@SuppressWarnings("unchecked")
	public <T> T proxyInterface(Class<T> interfaceClass, InvocationHandler invocationHandler) {
		return (T)Proxy.newProxyInstance(interfaceClass.getClassLoader(), new Class[]{interfaceClass}, invocationHandler);
	}

	/**
	 * new对象
	 *  classType
	 *
	 */
	public <T> T newInstance(Class<T> classType) {
		try {
			return (T)classType.newInstance();
		} catch (Throwable e) {
			logger.error("[ProxyService]: newInstance error, classType:" + classType.getName(), e);
			return null;
		}
	}

	/**
	 * new对象
	 *  classType
	 *
	 */
	@SuppressWarnings("unchecked")
	public <T> T newInstance(String classType) {
		Class<?> type = null;
		try {
			type = Class.forName(classType);
		} catch (Throwable e) {
			logger.error("[ProxyService]: newInstance Class.forName error, classType:" + classType, e);
			return null;
		}
		try {
			return (T)type.newInstance();
		} catch (Throwable e) {
			logger.error("[ProxyService]: newInstance error, classType:" + classType, e);
			return null;
		}
	}

	/**
	 * 调用方法
	 *  object
	 *  methodName
	 *  parameterTypes
	 *  arguments
	 *
	 */
	public Object invokeMethod(Object object, String methodName, Class<?>[] parameterTypes, Object[] arguments) {
		Method method = getMethod(object, methodName, parameterTypes);
		if(null == method) {
			return ResultCode.NO_SUCH_METHOD;
		}
		Object result = null;
		try {
			method.setAccessible(true);
			result = method.invoke(object, arguments);
		} catch (Throwable e) {
			String info = "[invokeMethod]: error, methodName:" + methodName;
			logger.error(info, e);
			throw new RuntimeException(info, e);
		}
		return result;
	}

	/**
	 * 获取方法
	 *  object
	 *  methodName
	 *  parameterTypes
	 *
	 */
	private Method getMethod(Object object, String methodName, Class<?>[] parameterTypes) {
		ClassKey classKey = new ClassKey(object, methodName, parameterTypes);
		Method method = methodCache.get(classKey);
		if(method != null) {
			return method;
		}
		method = tryFindMethod(object, methodName, parameterTypes);
		if(method != null) {
			/** 缓存方法 */
			methodCache.put(classKey, method);
		}
		return method;
	}

	/**
	 * 尝试查找方法
	 *  object
	 *  methodName
	 *  parameterTypes
	 *
	 */
	private Method tryFindMethod(Object object, String methodName, Class<?>[] parameterTypes) {
		Method method = null;
		try {
			method = object.getClass().getDeclaredMethod(methodName, parameterTypes);
		} catch (Throwable e) {
			String info = "[tryFindMethod]: failed, methodName:" + methodName;
			logger.error(info, e);
			throw new RuntimeException(info, e);
		}
		return method;
	}
	
}
