package com.alibaba.tmq.common.util;

import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

import com.alibaba.dts.common.logger.SchedulerXLoggerFactory;
import com.alibaba.dts.common.logger.innerlog.Logger;
import com.alibaba.tmq.client.util.StringUtil;
import com.alibaba.tmq.common.constants.Constants;
import com.alibaba.tmq.common.domain.KeyValuePair;
import com.alibaba.tmq.common.domain.Message;
import com.alibaba.tmq.common.domain.TracePoint;
import com.alibaba.tmq.common.logger.LoggerEvent;
import com.taobao.eagleeye.EagleEye;
import com.taobao.eagleeye.TraceLogger;

/**
 * 日志相关工具
 * @author tianyao.myc
 *
 */
public class LoggerUtil implements Constants {

	// 创建 TraceLogger
    private static final TraceLogger traceLogger = EagleEye.traceLogger(TRACE_LOGGER_NAME);

	private static final Logger logger = SchedulerXLoggerFactory.getLogger(LoggerUtil.class);
	
	private static final LinkedBlockingQueue<Runnable> eventQueue = new LinkedBlockingQueue<Runnable>();
    
	private static final ThreadPoolExecutor executors = new ThreadPoolExecutor(16, 16, 10 * 1000L, TimeUnit.MILLISECONDS, eventQueue, new ThreadFactory(){

		int index = 0;
		
		public Thread newThread(Runnable runnable) {
			index ++;
			return new Thread(runnable, "TMQ-LoggerEvent-thread-" + index);
		}
		
	});
	
    private static LoggerEvent loggerEvent;
    
    public static void setLoggerEvent(LoggerEvent loggerEvent) {
		LoggerUtil.loggerEvent = loggerEvent;
	}

	/**
     * 打印单条消息信息
     *  message
     *  action
     *  startTime
     *  localAddress
     */
    public static void info(final Message message, final String action, final long startTime, final String localAddress) {
    	
    	final long cost = System.currentTimeMillis() - startTime;
    	
    	//日志内容
    	final String logContent = TimeUtil.date2MilliSeconds(new Date()) 
    			+ ", action:" + action 
    			+ ", time:" + cost 
    			+ ", localAddress:" + localAddress 
    			+ ", message:" + message;
    	
    	//远程消息轨迹投送
    	if(loggerEvent != null) {
    		executors.execute(new Runnable() {

				@Override
				public void run() {
					
					try {
						
						List<TracePoint> tracePointList = new ArrayList<TracePoint>();
						tracePointList.add(tracePoint(message, action, startTime, cost, localAddress));
						
						loggerEvent.onEvent(tracePointList);
					} catch (Throwable e) {
						logger.error("[LoggerUtil]: onEvent error, logContent:" + logContent, e);
					}
					
				}
    			
    		});
    	}
    	
    	//打印日志
    	traceLogger.logLine(LOGGER_KEY, StringUtil.isBlank(message.getMessageId()) ? message.getMessageKey() : message.getMessageId(), logContent);
    }
    
    /**
     * 打印多条消息信息
     *  messageList
     *  action
     *  startTime
     *  localAddress
     */
    public static void info(final List<? extends Message> messageList, final String action, final long startTime, final String localAddress) {
    	
    	final long cost = System.currentTimeMillis() - startTime;
    	
    	//日志内容
    	final String logContent = TimeUtil.date2MilliSeconds(new Date()) 
    			+ ", action:" + action 
    			+ ", time:" + cost 
    			+ ", localAddress:" + localAddress 
    			+ ", messageList:" + messageList;
    	
    	//远程消息轨迹投送
    	if(loggerEvent != null) {
    		executors.execute(new Runnable() {

				@Override
				public void run() {
					
					try {
						
						List<TracePoint> tracePointList = new ArrayList<TracePoint>();
						
						for(Message message : messageList) {
							tracePointList.add(tracePoint(message, action, startTime, cost, localAddress));
						}
						
						loggerEvent.onEvent(tracePointList);
					} catch (Throwable e) {
						logger.error("[LoggerUtil]: onEvent error, logContent:" + logContent, e);
					}
					
				}
    			
    		});
    	}
    	
    	//打印日志
    	traceLogger.logLine(LOGGER_KEY, messageList.get(0).getMessageKey(), logContent);
    }

    public static TracePoint tracePoint(Message message, String action, long startTime, long cost, String localAddress) {
    	
    	TracePoint tracePoint = new TracePoint();
    	tracePoint.setGmtCreate(new Date());
    	tracePoint.setGmtModified(new Date());
		tracePoint.setMessageId(message.getMessageId());
		tracePoint.setMessageKey(message.getMessageKey());
		tracePoint.setAction(action);
		tracePoint.setTimestamp(startTime);
		tracePoint.setCost(cost);
		tracePoint.setLocalAddress(localAddress);
		tracePoint.setMessage(message.toString());
		tracePoint.setClusterId(message.getClusterId());
    	
		return tracePoint;
    }
    
    /**
     * 展现表格
     *  relationTable
     *
     */
    @SuppressWarnings({ "rawtypes", "unchecked" })
	public static String displayTable(ConcurrentHashMap<String, String> relationTable, 
			ConcurrentSkipListMap<String, ConcurrentSkipListMap<String, KeyValuePair<AtomicLong, AtomicLong>>> monitorTable) {
    	
    	String key = "Role";
    	List<String> header = new ArrayList<String>();
    	header.add("Topic - UniqueTable");
    	
    	Map<String, List<String>> body = new HashMap<String, List<String>>();
    	
    	Iterator iterator = monitorTable.entrySet().iterator();
		while (iterator.hasNext()) {
		    Map.Entry entry = (Map.Entry)iterator.next();
		    String roleId = (String)entry.getKey();
		    ConcurrentSkipListMap<String, KeyValuePair<AtomicLong, AtomicLong>> methodTable = (ConcurrentSkipListMap<String, KeyValuePair<AtomicLong, AtomicLong>>)entry.getValue();
		    
		    List<String> itemList = new ArrayList<String>();
		    
		    String tu = relationTable.get(roleId);
		    if(tu != null) {
		    	itemList.add(tu);
		    } else {
		    	itemList.add("");
		    }
		    
		    Iterator methodIterator = methodTable.entrySet().iterator();
			while (methodIterator.hasNext()) {
			    Map.Entry methodEntry = (Map.Entry)methodIterator.next();
			    String method = (String)methodEntry.getKey();
			    KeyValuePair<AtomicLong, AtomicLong> pair = (KeyValuePair<AtomicLong, AtomicLong>)methodEntry.getValue();
			    
			    if(! header.contains(method)) {
			    	header.add(method);
			    }
			    
			    AtomicLong counter = pair.getKey();
				AtomicLong totalTime = pair.getValue();
			    
			    int index = header.indexOf(method);
			    int size = itemList.size();
			    if(index >= itemList.size()) {
			    	for(int i = 0 ; i < index - size; i ++) {
			    		itemList.add("");
				    }
			    	itemList.add(counter.get() + " , " + (0L == counter.get() ? 0L : totalTime.get() / counter.get()));
			    } else {
			    	itemList.set(index, counter.get() + " , " + (0L == counter.get() ? 0L : totalTime.get() / counter.get()));
			    }
			    
			    counter.set(0L);//计数器清零
				totalTime.set(0L);//耗时清零
			}
			
			body.put(roleId, itemList);
		}
		
    	return displayTable(key, header, body);
    }
    
	/**
	 * 展现表格
	 *  header
	 *  body
	 *
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	public static String displayTable(String key, List<String> header, Map<String, List<String>> body) {
		StringBuilder table = new StringBuilder();
		
		List<AtomicInteger> maxLengthList = getMaxLengthList(key, header, body);//最大item长度列表
		
		//打印开始隔行
		List<String> head = new ArrayList<String>();
		for(int i = 0 ; i < header.size() ; i ++) {
			head.add("-");
		}
		table.append(displayLine(false, "-", head, maxLengthList));
		
		//打印表头行
		table.append(displayLine(true, key, header, maxLengthList));
		
		
		//打印表头隔行
		List<String> interlaced = new ArrayList<String>();
		for(int i = 0 ; i < header.size() ; i ++) {
			interlaced.add("-");
		}
		table.append(displayLine(false, "-", interlaced, maxLengthList));
		
		//打印body
		Iterator iterator = body.entrySet().iterator();
		while(iterator.hasNext()) {
			Map.Entry entry = (Map.Entry)iterator.next();
			String topic = (String)entry.getKey();
			List<String> itemList = (List<String>)entry.getValue();
			
			table.append(displayLine(true, topic, itemList, maxLengthList));//打印表内容行
		}
		
		//打印表尾隔行
		List<String> tail = new ArrayList<String>();
		for(int i = 0 ; i < header.size() ; i ++) {
			tail.add("-");
		}
		table.append(displayLine(false, "-", tail, maxLengthList));
		
		return table.toString();
	}

	/**
	 * 打印一行
	 *  isLine
	 *  key
	 *  itemList
	 *  maxLengthList
	 *
	 */
	private static String displayLine(boolean isLine, String key, List<String> itemList, List<AtomicInteger> maxLengthList) {
		StringBuilder line = new StringBuilder();

		line.append((isLine ? " | " : " + ") + key + displayBlank(isLine, maxLengthList.get(0), key));
		for(int i = 0 ; i < maxLengthList.size() - 1 ; i ++) {
			if(i >= itemList.size()) {
				line.append((isLine ? " | " : " + ") + "" + displayBlank(isLine, maxLengthList.get(i + 1), ""));
			} else {
				line.append((isLine ? " | " : " + ") + itemList.get(i) + displayBlank(isLine, maxLengthList.get(i + 1), itemList.get(i)));
			}
		}
		line.append((isLine ? " | " : " + ") + "\n");

		return line.toString();
	}
	
	/**
	 * 展现空格
	 *  isLine
	 *  length
	 *  item
	 *
	 */
	private static String displayBlank(boolean isLine, AtomicInteger length, String item) {
		StringBuilder blank = new StringBuilder(isLine ? " " : "-");
		for(int i = 0 ; i < length.get() - item.length() ; i ++) {
			blank.append(isLine ? " " : "-");
		}
		return blank.toString();
	}
	
	/**
	 * 获取最大item长度列表
	 *  key
	 *  header
	 *  body
	 *
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	private static List<AtomicInteger> getMaxLengthList(String key, List<String> header, Map<String, List<String>> body) {
		
		List<AtomicInteger> maxLengthList = new ArrayList<AtomicInteger>();//最大item长度列表
		for(int i = 0 ; i < 1 + header.size() ; i ++) {
			maxLengthList.add(new AtomicInteger(0));
		}
		
		refreshMaxLengthList(maxLengthList, key, header);//刷新header各个字段长度
		
		Iterator iterator = body.entrySet().iterator();
		while(iterator.hasNext()) {
			Map.Entry entry = (Map.Entry)iterator.next();
			String topic = (String)entry.getKey();
			List<String> itemList = (List<String>)entry.getValue();
			
			refreshMaxLengthList(maxLengthList, topic, itemList);//刷新body各个字段长度
		}
		
		return maxLengthList;
	}
	
	/**
	 * 刷新最大item长度列表
	 *  maxLengthList
	 *  key
	 *  itemList
	 */
	private static void refreshMaxLengthList(List<AtomicInteger> maxLengthList, String key, List<String> itemList) {
		for(int i = 0 ; i < maxLengthList.size() ; i ++) {
			AtomicInteger length = maxLengthList.get(i);
			
			if(i > itemList.size()) {
				continue ;
			}
			
			if(length.intValue() < (0 == i ? key.length() : itemList.get(i - 1).length())) {
				length.set(0 == i ? key.length() : itemList.get(i - 1).length());
			}
		}
	}
	
}
