/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.hbase.client;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.regex.Pattern;

import com.alibaba.lindorm.client.AdminService;
import com.alibaba.lindorm.client.SystemService;
import com.alibaba.lindorm.client.exception.LindormException;
import com.alibaba.lindorm.client.schema.LindormFamilyAttributes;
import com.alibaba.lindorm.client.schema.LindormTableDescriptor;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.ClusterStatus;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.NamespaceDescriptor;
import org.apache.hadoop.hbase.NamespaceNotFoundException;
import org.apache.hadoop.hbase.ProcedureInfo;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.TableNotFoundException;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.ipc.CoprocessorRpcChannel;
import org.apache.hadoop.hbase.protobuf.generated.AdminProtos;
import org.apache.hadoop.hbase.protobuf.generated.HBaseProtos;
import org.apache.hadoop.hbase.protobuf.generated.MasterProtos;
import org.apache.hadoop.hbase.quotas.QuotaFilter;
import org.apache.hadoop.hbase.quotas.QuotaRetriever;
import org.apache.hadoop.hbase.quotas.QuotaSettings;
import org.apache.hadoop.hbase.security.SecurityCapability;
import org.apache.hadoop.hbase.snapshot.HBaseSnapshotException;
import org.apache.hadoop.hbase.snapshot.SnapshotCreationException;
import org.apache.hadoop.hbase.snapshot.UnknownSnapshotException;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;

public class AliHBaseUEAdmin implements Admin {

  private AdminService adminService;
  private AliHBaseUEConnection connection;
  private int operationTimeout;
  private Configuration conf;
  private SystemService systemService;



  public AliHBaseUEAdmin(AliHBaseUEConnection connection) throws IOException {
    this.connection = connection;
    this.conf = connection.getConfiguration();
    this.operationTimeout = connection.getOperationTimeout();
    adminService = connection.getAdminService();
    systemService = connection.getSystemService();

  }

  @Override
  public int getOperationTimeout() {
    return operationTimeout;
  }

  @Override
  public void abort(String why, Throwable e) {

  }

  @Override
  public boolean isAborted() {
    return false;
  }

  @Override
  public void close() throws IOException {
  }

  @Override
  public Configuration getConfiguration() {
    return conf;
  }

  @Override
  public boolean tableExists(TableName tableName) throws IOException {
    return connection.getWideColumnService(tableName.getNamespaceAsString()).listTables()
        .contains(tableName.getQualifierAsString());
  }

  public boolean tableExists(byte[] tableName) throws IOException {
    return this.tableExists(TableName.valueOf(tableName));
  }

  public boolean tableExists(String tableName) throws IOException {
    return this.tableExists(TableName.valueOf(tableName));
  }

  @Override
  public Connection getConnection() {
    return connection;
  }

  @Override
  public HTableDescriptor[] listTables() throws IOException {
    TableName[] tableNames = listTableNames();
    HTableDescriptor[] tableDescriptors = new HTableDescriptor[tableNames.length];
    for (int i = 0; i < tableNames.length; i++) {
      tableDescriptors[i] = getTableDescriptor(tableNames[i]);
    }
    return tableDescriptors;
  }

  @Override
  public HTableDescriptor[] listTables(Pattern pattern) throws IOException {
    return listTables();
  }

  @Override
  public HTableDescriptor[] listTables(String regex) throws IOException {
    return listTables();
  }

  @Override
  public HTableDescriptor[] listTables(Pattern pattern, boolean includeSysTables)
      throws IOException {
    return listTables();
  }


  @Override
  public HTableDescriptor[] listTables(String regex, boolean includeSysTables) throws IOException {
    return listTables();
  }

  @Override
  public TableName[] listTableNames() throws IOException {
    List<String> tables = adminService.listTableNamesByPrefix("");
    TableName[] tableNames = new TableName[tables.size()];
    for (int i = 0; i < tables.size(); i++) {
      tableNames[i] = ElementConvertor.toHBaseTableName(tables.get(i));
    }
    return tableNames;
  }

  @Override
  public TableName[] listTableNames(Pattern pattern) throws IOException {
    return listTableNames();
  }

  @Override
  public TableName[] listTableNames(String regex) throws IOException {
    return listTableNames();

  }

  @Override
  public TableName[] listTableNames(Pattern pattern, boolean includeSysTables) throws IOException {
    return listTableNames();
  }

  @Override
  public TableName[] listTableNames(String regex, boolean includeSysTables) throws IOException {
    return listTableNames();
  }

  @Override
  public HTableDescriptor getTableDescriptor(TableName tableName)
      throws TableNotFoundException, IOException {
    LindormTableDescriptor lindormTableDescriptor = connection
        .getWideColumnService(tableName.getNamespaceAsString())
        .describeTable(tableName.getQualifierAsString());
    return ElementConvertor
        .toHbaseTableDescriptor(tableName.getNamespaceAsString(), lindormTableDescriptor);

  }

  @Override
  public HTableDescriptor[] listTableDescriptorsByNamespace(String name) throws IOException {
    TableName[] tableNames = listTableNamesByNamespace(name);
    HTableDescriptor[] hTableDescriptors = new HTableDescriptor[tableNames.length];
    for (int i = 0; i < tableNames.length; i++) {
      hTableDescriptors[i] = getTableDescriptor(tableNames[i]);
    }
    return hTableDescriptors;
  }

  @Override
  public HTableDescriptor[] getTableDescriptorsByTableName(List<TableName> tableNames)
      throws IOException {
    HTableDescriptor[] tableDescriptors = new HTableDescriptor[tableNames.size()];
    for (int i = 0; i < tableNames.size(); i++) {
      tableDescriptors[i] = getTableDescriptor(tableNames.get(i));
    }
    return tableDescriptors;
  }

  public List<HTableDescriptor> listTableDescriptors(List<TableName> tableNames) throws IOException {
    List<HTableDescriptor> tableDescriptors = new ArrayList<>();
    for (TableName tableName : tableNames) {
      tableDescriptors.add(getTableDescriptor(tableName));
    }
    return tableDescriptors;
  }

  @Override
  public HTableDescriptor[] getTableDescriptors(List<String> names) throws IOException {
    List<TableName> tableNames = new ArrayList<>();
    for (String name : names) {
      tableNames.add(TableName.valueOf(name));
    }
    return getTableDescriptorsByTableName(tableNames);
  }

  @Override
  public TableName[] listTableNamesByNamespace(String name) throws IOException {
    List<String> tables = connection.getWideColumnService(name).listTables();
    TableName[] tableNames = new TableName[tables.size()];
    for (int i = 0; i < tables.size(); i++) {
      tableNames[i] = TableName.valueOf(name, tables.get(i));
    }
    return tableNames;
  }

  @Override
  public void createTable(HTableDescriptor desc) throws IOException {
    createTable(desc, null);
  }

  @Override
  public void createTable(HTableDescriptor desc, byte[] startKey, byte[] endKey, int numRegions)
      throws IOException {
    if(numRegions < 3) {
      throw new IllegalArgumentException("Must create at least three regions");
    } else if(Bytes.compareTo(startKey, endKey) >= 0) {
      throw new IllegalArgumentException("Start key must be smaller than end key");
    }
    if (numRegions == 3) {
      createTable(desc, new byte[][]{startKey, endKey});
      return;
    }
    byte [][] splitKeys = Bytes.split(startKey, endKey, numRegions - 3);
    if(splitKeys == null || splitKeys.length != numRegions - 1) {
      throw new IllegalArgumentException("Unable to split key range into enough regions");
    }
    createTable(desc, splitKeys);
  }

  @Override
  public void createTable(HTableDescriptor hTableDescriptor, byte[][] bytes) throws IOException {
    LindormTableDescriptor lindormTableDescriptor = ElementConvertor.toLindormTableDescripter(hTableDescriptor, false);
    TableName tableName = hTableDescriptor.getTableName();
    connection.getWideColumnService(tableName.getNamespaceAsString()).createTable(lindormTableDescriptor, bytes);
  }

  @Override
  public void deleteTable(TableName tableName) throws IOException {
    connection.getWideColumnService(tableName.getNamespaceAsString())
        .deleteTable(tableName.getQualifierAsString());
  }

  public void deleteTable(String tableName) throws IOException {
    this.deleteTable(TableName.valueOf(tableName));
  }

  public void deleteTable(byte[] tableName) throws IOException {
    this.deleteTable(TableName.valueOf(tableName));
  }

  @Override
  public void truncateTable(TableName tableName, boolean preserveSplits) throws IOException {
    connection.getWideColumnService(tableName.getNamespaceAsString())
        .truncateTable(tableName.getQualifierAsString());
  }

  @Override
  public void enableTable(TableName tableName) throws IOException {
    connection.getWideColumnService(tableName.getNamespaceAsString())
        .onlineTable(tableName.getQualifierAsString());
  }

  public void enableTable(byte[] tableName) throws IOException {
    this.enableTable(TableName.valueOf(tableName));
  }

  public void enableTable(String tableName) throws IOException {
    this.enableTable(TableName.valueOf(tableName));
  }

  @Override
  public void disableTable(TableName tableName) throws IOException {
    connection.getWideColumnService(tableName.getNamespaceAsString())
        .offlineTable(tableName.getQualifierAsString());
  }

  public void disableTable(byte[] tableName) throws IOException {
    this.disableTable(TableName.valueOf(tableName));
  }

  public void disableTable(String tableName) throws IOException {
    this.disableTable(TableName.valueOf(tableName));
  }

  @Override
  public boolean isTableEnabled(TableName tableName) throws IOException {
    return connection.getWideColumnService(tableName.getNamespaceAsString())
        .isTableOnline(tableName.getQualifierAsString());
  }

  public boolean isTableEnabled(byte[] tableName) throws IOException {
    return this.isTableEnabled(TableName.valueOf(tableName));
  }

  public boolean isTableEnabled(String tableName) throws IOException {
    return this.isTableEnabled(TableName.valueOf(tableName));
  }

  @Override
  public boolean isTableDisabled(TableName tableName) throws IOException {
    return connection.getWideColumnService(tableName.getNamespaceAsString())
        .isTableOffline(tableName.getQualifierAsString());
  }

  public boolean isTableDisabled(byte[] tableName) throws IOException {
    return this.isTableDisabled(TableName.valueOf(tableName));
  }

  public boolean isTableDisabled(String tableName) throws IOException {
    return this.isTableDisabled(TableName.valueOf(tableName));
  }

  @Override
  public boolean isTableAvailable(TableName tableName) throws IOException {
    return isTableEnabled(tableName);
  }

  public boolean isTableAvailable(byte[] tableName) throws IOException {
    return this.isTableAvailable(TableName.valueOf(tableName));
  }

  public boolean isTableAvailable(String tableName) throws IOException {
    return this.isTableAvailable(TableName.valueOf(tableName));
  }

  @Override
  public boolean isTableAvailable(TableName tableName, byte[][] splitKeys) throws IOException {
    throw new UnsupportedOperationException("isTableAvailable with split keys is not supported");
  }

  @Override
  public void addColumn(TableName tableName, HColumnDescriptor columnFamily)
      throws IOException {
    connection.getWideColumnService(tableName.getNamespaceAsString())
        .addFamily(tableName.getQualifierAsString(),
            ElementConvertor.toLindormcolumnFamilyDescriptor(columnFamily, false));
  }

  public void addColumn(byte[] tableName, HColumnDescriptor column) throws IOException {
    this.addColumn(TableName.valueOf(tableName), column);
  }

  public void addColumn(String tableName, HColumnDescriptor column) throws IOException {
    this.addColumn(TableName.valueOf(tableName), column);
  }

  @Override
  public void deleteColumn(TableName tableName, byte[] columnFamily) throws IOException {
    connection.getWideColumnService(tableName.getNamespaceAsString())
        .deleteFamily(tableName.getQualifierAsString(), Bytes.toString(columnFamily));
  }

  public void deleteColumn(byte[] tableName, String columnName) throws IOException {
    this.deleteColumn(TableName.valueOf(tableName), Bytes.toBytes(columnName));
  }

  public void deleteColumn(String tableName, String columnName) throws IOException {
    this.deleteColumn(TableName.valueOf(tableName), Bytes.toBytes(columnName));
  }

  @Override
  public void modifyColumn(TableName tableName, HColumnDescriptor columnFamily)
      throws IOException {
    List<LindormFamilyAttributes> lindormFamilyAttributes = ElementConvertor
        .toLindormFamilyAttributes(columnFamily, false);
    connection.getWideColumnService(tableName.getNamespaceAsString())
        .modifyFamilyAttributes(tableName.getQualifierAsString(), lindormFamilyAttributes);
  }

  public void modifyColumn(String tableName, HColumnDescriptor descriptor) throws IOException {
    this.modifyColumn(TableName.valueOf(tableName), descriptor);
  }

  public void modifyColumn(byte[] tableName, HColumnDescriptor descriptor) throws IOException {
    this.modifyColumn(TableName.valueOf(tableName), descriptor);
  }

  @Override
  public void modifyTable(TableName tableName, HTableDescriptor td) throws IOException {
    modifyTable(td);
  }

  public void modifyTable(byte[] tableName, HTableDescriptor htd) throws IOException {
    this.modifyTable(TableName.valueOf(tableName), htd);
  }

  public void modifyTable(String tableName, HTableDescriptor htd) throws IOException {
    this.modifyTable(TableName.valueOf(tableName), htd);
  }


  public void modifyTable(HTableDescriptor td) throws IOException {
    LindormTableDescriptor lindormTableDescriptor = ElementConvertor
        .toLindormTableDescripter(td, false);
    connection.getWideColumnService(td.getTableName().getNamespaceAsString())
        .modifyTableAttributes(lindormTableDescriptor);
  }

  @Override
  public void modifyNamespace(NamespaceDescriptor descriptor) throws IOException {
    throw new UnsupportedOperationException("modifyNamespace is not supported");
  }

  @Override
  public void deleteNamespace(String name) throws IOException {
    adminService.removeNamespace(name);
  }

  @Override
  public NamespaceDescriptor getNamespaceDescriptor(String name)
      throws NamespaceNotFoundException, IOException {
    if (adminService.listNamespaces().contains(name)) {
      return NamespaceDescriptor.create(name).build();
    } else {
      throw new NamespaceNotFoundException(name + " is not found");
    }
  }

  @Override
  public NamespaceDescriptor[] listNamespaceDescriptors() throws IOException {
    List<String> namespaces = adminService.listNamespaces();
    NamespaceDescriptor[] namespaceDescriptors = new NamespaceDescriptor[namespaces.size()];
    for (int i = 0; i < namespaces.size(); i++) {
      namespaceDescriptors[i] = NamespaceDescriptor.create(namespaces.get(i)).build();
    }
    return namespaceDescriptors;
  }

  @Override
  public void createNamespace(NamespaceDescriptor descriptor) throws IOException {
    adminService.createNamespace(descriptor.getName());

  }

  @Override
  public void flush(TableName tableName) {
    for (String idc : systemService.getLConnection().getAllIDC()) {
      try {
        systemService.flush(ElementConvertor.toLindormTableFullName(tableName), idc);
      } catch (LindormException e) {
        throw new IllegalArgumentException(e);
      }
    }

  }

  public void flush(String tableNameOrRegionName) throws IOException, InterruptedException {
    this.flush(Bytes.toBytes(tableNameOrRegionName));
  }

  public void flush(byte[] tableNameOrRegionName) throws IOException, InterruptedException {
    try {
      this.flushRegion(tableNameOrRegionName);
    } catch (IllegalArgumentException var3) {
      this.flush(TableName.valueOf(tableNameOrRegionName));
    }

  }

  @Override
  public void flushRegion(byte[] regionName) {
    String idcName = systemService.getLConnection().getAllIDC().get(0);
    try {
      systemService.flush(Bytes.toString(regionName), idcName);
    } catch (LindormException e) {
      throw new IllegalArgumentException(e);
    }

  }

  @Override
  public void compact(TableName tableName) {
    for (String idc : systemService.getLConnection().getAllIDC()) {
      try {
        systemService.compact(ElementConvertor.toLindormTableFullName(tableName), idc);
      } catch (LindormException e) {
        throw new IllegalArgumentException(e);
      }
    }

  }

  public void compact(String tableNameOrRegionName) throws IOException {
    this.compact(Bytes.toBytes(tableNameOrRegionName));
  }

  @Override
  public void compactRegion(byte[] regionName) {
    String idcName = systemService.getLConnection().getAllIDC().get(0);
    try {
      systemService.compact(Bytes.toString(regionName), idcName);
    } catch (LindormException e) {
      throw new IllegalArgumentException(e);
    }

  }

  @Override
  public void compact(TableName tableName, byte[] columnFamily) {
    compact(tableName);

  }

  public void compact(byte[] tableNameOrRegionName) throws IOException {
    try {
      this.compactRegion(tableNameOrRegionName);
    } catch (IllegalArgumentException var3) {
      this.compact(TableName.valueOf(tableNameOrRegionName));
    }

  }

  public void compact(String tableOrRegionName, String columnFamily) throws IOException {
    this.compact(Bytes.toBytes(tableOrRegionName), Bytes.toBytes(columnFamily));
  }

  public void compact(byte[] tableNameOrRegionName, byte[] columnFamily) throws IOException {
    try {
      this.compactRegion(tableNameOrRegionName);
    } catch (IllegalArgumentException var4) {
      this.compact(TableName.valueOf(tableNameOrRegionName));
    }

  }

  @Override
  public void compactRegion(byte[] regionName, byte[] columnFamily) {
    compactRegion(regionName);

  }

  @Override
  public void majorCompact(TableName tableName) {
    for (String idc : systemService.getLConnection().getAllIDC()) {
      try {
        systemService.majorCompact(ElementConvertor.toLindormTableFullName(tableName), idc);
      } catch (LindormException e) {
        throw new IllegalArgumentException(e);
      }
    }

  }

  @Override
  public void majorCompactRegion(byte[] regionName) {
    String idcName = systemService.getLConnection().getAllIDC().get(0);
    try {
      systemService.majorCompact(Bytes.toString(regionName), idcName);
    } catch (LindormException e) {
      throw new IllegalArgumentException(e);
    }

  }


  public void majorCompact(String tableNameOrRegionName, String columnFamily) throws IOException {
    this.majorCompact(Bytes.toBytes(tableNameOrRegionName), Bytes.toBytes(columnFamily));
  }

  public void majorCompact(byte[] tableNameOrRegionName, byte[] columnFamily) throws IOException {
    try {
      this.majorCompactRegion(tableNameOrRegionName, columnFamily);
    } catch (IllegalArgumentException var4) {
      this.majorCompact(TableName.valueOf(tableNameOrRegionName), columnFamily);
    }

  }



  @Override
  public void majorCompact(TableName tableName, byte[] columnFamily) {
    majorCompact(tableName);

  }

  public void majorCompact(String tableNameOrRegionName) throws IOException {
    this.majorCompact(Bytes.toBytes(tableNameOrRegionName));
  }

  public void majorCompact(byte[] tableNameOrRegionName) throws IOException {
    try {
      this.majorCompactRegion(tableNameOrRegionName);
    } catch (IllegalArgumentException var3) {
      this.majorCompact(TableName.valueOf(tableNameOrRegionName));
    }

  }

  @Override
  public void majorCompactRegion(byte[] regionName, byte[] columnFamily) {
    majorCompactRegion(regionName);

  }

  // only avaliable in HBase-2.2+
  public boolean switchRpcThrottle(boolean enable) throws IOException {
    throw new UnsupportedOperationException("switchRpcThrottle is not supported");
  }

  // only avaliable in HBase-2.2+
  public boolean isRpcThrottleEnabled() throws IOException {
    throw new UnsupportedOperationException(
        "isRpcThrottleEnabled is not supported");
  }

  // only avaliable in HBase-2.2+
  public boolean exceedThrottleQuotaSwitch(boolean enable) throws IOException {
    throw new UnsupportedOperationException(
        "exceedThrottleQuotaSwitch is not supported");
  }

  @Override
  public HTableDescriptor[] disableTables(String regex) throws IOException {
    throw new UnsupportedOperationException("disableTables is not supported");
  }

  @Override
  public HTableDescriptor[] disableTables(Pattern pattern) throws IOException {
    throw new UnsupportedOperationException("disableTables by pattern not supported");
  }

  @Override
  public HTableDescriptor[] enableTables(String regex) throws IOException {
    throw new UnsupportedOperationException("enableTables by pattern not supported");
  }

  @Override
  public HTableDescriptor[] enableTables(Pattern pattern) throws IOException {
    throw new UnsupportedOperationException("enableTables by pattern not supported");
  }

  @Override
  public HTableDescriptor[] deleteTables(String regex) throws IOException {
    throw new UnsupportedOperationException("deleteTables by pattern not supported");
  }

  @Override
  public HTableDescriptor[] deleteTables(Pattern pattern) throws IOException {
    throw new UnsupportedOperationException("deleteTables by pattern not supported");

  }

  @Override
  public void closeRegion(String regionname, String serverName) {
    throw new UnsupportedOperationException("closeRegion not supported");

  }

  @Override
  public void closeRegion(byte[] regionname, String serverName) {
    throw new UnsupportedOperationException("closeRegion not supported");

  }

  @Override
  public boolean closeRegionWithEncodedRegionName(String encodedRegionName, String serverName) {
    throw new UnsupportedOperationException(
        "closeRegionWithEncodedRegionName not supported");
  }

  @Override
  public void closeRegion(ServerName sn, HRegionInfo hri) {
    throw new UnsupportedOperationException("closeRegion not supported");

  }

  @Override
  public List<HRegionInfo> getOnlineRegions(ServerName sn) {
    throw new UnsupportedOperationException("getOnlineRegions not supported");
  }

  public List<HRegionInfo> getRegions(ServerName serverName) {
    throw new UnsupportedOperationException("getRegions not supported");
  }

  // only avaliable in HBase-2.2+
  public Map<ServerName, Boolean> compactionSwitch(boolean switchState,
      List<String> serverNamesList) {
    throw new UnsupportedOperationException("compactionSwitch not supported");
  }

  // only avaliable in HBase-2.2+
  public void move(byte[] encodedRegionName) {
    throw new UnsupportedOperationException("move not supported");
  }

  // only avaliable in HBase-2.2+
  public void move(byte[] encodedRegionName, ServerName destServerName) {
    throw new UnsupportedOperationException("move not supported");
  }

  @Override
  public void assign(byte[] regionName) {
    throw new UnsupportedOperationException("assign not supported");

  }

  @Override
  public void unassign(byte[] regionName, boolean force) {
    throw new UnsupportedOperationException("unassign not supported");

  }

  @Override
  public void offline(byte[] regionName) {
    throw new UnsupportedOperationException("offline not supported");

  }

  @Override
  public boolean isBalancerEnabled() {
    throw new UnsupportedOperationException("isBalancerEnabled not supported");
  }

  public boolean normalize() {
    throw new UnsupportedOperationException("normalize not supported");
  }

  public boolean isNormalizerEnabled() {
    throw new UnsupportedOperationException("isNormalizerEnabled not supported");
  }

  @Override
  public boolean isCatalogJanitorEnabled() {
    throw new UnsupportedOperationException("isCatalogJanitorEnabled not supported");
  }

  public boolean runCleanerChore() {
    throw new UnsupportedOperationException("runCleanerChore not supported");
  }

  public boolean isCleanerChoreEnabled() {
    throw new UnsupportedOperationException("isCleanerChoreEnabled not supported");
  }

  @Override
  public void mergeRegions(byte[] nameOfRegionA, byte[] nameOfRegionB, boolean forcible) {
    throw new UnsupportedOperationException("mergeRegions not supported");

  }

  @Override
  public void split(TableName tableName) {
    throw new UnsupportedOperationException("split not supported");
  }

  @Override
  public void splitRegion(byte[] regionName) {
    throw new UnsupportedOperationException("splitRegion not supported");
  }

  @Override
  public void split(TableName tableName, byte[] splitPoint) {
    throw new UnsupportedOperationException("split not supported");
  }

  @Override
  public void splitRegion(byte[] regionName, byte[] splitPoint) {
    throw new UnsupportedOperationException("splitRegion not supported");
  }

  @Override
  public void shutdown() {
    throw new UnsupportedOperationException("shutdown not supported");
  }

  @Override
  public void stopMaster() {
    throw new UnsupportedOperationException("stopMaster not supported");

  }

  public boolean isMasterInMaintenanceMode() {
    throw new UnsupportedOperationException("isMasterInMaintenanceMode not supported");
  }

  @Override
  public void stopRegionServer(String hostnamePort) {
    throw new UnsupportedOperationException("stopRegionServer not supported");

  }


  @Override
  public List<HRegionInfo> getTableRegions(TableName tableName) {
    throw new UnsupportedOperationException("getTableRegions not supported");
  }

  @Override
  public boolean abortProcedure(long procId, boolean mayInterruptIfRunning) {
    throw new UnsupportedOperationException("abortProcedure not supported");
  }

  @Override
  public Future<Boolean> abortProcedureAsync(long procId, boolean mayInterruptIfRunning) {
    throw new UnsupportedOperationException("abortProcedureAsync not supported");
  }

  @Override
  public void rollWALWriter(ServerName serverName) {
    throw new UnsupportedOperationException("rollWALWriter not supported");

  }

  @Override
  public long getLastMajorCompactionTimestamp(TableName tableName) {
    throw new UnsupportedOperationException(
        "getLastMajorCompactionTimestamp not supported");
  }

  @Override
  public long getLastMajorCompactionTimestampForRegion(byte[] regionName) {
    throw new UnsupportedOperationException(
        "getLastMajorCompactionTimestampForRegion not supported");
  }

  @Override
  public void snapshot(String snapshotName, TableName tableName) {
    throw new UnsupportedOperationException("snapshot not supported");

  }

  @Override
  public void snapshot(byte[] snapshotName, TableName tableName) {
    throw new UnsupportedOperationException("snapshot not supported");

  }

  @Override
  public void restoreSnapshot(byte[] snapshotName) {
    throw new UnsupportedOperationException("restoreSnapshot not supported");

  }

  @Override
  public void restoreSnapshot(String snapshotName) {
    throw new UnsupportedOperationException("restoreSnapshot not supported");

  }

  @Override
  public void restoreSnapshot(byte[] snapshotName, boolean takeFailSafeSnapshot) {
    throw new UnsupportedOperationException("restoreSnapshot not supported");

  }

  @Override
  public void restoreSnapshot(String snapshotName, boolean takeFailSafeSnapshot) {
    throw new UnsupportedOperationException("restoreSnapshot not supported");

  }

  public void restoreSnapshot(String snapshotName, boolean takeFailSafeSnapshot,
      boolean restoreAcl) {
    throw new UnsupportedOperationException("restoreSnapshot not supported");

  }

  @Override
  public void cloneSnapshot(byte[] snapshotName, TableName tableName) {
    throw new UnsupportedOperationException("cloneSnapshot not supported");

  }

  public void cloneSnapshot(String snapshotName, TableName tableName, boolean restoreAcl) {
    throw new UnsupportedOperationException("cloneSnapshot not supported");

  }

  @Override
  public void cloneSnapshot(String snapshotName, TableName tableName) {
    throw new UnsupportedOperationException("cloneSnapshot not supported");

  }

  @Override
  public void execProcedure(String signature, String instance, Map<String, String> props) {
    throw new UnsupportedOperationException("execProcedure not supported");

  }

  @Override
  public boolean isProcedureFinished(String signature, String instance, Map<String, String> props) {
    throw new UnsupportedOperationException("isProcedureFinished not supported");
  }

  @Override
  public void deleteSnapshot(byte[] snapshotName) {
    throw new UnsupportedOperationException("deleteSnapshot not supported");

  }

  @Override
  public void deleteSnapshot(String snapshotName) {
    throw new UnsupportedOperationException("deleteSnapshot not supported");

  }

  @Override
  public void deleteSnapshots(String regex) {
    throw new UnsupportedOperationException("deleteSnapshots not supported");

  }

  @Override
  public void deleteSnapshots(Pattern pattern) {
    throw new UnsupportedOperationException("deleteSnapshots not supported");

  }

  public void deleteTableSnapshots(String tableNameRegex, String snapshotNameRegex) {
    throw new UnsupportedOperationException("deleteTableSnapshots not supported");

  }

  public void deleteTableSnapshots(Pattern tableNamePattern, Pattern snapshotNamePattern) {
    throw new UnsupportedOperationException("deleteTableSnapshots not supported");

  }

  @Override
  public void setQuota(QuotaSettings quota) {
    throw new UnsupportedOperationException("setQuota not supported");

  }

  @Override
  public QuotaRetriever getQuotaRetriever(QuotaFilter filter) {
    throw new UnsupportedOperationException("getQuotaRetriever not supported");
  }

  @Override
  public CoprocessorRpcChannel coprocessorService() {
    throw new UnsupportedOperationException("coprocessorService not supported");
  }

  @Override
  public CoprocessorRpcChannel coprocessorService(ServerName serverName) {
    throw new UnsupportedOperationException("coprocessorService not supported");
  }

  @Override
  public void updateConfiguration(ServerName server) {
    throw new UnsupportedOperationException("updateConfiguration not supported");

  }

  @Override
  public void updateConfiguration() {
    throw new UnsupportedOperationException("updateConfiguration not supported");

  }

  public List<SecurityCapability> getSecurityCapabilities() {
    throw new UnsupportedOperationException("getSecurityCapabilities not supported");
  }


  public List<ServerName> clearDeadServers(List<ServerName> servers) {
    throw new UnsupportedOperationException("clearDeadServers not supported");
  }

  @Override
  public void move(byte[] bytes, byte[] bytes1) throws IOException {
    throw new UnsupportedOperationException("move not supported");
  }

  @Override
  public void createTableAsync(HTableDescriptor hTableDescriptor, byte[][] bytes)
      throws IOException {
    throw new UnsupportedOperationException("createTableAsync not supported");
  }

  @Override
  public void enableTableAsync(TableName tableName) throws IOException {
    throw new UnsupportedOperationException("enableTableAsync not supported");
  }

  @Override
  public void disableTableAsync(TableName tableName) throws IOException {
    throw new UnsupportedOperationException("disableTableAsync not supported");
  }

  @Override
  public Pair<Integer, Integer> getAlterStatus(TableName tableName) throws IOException {
    // return a fake pair
    return new Pair<Integer, Integer>(0,1);
  }

  @Override
  public Pair<Integer, Integer> getAlterStatus(byte[] bytes) throws IOException {
    // return a fake pair
    return new Pair<Integer, Integer>(0,1);
  }

  @Override
  public void compactRegionServer(ServerName serverName, boolean b)
      throws IOException, InterruptedException {
    throw new UnsupportedOperationException("compactRegionServer not supported");
  }

  @Override
  public boolean setBalancerRunning(boolean b, boolean b1) throws IOException {
    throw  new UnsupportedOperationException("setBalancerRunning not supported");
  }

  @Override
  public boolean balancer() throws IOException {
    throw new UnsupportedOperationException("balancer not supported");
  }

  public boolean balancer(boolean b) throws IOException {
    throw new UnsupportedOperationException("balancer not supported");
  }

  public boolean setNormalizerRunning(boolean b) throws IOException {
    throw new UnsupportedOperationException("setNormalizerRunning not supported");
  }

  @Override
  public boolean enableCatalogJanitor(boolean b) throws IOException {
    throw new UnsupportedOperationException("enableCatalogJanitor not supported");
  }

  @Override
  public int runCatalogScan() throws IOException {
    throw new UnsupportedOperationException("runCatalogScan not supported");
  }

  public boolean setCleanerChoreRunning(boolean b) throws IOException {
    throw new UnsupportedOperationException("setCleanerChoreRunning not supported");
  }

  @Override
  public ClusterStatus getClusterStatus() throws IOException {
    throw new UnsupportedOperationException("getClusterStatus not supported");
  }

  @Override
  public ProcedureInfo[] listProcedures() throws IOException {
    throw new UnsupportedOperationException("listProcedures not supported");
  }

  @Override
  public String[] getMasterCoprocessors() throws IOException {
    throw new UnsupportedOperationException("getMasterCoprocessors not supported");
  }

  @Override
  public AdminProtos.GetRegionInfoResponse.CompactionState getCompactionState(TableName tableName)
      throws IOException {
    throw new UnsupportedOperationException("getCompactionState not supported");
  }

  @Override
  public AdminProtos.GetRegionInfoResponse.CompactionState getCompactionStateForRegion(byte[] bytes)
      throws IOException {
    throw new UnsupportedOperationException("getCompactionStateForRegion not supported");
  }

  @Override
  public void snapshot(String s, TableName tableName, HBaseProtos.SnapshotDescription.Type type)
      throws IOException, SnapshotCreationException, IllegalArgumentException {
    throw new UnsupportedOperationException("snapshot not supported");
  }

  @Override
  public void snapshot(HBaseProtos.SnapshotDescription snapshotDescription)
      throws IOException, SnapshotCreationException, IllegalArgumentException {
    throw new UnsupportedOperationException("snapshot not supported");
  }

  @Override
  public MasterProtos.SnapshotResponse takeSnapshotAsync(
      HBaseProtos.SnapshotDescription snapshotDescription)
      throws IOException, SnapshotCreationException {
    throw new UnsupportedOperationException("takeSnapshotAsync not supported");
  }

  @Override
  public boolean isSnapshotFinished(HBaseProtos.SnapshotDescription snapshotDescription)
      throws IOException, HBaseSnapshotException, UnknownSnapshotException {
    throw new UnsupportedOperationException("isSnapshotFinished not supported");
  }

  @Override
  public byte[] execProcedureWithRet(String s, String s1, Map<String, String> map)
      throws IOException {
    throw new UnsupportedOperationException("execProcedureWithRet not supported");
  }

  @Override
  public List<HBaseProtos.SnapshotDescription> listSnapshots() throws IOException {
    throw new UnsupportedOperationException("listSnapshots not supported");
  }

  @Override
  public List<HBaseProtos.SnapshotDescription> listSnapshots(String s) throws IOException {
    throw new UnsupportedOperationException("listSnapshots not supported");
  }

  @Override
  public List<HBaseProtos.SnapshotDescription> listSnapshots(Pattern pattern) throws IOException {
    throw new UnsupportedOperationException("listSnapshots not supported");
  }

  public List<HBaseProtos.SnapshotDescription> listTableSnapshots(String s, String s1)
      throws IOException {
    throw new UnsupportedOperationException("listTableSnapshots not supported");
  }

  public List<HBaseProtos.SnapshotDescription> listTableSnapshots(Pattern pattern, Pattern pattern1)
      throws IOException {
    throw new UnsupportedOperationException("listTableSnapshots not supported");
  }

  public int getMasterInfoPort() throws IOException {
    throw new UnsupportedOperationException("getMasterInfoPort not supported");
  }

  public List<ServerName> listDeadServers() throws IOException {
    throw new UnsupportedOperationException("listDeadServers not supported");
  }
}
