package com.aliyun.drc.client;

import java.io.IOException;
import java.util.Map;

import com.aliyun.drc.client.enums.DBType;
import com.aliyun.drc.client.impl.Checkpoint;

/**
 * Client APIs for DRC service.
 */
public interface DRCClient {

	
    /**
     * Used by users to initialize the service by gmtModified.
     * 
     * @param groupName is the name of the users' group.
     * @param dbName is the partial database name.
     * @param identification is the user's password.
     * @param startingPoint is either "checkpoint" or "gmt_modified".
     * @param localFilename path to binary log file. 
     *
     * @throws Exception log file related IO errors.
     */
   public void initService(final String groupName,
                            final String dbName,
                            final String identification,
                            final String startingPoint,
                            final String localFilename)
           throws Exception;

    /**
     * Used by users to reconnect according to the log fle.
     * @param groupName groupName is the name of the users' group.
     * @param dbName dbName is the partial database name.
     * @param identification is the user's password.
     * @param localFilename is path to binary log file.
     * 
     * @throws DRCClientException if the log file not exists.
     * @throws Exception if the log file is not readable.
     */
    public void initService(final String groupName,
                            final String dbName,
                            final String identification,
                            final String localFilename)
            throws Exception;
 
    /**
     * Initialize the service by manually setting restarted checkpoint.
     * Notice that the user should also provide the respective meta data version
     * and assigned task name.
     *
     * @param groupName is the name of the users' group.
     * @param dbName is the partial database name.
     * @param identification is the user's password.
     * @param startingPoint is either "checkpoint" or "gmt_modified".
     * @param metaVersion is the metadata version.
     * @param taskname is the task name.
     * @param mysql is the address of the last database instance.
     * @param localFilename  is path to binary log file.
     * 
     * @throws Exception if the log file cannot be created.
     */
    public void initService(final String groupName,
                            final String dbName,
                            final String identification,
                            final String startingPoint,
                            final String metaVersion,
                            final String taskname,
                            final String mysql,
                            final String localFilename)
            throws Exception;

    /**
     * Initialize the service by checkpoint in memory.
     * 
     * @param groupName is the name of the users' group.
     * @param dbName is the partial database name.
     * @param identification is the user's password.
     * @param checkpoint is the starting location in memory.
     * @param localFilename is path to binary log file.
     * @throws Exception if the log file cannot be created.
     */
    public void initService(final String groupName,
    						final String dbName,
    						final String identification,
    						final Checkpoint checkpoint,
    						final String localFilename) throws Exception;
    
    /**
     * Start the service running in a thread, the method does not block. 
     * @return The launched thread.
     * @throws Exception DRCClient internal Exception.
     */
    public Thread startService() throws Exception;

    /**
     * Stop the service thread;
     * @throws Exception DRCClient internal Exception.
     */
    public void stopService() throws Exception;

    /**
     * Closed active connections and set the status as not quited.
     * @throws Exception DRCClient internal Exception.
     */
    public void resetService() throws Exception;
    
    /**
     * Provide the listener for data.
     * @param listener the listener who receives the message.
     */
    public void addListener(Listener listener);

    /**
     * Add drc client required properties.
     * @param name is the name of the property.
     * @param value is the value of the property.
     */
    public void addDRCConfigure(final String name, final String value); 

    /**
     * Get all configures of drc client.
     * @return all configures of drc client.
     */
    public Map<String, String> getDRCConfigures();

    /**
     * Get one configure for drc client.
     * @param name is the name of the configure.
     * @return the value of the configure. 
     */
    public String getDRCConfigure(final String name);

    /**
     * Add user-defined parameter which is sent to the server.
     * @param name is the name of the parameter.
     * @param value is the value of the parameter.
     */
    public void addUserParameter(final String name, final String value);

    /**
     * Get all parameters added by the user.
     * @return all parameters.
     */
    public Map<String, String> getUserParameters();

    /**
     * Get one parameter added by the user.
     * @param name is the name of the parameter.
     * @return the value of the parameter.
     */
    public String getUserParameter(final String name);

    /**
     * Get partial database name.
     * @return database name.
     */
    public String getDbName();
 
    /**
     * Add data filter to provide required tables and columns.
     * @param filter {@link DataFilter DataFilter}
     */
    @Deprecated
    public void addDataFilter(DataFilter filter);
    /**
     * Add data filter to provide required tables and columns.
     * DataFilterV2 is recommended.
     * @param filter {@link DataFilterBase DataFilter}
     */
    public void addDataFilter(DataFilterBase filter);

    /**
     * Set the frequency of heartbeat records received, one heartbeat every seconds.
     * @param everySeconds after how many seconds, has been sent one heartbeat. 
     */
    public void setHeartbeatFrequency(int everySeconds);

    /**
     * Set whether requiring record type begin/commit.
     * @param required ture if required, false otherwise, default is true.
     */
    public void requireTxnMark(boolean required);

    /**
     * Set the time period to notify the listener by the time consumed.
     * @param sec is the time period.
     */
    public void setNotifyRuntimePeriodInSec(long sec);

    /**
     * Set the number of records in one batch by the server. 
     * @param threshold is the number of records.
     */
    public void setNumOfRecordsPerBatch(int threshold);

    /**
     * Get the type of source database, current is "mysql" or "oceanbase"
     * @return the type of the source database
     * @throws DRCClientException DRCClient internal Exception.
     */
    public DBType getDatabaseType() throws DRCClientException ;
    
    /**
     * Set the group name of the client. The default name is the username,
     * but client can set it anyway.
     * @param group group
     */
    public void setGroup(final String group);

    /**
     * Set the subgroup name of the client. The default name is the dbname,
     * but the client can set it anyway.
     * @param subGroup subGroup
     */
    public void setSubGroup(final String subGroup);

    /**
     * set regex (drc.t*x_begin4unit_mark_[0-9]* ,get current unit data)
     * @param mark regex
     */
    public void setDrcMark(String mark);

    /**
     * get current unit data
     */
    public void askSelfUnit();

    /**
     * db.table|db.table
     * @param blackList blackList
     */
    public void setBlackList(String blackList);

    /**
     * suspend drc client
     */
    public void suspend();

    /**
     * resume drc client
     */
    public void resume();

    /**
     * user public ip or private ip by default
     */
    public void usePublicIp();

    /**
     * upper case and lower case
     */
    public void useCaseSensitive();
    
    /**
     * Add where conditions to filter unnecessary records.
     * @param filter {@link DataFilter DataFilter}
     * 
     * @deprecated because now not supported
     */
    @Deprecated
    public void addWhereFilter(DataFilterBase filter);

    /**
     * Filter a record if none of its columns are  really changed.
     * 
     * @deprecated because now not supported
     */
    @Deprecated
    public void useStrictFilter();

    /**
     * Get the task name.
     * @return the task name.
     * 
     * @deprecated because no task name generated in servers
     */
    @Deprecated
    public String getTaskName();

    /**
     * Get the address of the database instance.
     * @return the address of database instance.
     * 
     * @deprecated because each record has its own instance (db address)
     */
    @Deprecated
    public String getInstance();

    /**
     * trim blob or other type data to small size,omit some data
     */
    public void trimLongType();

    public void useCRC32Check();

    public DBType getDBType();
}

